/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-11-03
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "AbstractNameLine.h"
#include "LineLayer.h"
#include "XMLSciFigs.h"

namespace SciFigs {

/*!
  \class AbstractNameLine AstractNameLine.h
  \brief Brief description of class still missing
  Abstract line with points and names
*/

const QString AbstractNameLine::xmlAbstractNameLineTag="NameLine";

AbstractNameLine::AbstractNameLine(const AbstractNameLine& o)
  : AbstractLine(o)
{
  TRACE;
  _textColor=o._textColor;
  _textDx=o._textDx;
  _textDy=o._textDy;
  _textOrientation=o._textOrientation;
}

void AbstractNameLine::operator=(const AbstractLine& o)
{
  const AbstractNameLine * co=dynamic_cast<const AbstractNameLine *>(&o);
  ASSERT(co);
  _textColor=co->_textColor;
  _textDx=co->_textDx;
  _textDy=co->_textDy;
  _textOrientation=co->_textOrientation;
  AbstractLine::operator=(o);
}

QString AbstractNameLine::toString(int index, const CurvePointOptions * options) const
{
  TRACE;
  NamedPoint p=point(index, options);
  p.setName(name(index));
  return p.toString('g', 20);
}

bool AbstractNameLine::fromString(int index, const StringSection& str, const QChar *& ptr, const CurvePointOptions * options)
{
  TRACE;
  bool ok=true;
  StringSection f;
  f=str.nextField(ptr);
  if(f.isValid()) {
    setX(index, f.toDouble(ok));
  } else {
    return false;
  }
  f=str.nextField(ptr);
  if(f.isValid()) {
    setY(index, f.toDouble(ok), options);
  } else {
    return false;
  }
  const QChar * oldPtr=ptr;
  f=str.nextField(ptr);
  if(f.isValid()) {
    double z=f.toDouble(ok);
    if(ok) { // Z is optional
      setZ(index, z, options);
      oldPtr=ptr;
    } else {
      setZ(index, 0.0, options);
      ptr=oldPtr;
      ok=true;
    }
    f=str.nextField(ptr, "\r\n", false);
    if(f.isValid()) {
      setName(index, f.toStringBuffer());
    } else {
      setName(index, QString());
      ptr=oldPtr;
    }
  } else { // Z and Name are optional
    setZ(index, 0.0, options);
    setName(index, QString());
    ptr=oldPtr;
  }
  return ok;
}

void AbstractNameLine::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  XMLSciFigs * scifigsContext=static_cast<LineLayerContext *>(context)->parent();
  if(scifigsContext->makeUp()) {
    writeProperty(s, "color",_textColor.name());
    writeProperty(s, "dx",_textDx);
    writeProperty(s, "dy",_textDy);
    writeProperty(s, "orientation",_textOrientation);
  }
  AbstractLine::xml_writeProperties(s, context);
}

XMLMember AbstractNameLine::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  XMLSciFigs * scifigsContext=static_cast<LineLayerContext *>(context)->parent();
  if(scifigsContext->data()) {
    if(tag=="points") return XMLMember(0);
  }
  if(scifigsContext->makeUp()) {
    if(tag=="dx") return XMLMember(1);
    else if(tag=="dy") return XMLMember(2);
    else if(tag=="orientation") return XMLMember(3);
    else if(tag=="color") return XMLMember(4);
  }
  return AbstractLine::xml_member(tag, attributes, context)+5;
}

bool AbstractNameLine::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  bool ok=true;
  LineLayerContext * lineContext=static_cast<LineLayerContext *>(context);
  switch(memberID) {
  case 0: {
      const QChar * ptr=0;
      clear();
      int i=0;
      while(true) {
        append();
        if(!content.nextLine(ptr)) break;
        if(!fromString(i, content, ptr, lineContext->options())) break;
        i++;
      }
      removeLast();
    }
    return true;
  case 1:
    _textDx=content.toDouble(ok);
    return ok;
  case 2:
    _textDy=content.toDouble(ok);
    return ok;
  case 3:
    _textOrientation=content.toDouble(ok);
    return ok;
  case 4: {
      Color c;
      c.fromString(content.toStringView(), ok);
      guiColor(c, _textColor);
      return ok;
    }
  default:
    return AbstractLine::xml_setProperty(memberID-5, tag, attributes, content, context);
  }
}

} // namespace SciFigs
