/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-04-01
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QtGui>
#if(QT_VERSION > QT_VERSION_CHECK(5, 0, 0))
#include <QtWidgets>
#endif

#include <QGpCoreTools.h>

#include "PropertyValue.h"

namespace QGpGuiTools {

/*!
  \class PropertyValue PropertyValue.h
  \brief A property value managing red and bold indicators for constant and touched states, respectively.

  A class that records a property value. A property can be shared by various objects.
  It means that changing its value may alter various objects. To know if all objects
  have already the same value and if the value displayed in a control is really shared
  by all objects, the label of the control is colored. A red color indicates that all
  objects have currently not the same value. Hence, changing this value will uniformize
  these object reagarding this property. This is the "constant" state.
*/

/*!
  Constructor. \a w or \a label can be null. \a w and \a label can point to the same
  widget (e.g. for a checkbox).
*/
PropertyValue::PropertyValue(int id, QWidget * w, QWidget * label, WidgetType widgetType)
{
  TRACE;
  _id=id;
  _widget=w;
  _label=label;
  _widgetType=widgetType;
  _constant=true;
  _frozen=false;
  _touched=false;
}

/*!
  \fn void PropertyValue::touched()
  The property widget has been modified by user, commit changed to object
*/

/*!
  Set current value to \a val. If the current value is already valid
  and if it differs from \a val, the contant state is changed to false
  and the color of label is set to red.
*/
void PropertyValue::setValue(QVariant val)
{
  TRACE;
  if(_constant && _value.isValid()) {
    if(val!=_value) {
      _constant=false;
      if(_label) {
        QPalette palette;
        palette.setColor(QPalette::WindowText, Qt::red);
        _label->setPalette(palette);
      }
      emit constantChanged(false);
    }
  }
  _value=val;
}

/*!
  \fn void PropertyValue::setCurrentValue(QVariant val)
  For non constant values, change the current value to \a val
*/

/*!
  Set current value as invalid, constant and untouched
*/
void PropertyValue::reset()
{
  if(!_constant && _label) {
    QPalette palette;
    palette.setColor(QPalette::WindowText, Qt::black);
    _label->setPalette(palette);
  }
  _constant=true;
  emit constantChanged(true);
  _value=QVariant();
}

} // namespace QGpGuiTools
