/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  QGpGuiTools is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpGuiTools is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-05-21
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ItemDelegate.h"
#include "DoubleSpinBox.h"

namespace QGpGuiTools {

  /*!
    \class ItemDelegate ItemDelegate.h
    \brief Brief description of class still missing

  */

  /*!
    Description of constructor still missing
  */
  ItemDelegate::ItemDelegate(QObject *parent)
    : QItemDelegate(parent)
  {
    TRACE;
  }

  /*!
    Description of destructor still missing
  */
  ItemDelegate::~ItemDelegate()
  {
    TRACE;
  }

  /*!
    Use this function in createEditor() instead of directly the constructor.
    May help future the implementation of common options.
  */
  QLineEdit * ItemDelegate::createLineEdit(QWidget * parent) const
  {
    return new QLineEdit(parent);
  }

  /*!
    Use this function in createEditor()  instead of directly the constructor.
    May help future the implementation of common options.
  */
  QComboBox * ItemDelegate::createComboBox(QWidget * parent) const
  {
    return new QComboBox(parent);
  }

  /*!
    Use this function in createEditor()  instead of directly the constructor.
    May help future the implementation of common options.
  */
  DoubleSpinBox * ItemDelegate::createDoubleSpinBox(QWidget * parent) const
  {
    return new DoubleSpinBox(parent);
  }

  /*!
    Use this function in createEditor()  instead of directly the constructor.
    May help future the implementation of common options.
  */
  QSpinBox * ItemDelegate::createSpinBox(QWidget * parent) const
  {
    return new QSpinBox(parent);
  }

  void ItemDelegate::setEditorData(QWidget *editor, const QModelIndex &index) const
  {
    TRACE;
    const char * className=editor->metaObject()->className();
    bool accepted=false;
    switch (className[1]) {
    case 'C':
      if(strcmp(className, "QComboBox")==0) {
        accepted=true;
        QComboBox * w=qobject_cast<QComboBox *>(editor);
        w->blockSignals(true);
        QVariant val=index.model()->data(index);
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
        if(val.typeId()==QMetaType::Int) {   // Items can refered by their index
#else
        if(val.type()==QVariant::Int) {   // Items can refered by their index
#endif
          w->setCurrentIndex(val.toInt());
        } else {                          // Or by their plain text
          w->setCurrentText(val.toString());
        }
        w->blockSignals(false);
      }
      break;
    case 'G':
      if(strcmp(className, "QGpGuiTools::DoubleSpinBox")==0) {
        accepted=true;
        DoubleSpinBox * w=qobject_cast<DoubleSpinBox *>(editor);
        w->blockSignals(true);
        w->setValue(index.model()->data(index).toDouble());
        w->blockSignals(false);
      }
      break;
    case 'L':
      if(strcmp(className, "QLineEdit")==0) {
        accepted=true;
        QLineEdit * w=qobject_cast<QLineEdit *>(editor);
        w->blockSignals(true);
        w->setText(index.model()->data(index).toString());
        w->selectAll();
        w->setCursorPosition(0);
        w->blockSignals(false);
      }
      break;
    case 'S':
      if(strcmp(className, "QSpinBox")==0) {
        accepted=true;
        QSpinBox * w=qobject_cast<QSpinBox *>(editor);
        w->blockSignals(true);
        w->setValue(index.model()->data(index).toInt());
        w->blockSignals(false);
      }
      break;
    default:
      break;
    }
    if(!accepted) {
      QItemDelegate::setEditorData(editor, index);
    }
  }

  void ItemDelegate::setModelData(QWidget *editor, QAbstractItemModel *model,
                                  const QModelIndex &index) const
  {
    TRACE;
    const char * className=editor->metaObject()->className();
    bool accepted=false, touched=false;
    StringDiff string;
    switch (className[1]) {
    case 'C':
      if(strcmp(className, "QComboBox")==0) {
        accepted=true;
        QComboBox * w=qobject_cast<QComboBox *>(editor);
        QVariant val=index.model()->data(index);
        if(w->count()>0 &&
           w->itemText(0).isEmpty() &&
           !w->itemIcon(0).isNull() &&
   #if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
           val.typeId()==QMetaType::Int) {
   #else
           val.type()==QVariant::Int) {
   #endif
          if(val.toInt()!=w->currentIndex()) {
            model->setData(index, w->currentIndex());
            touched=true;
          }
        } else {
          if(val.toString()!=w->currentText()) {
            model->setData(index, w->currentText());
            touched=true;
          }
        }
      }
      break;
    case 'G':
      if(strcmp(className, "QGpGuiTools::DoubleSpinBox")==0) {
        accepted=true;
        DoubleSpinBox * w=qobject_cast<DoubleSpinBox *>(editor);
        if(model->data(index).toDouble()!=w->value()) {
          model->setData(index, w->value());
          touched=true;
        }
      }
      break;
    case 'L':
      if(strcmp(className, "QLineEdit")==0) {
        accepted=true;
        QLineEdit * w=qobject_cast<QLineEdit *>(editor);
        if(w->isModified() && model->data(index).toString()!=w->text()) {
          bool ok;
          QString oldStr=model->data(index).toString();
          QString newStr=w->text();
          oldStr.toDouble(&ok);
          if(!ok) {
            newStr.toDouble(&ok);
            if(!ok) { // Both strings are not numbers
              string.setDifferences(oldStr, newStr);
            }
          }
          model->setData(index, newStr);
          touched=true;
        }
      }
      break;
    case 'S':
      if(strcmp(className, "QSpinBox")==0) {
        accepted=true;
        QSpinBox * w=qobject_cast<QSpinBox *>(editor);
        if(model->data(index).toInt()!=w->value()) {
          model->setData(index, w->value());
          touched=true;
        }
      }
      break;
    default:
      break;
    }
    if(!accepted) {
      QVariant val=model->data(index);
      QItemDelegate::setModelData(editor, model, index);
      touched=(val==model->data(index));
    }
    // If several rows are selected, apply the modification to all rows
    if(touched) {
      emit dataChanged(index, string);
    }
  }

  /*!
    When creating widgets, connect to this slot if you want an immediate modification
    of the model data whenever the widget is modified. The default behavior is waiting
    for a validation before commiting (enter or return for QLineEdit).
  */
  void ItemDelegate::commitWidget(QWidget * w)
  {
    TRACE;
    if(!w) {
      w=qobject_cast<QWidget *>(sender());
      ASSERT(w);
    }
    emit commitData(w);
  }

} // namespace QGpGuiTools

