/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-03-30
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ExpandTabTitle.h"

namespace QGpGuiTools {

/*!
  \class ExpandTabTitle ExpandTabTitle.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

const int ExpandTabTitle::_spacing=5;
const int ExpandTabTitle::_expandSize=9; // avoid dot lines
const int ExpandTabTitle::_upDownSize=15;

/*!
  Description of constructor still missing
*/
ExpandTabTitle::ExpandTabTitle(QWidget * parent)
    : QWidget(parent)
{
  TRACE;
  _content=0;
  _expanded=true;
  _docked=true;
  _expandHover=false;
  _moveUpHover=false;
  _moveDownHover=false;
  _dockHover=false;
  _expandPress=false;
  _moveUpPress=false;
  _moveDownPress=false;
  _dockPress=false;
  _resizeStart=-1;
  setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
}

/*!
  Description of destructor still missing
*/
ExpandTabTitle::~ExpandTabTitle()
{
  TRACE;
}

QSize ExpandTabTitle::minimumSizeHint() const
{
  TRACE;
  int h=fontMetrics().lineSpacing ();
  if(h<_upDownSize) h=_upDownSize;
  return QSize(5*_spacing+2*_upDownSize+_expandSize+50,h+_spacing);
}

inline QRect ExpandTabTitle::expandRect() const
{
  TRACE;
  return QRect(_spacing, (height() - _expandSize)/2, _expandSize, _expandSize);
}

inline QRect ExpandTabTitle::moveUpRect() const
{
  TRACE;
  return QRect(width() - 3*_upDownSize - 3*_spacing, (height() - _upDownSize)/2, _upDownSize, _upDownSize);
}

inline QRect ExpandTabTitle::moveDownRect() const
{
  TRACE;
  return QRect(width() - 2*_upDownSize - 2*_spacing, (height() - _upDownSize)/2, _upDownSize, _upDownSize);
}

inline QRect ExpandTabTitle::dockRect() const
{
  TRACE;
  return QRect(width() - _upDownSize - _spacing, (height() - _upDownSize)/2, _upDownSize, _upDownSize);
}

void ExpandTabTitle::paintEvent(QPaintEvent * )
{
  TRACE;
  QPainter p(this);

  QStyleOptionButton option;
  option.state=QStyle::State_Active;
#ifdef Q_OS_DARWIN
  option.state |= QStyle::State_Raised;
#endif
  option.state &= ~QStyle::State_HasFocus;
  option.rect=rect();
  option.palette=palette();
  option.features=QStyleOptionButton::None;
  style()->drawControl(QStyle::CE_PushButton, &option, &p, this);

  QStyleOption buttonOption;
  buttonOption.palette=palette();

  // Expanded/Collapse button
  buttonOption.rect=expandRect();
  buttonOption.state=QStyle::State_Children;
  if(_expandHover) {
    buttonOption.rect.translate(1,1);
    if(!_expanded) buttonOption.state |= QStyle::State_Open;
  } else {
    if(_expanded) buttonOption.state |= QStyle::State_Open;
  }
  style()->drawPrimitive(QStyle::PE_IndicatorBranch, &buttonOption, &p, this);

  // Move up button
  buttonOption.rect=moveUpRect();
  if(_moveUpHover) buttonOption.rect.translate(1,1);
  buttonOption.state=isEnabled() ? QStyle::State_Enabled : QStyle::State_None;
  style()->drawPrimitive(QStyle::PE_IndicatorArrowUp, &buttonOption, &p, this);

  // Move down button
  buttonOption.rect=moveDownRect();
  if(_moveDownHover) buttonOption.rect.translate(1,1);
  style()->drawPrimitive(QStyle::PE_IndicatorArrowDown, &buttonOption, &p, this);

  // Dock/Undock button
  buttonOption.rect=dockRect();
  if(_dockHover) buttonOption.rect.translate(1,1);
  if(_docked)
    style()->drawItemPixmap(&p, buttonOption.rect, Qt::AlignCenter, style()->standardPixmap(QStyle::SP_TitleBarNormalButton) );
  else
    style()->drawItemPixmap(&p, buttonOption.rect, Qt::AlignCenter, style()->standardPixmap(QStyle::SP_TitleBarMaxButton) );

  // Title
  QRect textrect=QRect(2*_spacing + _expandSize, 0, width() - 2*_upDownSize - _expandSize - 5*_spacing, height());
  style()->drawItemText(&p, textrect, Qt::AlignCenter, palette(), isEnabled(), _title);
}

void ExpandTabTitle::setExpanded(bool e)
{
  TRACE;
  if(e) {
    if(_content) _content->show();
  } else {
    if(_content) _content->hide();
  }
  _expanded=e;
  update();
}

void ExpandTabTitle::mousePressEvent(QMouseEvent * e)
{
  TRACE;
  if(expandRect().contains(e->pos()) ) {
    _expandPress=true;
    _expandHover=true;
  } else if(moveUpRect().contains(e->pos()) ) {
    _moveUpPress=true;
    _moveUpHover=true;
  } else if(moveDownRect().contains(e->pos()) ) {
    _moveDownPress=true;
    _moveDownHover=true;
  } else if(dockRect().contains(e->pos()) ) {
    _dockPress=true;
    _dockHover=true;
  } else {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
    _resizeStart=e->globalPosition().y();
#else
    _resizeStart=e->globalPos().y();
#endif
  }
  update();
}

void ExpandTabTitle::mouseMoveEvent(QMouseEvent * e)
{
  TRACE;
  _expandHover=false;
  _moveUpHover=false;
  _moveDownHover=false;
  if(_expandPress && expandRect().contains(e->pos())) {
    _expandHover=true;
  } else if(_moveUpPress && moveUpRect().contains(e->pos())) {
    _moveUpHover=true;
  } else if(_moveDownPress && moveDownRect().contains(e->pos())) {
    _moveDownHover=true;
  } else if(_dockPress && dockRect().contains(e->pos())) {
    _dockHover=true;
  } else if(_resizeStart>=0) {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
    emit resize(e->globalPosition().y()-_resizeStart);
    _resizeStart=e->globalPosition().y();
#else
    emit resize(e->globalPos().y()-_resizeStart);
    _resizeStart=e->globalPos().y();
#endif
  }
  update();
}

void ExpandTabTitle::mouseReleaseEvent(QMouseEvent * )
{
  TRACE;
  if(_expandHover) {
    setExpanded( !_expanded);
    _expandHover=false;
  } else if(_moveUpHover) {
    emit moveUp();
    _moveUpHover=false;
  } else if(_moveDownHover) {
    emit moveDown();
    _moveDownHover=false;
  } else if(_dockHover) {
    emit dock();
    _dockHover=false;
  } else if(_resizeStart>=0) {
    _resizeStart=-1;
  }
  _expandPress=false;
  _moveUpPress=false;
  _moveDownPress=false;
  _dockPress=false;
  update();
}

void ExpandTabTitle::contextMenuEvent(QContextMenuEvent * e)
{
  TRACE;
  QMenu m;
  emit addActions(&m);
  if(!m.actions().isEmpty())
    m.exec(mapToGlobal( e->pos()) );
}

} // namespace QGpGuiTools
