/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-01-10
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ImageLayerProperties.h"
#include "ImageLayer.h"
#include "ImageScalingPoint.h"

namespace QGpGuiMath {

  ImageLayerProperties::ImageLayerProperties(QWidget * parent)
      : PropertyWidget(parent),
        _ui(new Ui::ImageLayerProperties)
  {
    TRACE;
    _ui->setupUi(this);

    _ui->x->setMinimum(-std::numeric_limits<double>::infinity());
    _ui->y->setMinimum(-std::numeric_limits<double>::infinity());
    _ui->xScale->setMinimum(-std::numeric_limits<double>::infinity());
    _ui->yScale->setMinimum(-std::numeric_limits<double>::infinity());
    _ui->x->setMaximum(std::numeric_limits<double>::infinity());
    _ui->y->setMaximum(std::numeric_limits<double>::infinity());
    _ui->xScale->setMaximum(std::numeric_limits<double>::infinity());
    _ui->yScale->setMaximum(std::numeric_limits<double>::infinity());

    addProperty(XOrigin, _ui->x, _ui->xLabel);
    addProperty(YOrigin, _ui->y, _ui->yLabel);
    addProperty(XScale, _ui->xScale, _ui->xScaleLabel);
    addProperty(YScale, _ui->yScale, _ui->yScaleLabel);
    addProperty(ImageFile, _ui->image, _ui->imageLabel);
  }

  ImageLayerProperties::~ImageLayerProperties()
  {
    TRACE;
  }

  void ImageLayerProperties::addLayer(ImageLayer * layer)
  {
    TRACE;
    if(_layerSet.contains(layer)) return;
    _layerSet.insert(layer);
  }

  void ImageLayerProperties::removeLayer(ImageLayer * layer)
  {
    TRACE;
    if(!_layerSet.contains(layer)) return;
    _layerSet.remove(layer);
  }

  /*!
    Return true if the \a fileName can be loaded (exists and user did not cancel).
  */
  bool ImageLayerProperties::loadImage(const QString &fileName)
  {
    TRACE;
    QFileInfo fi(fileName);
    if(fi.exists() && _layerSet.count()==1) {
      QImage im;
      im.load(fileName);
      AxisWindow * w=(*_layerSet.begin())->graph();
      switch(Message::question(MSG_ID, tr("Image layer"),
                               tr("Do you want to adjust image origin and scale automatically?"),
                               Message::yes(), Message::no(), Message::cancel(), true)) {
      case Message::Answer0: {
          switch(w->xAxis()->scaleType()) {
          case Scale::Linear:
          case Scale::AbsoluteTime:
          case Scale::RelativeTime:
            _ui->x->setValue(w->xAxis()->minimum());
            _ui->xScale->setValue((w->xAxis()->maximum()-w->xAxis()->minimum())/im.width());
            break;
          case Scale::Log:
            _ui->x->setValue(w->xAxis()->minimum());
            _ui->xScale->setValue((log10(w->xAxis()->maximum())-log10(w->xAxis()->minimum()))/im.width());
            break;
          case Scale::Inverse:
          case Scale::InverseLog:
            _ui->x->setValue(1.0/w->xAxis()->maximum());
            _ui->xScale->setValue((1.0/w->xAxis()->minimum()-1.0/w->xAxis()->maximum())/im.width());
            break;
          }
          switch(w->yAxis()->scaleType()) {
          case Scale::Linear:
          case Scale::AbsoluteTime:
          case Scale::RelativeTime:
            _ui->y->setValue(w->yAxis()->maximum());
            _ui->yScale->setValue((w->yAxis()->minimum()-w->yAxis()->maximum())/im.height());
            break;
          case Scale::Log:
            _ui->y->setValue(w->yAxis()->maximum());
            _ui->yScale->setValue((log10(w->yAxis()->minimum())-log10(w->yAxis()->maximum()))/im.height());
            break;
          case Scale::Inverse:
          case Scale::InverseLog:
            _ui->y->setValue(1.0/w->yAxis()->minimum());
            _ui->yScale->setValue((1.0/w->yAxis()->maximum()-1.0/w->yAxis()->minimum())/im.height());
            break;
          }
        }
        return true;
      case Message::Answer1:
        return true;
      case Message::Answer2:
        break;
      }
    }
    return false;
  }

  void ImageLayerProperties::on_image_textEdited(const QString& text)
  {
    TRACE;
    loadImage(text);
  }

  void ImageLayerProperties::on_imageBrowse_clicked()
  {
    TRACE;
    QString fileName=Message::getOpenFileName(tr("Open an existing image"),
                                                   GraphicObject::pixelImageFilter);
    if(!fileName.isEmpty()) {
      if(loadImage(fileName)) {
        _ui->image->setText(fileName);
      }
    }
  }

  void ImageLayerProperties::on_reference_clicked()
  {
    TRACE;
    if(_layerSet.count()!=1) {
      return;
    }
    ImageLayer * layer=*_layerSet.begin();
    ImageScalingPoint * d=new ImageScalingPoint(this);
    d->setLayer(layer);
    if(d->exec()==Dialog::Accepted) {
      layer->setReferences(d->references());
      layer->setScaling();

      // Update axis scales to view the new scaled image
      Rect r=layer->boundingRect();
      AxisWindow * w=layer->graph();
      // AxisWindow::setMapScale does not support zoomed graphs
      while(w->isZoomed()) {
        w->zoomOut();
      }
      w->setMapScale(r.x1(), r.y1(), r.x2(), r.y2());
      emit refreshValues();
      layer->deepUpdate();
    }
    delete d;
  }

} // namespace QGpGuiMath
