/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2013-02-20
**  Copyright: 2013-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "CurvePropertiesWidget.h"
#include "CurveBrowser.h"
#include "CurveBrowserDelegate.h"
#include "CurveBrowserItem.h"

namespace QGpGuiMath {

  /*!
    \class CurvePropertiesWidget CurvePropertiesWidget.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  CurvePropertiesWidget::CurvePropertiesWidget(QWidget * parent)
    : QWidget(parent)
  {
    TRACE;
    setupUi(this);

    curveTable->setItemDelegate(new CurveBrowserDelegate(this));
    connect(curveTable, SIGNAL(allDataChanged()), this, SLOT(updateCurves()));
    curveTable->resizeColumnsToContents();

    _proxy=nullptr;
    _line=nullptr;
    _currentLayer=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  CurvePropertiesWidget::~CurvePropertiesWidget()
  {
    TRACE;
    Settings::setColumnWidth(curveTable, _proxy->tag());
    delete _proxy;
  }

  void CurvePropertiesWidget::setProxy(CurveProxy * proxy)
  {
    _proxy=proxy;
    CurveBrowserItem * model=new CurveBrowserItem(_proxy, this);
    curveTable->setModel(model);
    curveName->setText(proxy->name());
    Settings::columnWidth(curveTable, _proxy->tag());
    connect(model, SIGNAL(dataChanged(const QModelIndex &, const QModelIndex &)), this, SIGNAL(curveModified()));
  }

  void CurvePropertiesWidget::setEditable(bool e)
  {
    curveName->setEnabled(e);
    CurveBrowserItem * item=qobject_cast<CurveBrowserItem *>(curveTable->model());
    item->setEditable(e);
  }

  void CurvePropertiesWidget::setLine(AbstractLine * line)
  {
    _line=line;
    visibleBut->blockSignals(true);
    visibleBut->setChecked(_line->isVisible());
    visibleBut->blockSignals(false);
  }

  void CurvePropertiesWidget::setCurveVisible(bool v)
  {
    visibleBut->setChecked(v);
  }

  void CurvePropertiesWidget::setCurveName(const QString& name)
  {
    _proxy->setName(name);
    if(curveName->text()!=name) {
      curveName->setText(name);
    }
  }

  void CurvePropertiesWidget::updateCurves()
  {
    TRACE;
    _currentLayer->deepUpdate();
    emit curveModified();
  }

  void CurvePropertiesWidget::beginCurveChange()
  {
    TRACE;
    CurveBrowserItem * model=static_cast<CurveBrowserItem *>(curveTable->model());
    model->beginCurveChange();
  }

  void CurvePropertiesWidget::endCurveChange()
  {
    TRACE;
    CurveBrowserItem * model=static_cast<CurveBrowserItem *>(curveTable->model());
    model->endCurveChange();
    _currentLayer->deepUpdate();
    emit curveModified();
  }

  void CurvePropertiesWidget::resample(int n, double minimum, double maximum, SamplingOptions options, bool distance)
  {
    TRACE;
    if(minimum>maximum) {
      qSwap(minimum, maximum);
    }
    QString sMode, unit;
    if(options & LogScale) {
      sMode=tr("log(%1)").arg(_proxy->xName());
      unit=_proxy->xUnit();
    } else if(options & InverseScale) {
      sMode=_proxy->xInverseName();
      unit=_proxy->xInverseUnit();
    } else {
      sMode=_proxy->xName();
      unit=_proxy->xUnit();
    }
    beginCurveChange();
    if(distance) {
      _proxy->resample(n, minimum, maximum,
                       0.5*(_proxy->minimumX()+_proxy->maximumX()),
                       0.5*(_proxy->minimumY(_currentLayer->pointOptions())+
                            _proxy->maximumY(_currentLayer->pointOptions())),
                       options, _currentLayer->pointOptions());
      _proxy->addLog(tr( "Resampled with constant distance along curve[%1] from %2 %3 to %4 %5 with %6 samples\n")
                     .arg(sMode).arg(minimum).arg(unit).arg(maximum).arg(unit).arg(n));
    } else {
      _proxy->resample(n, minimum, maximum, options);
      _proxy->addLog(tr("Resampled on a %1 scale from %2 %3 to %4 %5 with %6 samples\n")
                     .arg(sMode).arg(minimum).arg(unit).arg(maximum).arg(unit).arg(n));
    }
    endCurveChange();
  }

  /*!

  */
  void CurvePropertiesWidget::cut(double minimum, double maximum, SamplingOptions options)
  {
    TRACE;
    if(minimum>maximum) {
      qSwap(minimum, maximum);
    }
    QString unit;
    options|=Interpole;
    if(options & InverseScale) {
      unit=_proxy->xInverseUnit();
    } else {
      unit=_proxy->xUnit();
    }
    beginCurveChange();
    _proxy->cut(minimum, maximum, options);
    _proxy->addLog(tr("Cut from %2 %3 to %4 %5\n")
                   .arg(minimum).arg(unit).arg(maximum).arg(unit) );
    endCurveChange();
  }

  /*!

  */
  void CurvePropertiesWidget::validate(double minimum, double maximum, SamplingOptions options, bool value)
  {
    TRACE;
    if(minimum>maximum) {
      qSwap(minimum, maximum);
    }
    QString unit;
    if(options & InverseScale) {
      unit=_proxy->xInverseUnit();
    } else {
      unit=_proxy->xUnit();
    }
    beginCurveChange();
    _proxy->validate(minimum, maximum, value, options);
    _proxy->addLog(tr("Set valid to %1 from %2 %3 to %4 %5\n")
                   .arg(value).arg(minimum).arg(unit).arg(maximum).arg(unit) );
    endCurveChange();
  }

  /*!

  */
  void CurvePropertiesWidget::smooth(const SamplingParameters& sampling, const SmoothingParameters& param)
  {
    TRACE;
    QString sMode, unit;
    switch(sampling.scaleType()) {
    case SamplingParameters::Linear:
      sMode=_proxy->xName();
      unit=_proxy->xUnit();
      break;
    case SamplingParameters::Log:
     sMode=tr("log(%1)").arg(_proxy->xName());
     unit=_proxy->xUnit();
     break;
    case SamplingParameters::Inverse:
     sMode=_proxy->xInverseName();
     unit=_proxy->xInverseUnit();
     break;
    }
    beginCurveChange();
    _proxy->smooth(sampling, param);
    _proxy->addLog(tr("Smoothed on a %1 scale from %2 %3 to %4 %5\n")
                   .arg(sMode)
                   .arg(sampling.minimum())
                   .arg(unit)
                   .arg(sampling.maximum())
                   .arg(unit));
    endCurveChange();
  }

  void CurvePropertiesWidget::on_curveName_textChanged(QString text)
  {
    TRACE;
    _proxy->setName(text);
    emit nameChanged(text);
  }

  void CurvePropertiesWidget::on_visibleBut_toggled(bool checked)
  {
    TRACE;
    if(_line->isVisible()!=checked) {
      LayerLocker ll(_currentLayer);
      _line->setVisible(checked);
      _currentLayer->deepUpdate();
    }
  }

} // namespace QGpGuiMath
