/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-09-22
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MODALCURVE_H
#define MODALCURVE_H

#include <QGpCoreTools.h>
#include "Mode.h"
#include "FactoryPoint.h"
#include "QGpCoreWaveDLLExport.h"

namespace QGpCoreWave {

class ModalFactory;
class Seismic1DModel;

class QGPCOREWAVE_EXPORT ModalCurve : public Curve<FactoryPoint>, public CurveProperties, public XMLClass
{
  TRANSLATIONS("ModalCurve")
public:
  inline ModalCurve();
  inline ModalCurve(int n);
  inline ModalCurve(const Curve<FactoryPoint>& o);
  inline ModalCurve(const Curve<RealStatisticalPoint>& o);
  ~ModalCurve();

  inline bool operator==(const ModalCurve& o) const;
  void operator=(const Curve<RealStatisticalPoint>& o);

  const QString& xml_tagName() const {return xmlModalCurveTag;}
  static const QString xmlModalCurveTag;

  void clear();

  void addMode(const Mode& m);
  void removeMode(int index);
  void linkX(const ::VectorList<double>& x);
  const ::QList<Mode>& modes() const {return _modes;}
  ::QList<Mode>& modes() {return _modes;}
  bool hasMode(const Mode& m) const;
  int indexOf(int factoryIndex) const;

  void setEnabled(bool e) {_enabled=e;}
  bool isEnabled() const {return _enabled;}

  double misfit(int& nValues, int& nData, const ModalFactory& f, Misfit::Type type, double minMisfit) const;
  double determinantMisfit(const Seismic1DModel * m) const;
protected:
  void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  XMLMember xml_member(XML_MEMBER_ARGS);
  bool xml_setProperty(XML_SETPROPERTY_ARGS);
private:
  ::QList<Mode> _modes;
  bool _enabled;
};

inline ModalCurve::ModalCurve()
  : Curve<FactoryPoint>(),
    CurveProperties(),
    XMLClass()
{
  _enabled=true;
}

inline ModalCurve::ModalCurve(int n)
  : Curve<FactoryPoint>(n),
    CurveProperties(),
    XMLClass()
{
  _enabled=true;
}

inline ModalCurve::ModalCurve(const Curve<FactoryPoint>& o)
  : Curve<FactoryPoint>(o),
    CurveProperties(),
    XMLClass()
{
  _enabled=true;
}

inline ModalCurve::ModalCurve(const Curve<RealStatisticalPoint>& o)
  : Curve<FactoryPoint>(),
    CurveProperties(),
    XMLClass()
{
  *this=o;
}

inline bool ModalCurve::operator==(const ModalCurve& o) const
{
  return CurveProperties::operator==(o) &&
         _modes==o._modes &&
         _enabled==o._enabled &&
         Curve<FactoryPoint>::operator==(o);
}

} // namespace QGpCoreWave

#endif // MODALCURVE_H
