/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-12-26
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Thread.h"
#include "TraceInfo.h"
#include "CoreApplication.h"

namespace QGpCoreTools {

/*!
  \class Thread Thread.h
  \brief Thread with an internal stack to record trace points.
*/

Mutex Thread::_threadsMutex;
QList<Thread *> Thread::_threads;

Thread::Thread(QObject * parent)
   : QThread(parent)
{
  QMutexLocker ml(&_threadsMutex);
  _threads.append(this);
}

Thread::~Thread()
{
  QMutexLocker ml(&_threadsMutex);
  int index=_threads.indexOf(this);
  if(index>=0) {
    _threads.removeAt(index);
    CoreApplication::instance()->threadDeleted(this);
  }
  qDeleteAll(_logVector);
}

QStringList Thread::threadNames()
{
  QMutexLocker ml(&_threadsMutex);
  QStringList list;
  list.append("main");
  for(QList<Thread *>::iterator it=_threads.begin(); it!=_threads.end(); it++ ) {
    list.append((*it)->objectName());
  }
  return list;
}

#ifdef TRACE_ENABLED
QList< QStack<const TraceStamp *> * > Thread::stacks()
{
  QList< QStack<const TraceStamp *> * > list;
  list.append(Trace::mainStack());
  for(QList<Thread *>::iterator it=_threads.begin(); it!=_threads.end(); it++ ) {
    list.append((*it)->stack());
  }
  return list;
}
#endif

QList< QStack<TraceBug *> * > Thread::bugStacks()
{
  QList< QStack<TraceBug *> * > list;
  list.append(TraceBug::mainStack());
  for(QList<Thread *>::iterator it=_threads.begin(); it!=_threads.end(); it++ ) {
    list.append((*it)->bugStack());
  }
  return list;
}

QList< VectorList<TraceLog *> * > Thread::logVectors()
{
  QList< VectorList<TraceLog *> * > list;
  list.append(TraceLog::mainVector());
  for(QList<Thread *>::iterator it=_threads.begin(); it!=_threads.end(); it++ ) {
    list.append((*it)->logVector());
  }
  return list;
}

} // namespace QGpCoreTools
