/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  QGpCoreTools is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreTools is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-09-12
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "EnumAsString.h"
#include "CoreApplicationPrivate.h"

namespace QGpCoreTools {

  /*!
    \class EnumAsString EnumAsString.h
    \brief Convenient class to convert enums to strings

    For ENUM_AS_STRING_INIT_* macros, the values must be sorted alphabetically.
    Copy all values to a file like "enums" and run:
      for k in $(cat enums | sed "s/,/ /g"); do echo $k; done | \
      sort | awk '{printf("%s, ", $0)}END{printf("\nSize %i\n",NR)}'
      rm enums

    In header:
      ENUM_AS_STRING_DECL(LineStyle)
    In implementation:
      ENUM_AS_STRING_BEGIN(Pen, LineStyle)
      ENUM_AS_STRING_DATA_3(Dash2Space2Line, SolidLine, NoLine);
      ENUM_AS_STRING_SYNONYM("NoPen", NoLine);
      ENUM_AS_STRING_DEFAULT_VALUE(SolidLine);
      ENUM_AS_STRING_END
  */

  QList<EnumAsString *> EnumAsString::_allEnums;

  EnumAsString * EnumAsString::allocate(const char * name, const char** keys, const int * values)
  {
    EnumAsString * p=new EnumAsString(name, keys, values);
    _allEnums.append(p);
    return p;
  }

  void EnumAsString::deleteAllEnums()
  {
    qDeleteAll(_allEnums);
  }

  EnumAsString::EnumAsString(const char * name, const char** keys, const int * values)
  {
    _name=name;
    _defaultValue=0;
    _size=0;
    _keys=keys;
    _values=values;
    int maxValue=0;
    for(int i=0; _keys[i]; i++) {
      ASSERT(_values[i]>=0);
      if(_values[i]>maxValue) {
        maxValue=_values[i];
      }
      _size++;
    }
    _size2=2;
    while(_size2<_size) {
      _size2=_size2 << 1;
    }
    _hash=new const char*[maxValue+1];
    for(int i=maxValue; i>=0; i--) {
      _hash[i]=nullptr;
    }
    for(int i=0; i<_size; i++) {
      _hash[_values[i]]=_keys[i];
    }
    /*printf("EnumAsString created for %s: ", name);
    for(int i=0; i<_size && i<3; i++) {
      printf("%s, ",_keys[i]);
    }
    printf("\n");*/
  }

  void EnumAsString::assertValidity()
  {
#ifdef QT_DEBUG
    // Check that keys are sorted
    for(int i=1; i<_size; i++) {
      if(qstricmp(_keys[i-1], _keys[i])>=0) {
        printf("Keys are not sorted at index %i for enum %s: '%s' '%s'\n",
               i, _name, _keys[i-1], _keys[i]);
        exit(2);
      }
    }
    bool ok=true;
    /*printf("Checking search function for enum %s...\n", _name);
    for(int i=0; i<_size; i++) {
      printf("  %i. %s --> %i\n", i, _keys[i], _values[i]);
    }
    for(int i=0; i<_size; i++) {
      printf("  looking for key '%s' at index %i\n", _keys[i], i);
      QString k(_keys[i]);
      ASSERT(value(k, ok)==_values[i] && ok);
    }*/
    if(!_synonyms.isEmpty()) {
      // If synonym are already defined in main list, assert that is has no effect on value
      //printf("Checking search function for synonyms %s...\n", _name);
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
      for(QMap<QLatin1StringView, int>::iterator it=_synonyms.begin(); it!=_synonyms.end(); it++) {
#else
      for(QMap<QLatin1String, int>::iterator it=_synonyms.begin(); it!=_synonyms.end(); it++) {
#endif
        //printf("  looking for key '%s'\n", it.key().data());
        QString k(it.key());
        ASSERT(value(k, ok)==it.value() && ok);
      }
    }
#endif
  }

  void EnumAsString::addSynonym(const char * key, int value)
  {
    //printf("add synonym '%s' --> %i\n", key, value);
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
    _synonyms.insert(QLatin1StringView(key), value);
#else
    _synonyms.insert(QLatin1String(key), value);
#endif
  }

  const char * EnumAsString::key(int value) const
  {
    return _hash[value];
  }

  int EnumAsString::value(const StringView& key, bool& ok) const
  {
    QByteArray k=key.toLatin1();
    int i=_size2-1;
    int step2=_size2 >> 1;
    int c;
    while(step2>0) {
      if(i>=_size) {
        i-=step2;
      } else {
        c=qstricmp(k.data(),_keys[i]);
        if(c<0) {
          i-=step2;
        } else if(c>0) {
          i+=step2;
        } else {
          return _values[i];
        }
      }
      step2=step2 >> 1;
    }
    if(i<_size && qstricmp(k.data(),_keys[i])==0) {
      return _values[i];
    }
    // Not found, check for synonyms
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
    QMap<QLatin1StringView, int>::const_iterator it;
    it=_synonyms.find(QLatin1StringView(k.data()));
#else
    QMap<QLatin1String, int>::const_iterator it;
    it=_synonyms.find(QLatin1String(k.data()));
#endif
    if(it!=_synonyms.end()) {
      return it.value();
    } else {
      ok=false;
      App::log(tr("Bad keyword '%1' for %2\n").arg(key).arg(_name));
      return _defaultValue;
    }
  }

} // namespace QGpCoreTools

