/***************************************************************************
**
**  This file is part of QGpCoreTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-09-02
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Color.h"
#include "Trace.h"
#include "CoreApplication.h"

namespace QGpCoreTools {

  /*!
    \class Color Color.h
    \brief A rgba color

    QColor belongs to QtGui which prevent its usage (e.g. for storage) in Core libraries.
    This color just ensure an equivalent storage.
    QGpGuiTools::GuiGlobal provides the global functions to transform to a QColor
  */

  Color::Color(Qt::GlobalColor c)
  {
    TRACE;
    switch(c) {
    case Qt::white:
    case Qt::color1:
      _rgba=0xFFFFFFFFFFFFFFFF;
      break;
    case Qt::black:
      _rgba=0xFFFF000000000000;
      break;
    case Qt::red:
      _rgba=0xFFFFFFFF00000000;
      break;
    case Qt::darkRed:
      _rgba=0xFFFF800000000000;
      break;
    case Qt::green:
      _rgba=0xFFFF0000FFFF0000;
      break;
    case Qt::darkGreen:
      _rgba=0xFFFF000080000000;
      break;
    case Qt::blue:
      _rgba=0xFFFF00000000FFFF;
      break;
    case Qt::darkBlue:
      _rgba=0xFFFF000000008000;
      break;
    case Qt::cyan:
      _rgba=0xFFFF0000FFFFFFFF;
      break;
    case Qt::darkCyan:
      _rgba=0xFFFF000080008000;
      break;
    case Qt::magenta:
      _rgba=0xFFFFFFFF0000FFFF;
      break;
    case Qt::darkMagenta:
      _rgba=0xFFFF800000008000;
      break;
    case Qt::yellow:
      _rgba=0xFFFFFFFFFFFF0000;
      break;
    case Qt::darkYellow:
      _rgba=0xFFFF800080000000;
      break;
    case Qt::gray:
      _rgba=0xFFFFA000A000A400;
      break;
    case Qt::darkGray:
      _rgba=0xFFFF800080008000;
      break;
    case Qt::lightGray:
      _rgba=0xFFFFC000C000C000;
      break;
    case Qt::transparent:
    case Qt::color0:
      _rgba=0x0000000000000000;
      break;
    }
  }

  /*!
    Return color under format #AAAARRRRGGGGBBBB
  */
  QString Color::toString() const
  {
    TRACE;
    return QString("#%1%2%3%4")
        .arg(alpha64(), 4, 16, QChar('0'))
        .arg(red64(), 4, 16, QChar('0'))
        .arg(green64(), 4, 16, QChar('0'))
        .arg(blue64(), 4, 16, QChar('0'));
  }

  void Color::fromString(const StringView& n, bool& ok)
  {
    TRACE;
    Color c;
    switch(n.size()) {
    case 4:
      c.setRgba64(n.mid(1, 1).toUInt(ok ? &ok : nullptr, 16) << 12,
                  n.mid(2, 1).toUInt(ok ? &ok : nullptr, 16) << 12,
                  n.mid(3, 1).toUInt(ok ? &ok : nullptr, 16) << 12);
      break;
    case 5:
      c.setRgba64(n.mid(2, 1).toUInt(ok ? &ok : nullptr, 16) << 12,
                  n.mid(3, 1).toUInt(ok ? &ok : nullptr, 16) << 12,
                  n.mid(4, 1).toUInt(ok ? &ok : nullptr, 16) << 12,
                  n.mid(1, 1).toUInt(ok ? &ok : nullptr, 16) << 12);
      break;
    case 7:
      c.setRgba64(n.mid(1, 2).toUInt(ok ? &ok : nullptr, 16) << 8,
                  n.mid(3, 2).toUInt(ok ? &ok : nullptr, 16) << 8,
                  n.mid(5, 2).toUInt(ok ? &ok : nullptr, 16) << 8);
      break;
    case 9:
      c.setRgba64(n.mid(3, 2).toUInt(ok ? &ok : nullptr, 16) << 8,
                  n.mid(5, 2).toUInt(ok ? &ok : nullptr, 16) << 8,
                  n.mid(7, 2).toUInt(ok ? &ok : nullptr, 16) << 8,
                  n.mid(1, 2).toUInt(ok ? &ok : nullptr, 16) << 8);
      break;
    case 10: // 3 digits does not support alpha, cannot discriminate with 3 channels with 4 digits
      c.setRgba64(n.mid(1, 3).toUInt(ok ? &ok : nullptr, 16) << 4,
                  n.mid(4, 3).toUInt(ok ? &ok : nullptr, 16) << 4,
                  n.mid(7, 3).toUInt(ok ? &ok : nullptr, 16) << 4);
      break;
    case 13:
      c.setRgba64(n.mid(1, 4).toUInt(ok ? &ok : nullptr, 16),
                  n.mid(5, 4).toUInt(ok ? &ok : nullptr, 16),
                  n.mid(9, 4).toUInt(ok ? &ok : nullptr, 16));
      break;
    case 17:
      c.setRgba64(n.mid(5, 4).toUInt(ok ? &ok : nullptr, 16),
                  n.mid(9, 4).toUInt(ok ? &ok : nullptr, 16),
                  n.mid(13, 4).toUInt(ok ? &ok : nullptr, 16),
                  n.mid(1, 4).toUInt(ok ? &ok : nullptr, 16));
      break;
    default:
      ok=false;
      break;
    }
    *this=c;
  }

  /*!
    Same code as in QColor

    Values are from 0 to 255.
  */
  void Color::setHsva(int h, int s, int v, int a)
  {
    TRACE;
    a=a << 8;
    if(s==0 || h==360) { // achromatic case
      v=v << 8;
      setRgba64(v, v, v, a);
    }
    const double hp=h/60.0;
    const double sp=s/255.0;
    const double vp=v/255.0;
    const int i=hp;
    const double f=hp-i;
    const double p=vp*(1.0-sp);
    double rp, gp, bp;
    if (i & 1) {
      const double q=vp*(1.0-(sp*f));
      switch (i) {
      case 1:
        rp=q;
        gp=vp;
        bp=p;
        break;
      case 3:
        rp=p;
        gp=q;
        bp=vp;
        break;
      case 5:
        rp=vp;
        gp=p;
        bp=q;
        break;
      default:
        rp=0.0;
        gp=0.0;
        bp=0.0;
        break;
      }
    } else {
      const double t=vp*(1.0-(sp*(1.0-f)));
      switch (i) {
      case 0:
        rp=vp;
        gp=t;
        bp=p;
        break;
      case 2:
        rp=p;
        gp=vp;
        bp=t;
        break;
      case 4:
        rp=t;
        gp=p;
        bp=vp;
        break;
      default:
        rp=0.0;
        gp=0.0;
        bp=0.0;
        break;
      }
    }
    setRgba64(qRound(rp*COLOR_COMPONENT_MAX),
              qRound(gp*COLOR_COMPONENT_MAX),
              qRound(bp*COLOR_COMPONENT_MAX), a);
  }

  /*!
    Same code as in QColor. Qcolor stores unsigned short values and
    scale them to return integers from 0 to 255 (saturation and value) and 360 (hue). This function
    has exactly the same rounding errors as QColor output functions.

    Values are from 0 to 255.
  */
  void Color::hsv(int& h, int& s, int& v) const
  {
    TRACE;
    double rp=unit(red64());
    double gp=unit(green64());
    double bp=unit(blue64());
    double cmax, cmin;
    minMax(rp, gp, bp, cmin, cmax);
    const double delta=cmax-cmin;
    if(cmax==0) {
      s=0;
    } else {
      s=qRound((delta/cmax)*COLOR_COMPONENT_MAX) >> 8;
    }
    v=qRound(cmax*COLOR_COMPONENT_MAX) >> 8;
    double hp;
    if(delta==0) {
      h=-1;  // achromatic case, hue is undefined
      s=0;
      return;
    } else if(cmax==rp) {
      hp=(gp-bp)/delta;
    } else if(cmax==gp) {
      hp=(bp-rp)/delta+2.0;
    } else {
      hp=(rp-gp)/delta+4.0;
    }
    if(hp<0.0) {
      hp+=6.0;
    }
    h=qRound(6000.0*hp)/100;
  }

  double Color::gammaExpansion(double csrgb)
  {
    if(csrgb>0.04045) {
      return ::pow((csrgb+0.0555)/1.055, 2.4);
    } else {
      return csrgb/12.92;
    }
  }

  double Color::gammaCompression(double clinear)
  {
    if(clinear>0.0031308) {
      return 1.055*::pow(clinear, 1.0/2.4)-0.055;
    } else {
      return 12.92*clinear;
    }
  }

  double Color::unit(quint16 c)
  {
    return static_cast<double>(c)/static_cast<double>(COLOR_COMPONENT_MAX);
  }

  /*!
    Source: https://en.wikipedia.org/wiki/Grayscale
  */
  void Color::toGray()
  {
    TRACE;
    double rLinear=gammaExpansion(unit(red64()));
    double gLinear=gammaExpansion(unit(green64()));
    double bLinear=gammaExpansion(unit(blue64()));
    double luminanceLinear=0.2126*rLinear+0.7152*gLinear+0.0722*bLinear;
    if(luminanceLinear>1.0) { // Observed values above 1 for Oslo palette
      luminanceLinear=1.0;
    }
    quint16 luminanceSrgb=qRound(gammaCompression(luminanceLinear)*COLOR_COMPONENT_MAX);
    setRgba64(luminanceSrgb, luminanceSrgb, luminanceSrgb, alpha64());
  }

  /*!
    Returns the best font color for this background
  */
  bool Color::isDark() const
  {
    int h, s, v;
    hsv(h, s, v);
    return v<128;
  }

  /*!
    Returns true if it is close to white
  */
  bool Color::isNearlyWhite() const
  {
    return red64()>64000 && green64()>64000 && blue64()>64000;
  }

  quint32 Color::argb32() const
  {
#if Q_BYTE_ORDER == Q_BIG_ENDIAN
    quint32 c=((alpha32() << 0) |
               (red32() << 8) |
               (green32() << 16) |
               (blue32() << 24));
#else // little endian:
    quint32 c=((alpha32() << 24) |
               (red32() << 16) |
               (green32() << 8) |
               (blue32() << 0));
#endif
    return c;
  }

} // namespace QGpCoreTools

