/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-12-18
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef HISTOGRAM2D_H
#define HISTOGRAM2D_H

#include <QGpCoreMath.h>

#include "Scale.h"
#include "IrregularGrid2D.h"
#include "StatisticalPoint.h"
#include "Histogram.h"
#include "SampleClassificationParameters.h"

namespace QGpCoreStat {

  class QGPCORESTAT_EXPORT Histogram2D : public IrregularGrid2D
  {
  public:
    Histogram2D(int nx, int ny);
    Histogram2D(const Histogram2D& o);
    ~Histogram2D();

    class QGPCORESTAT_EXPORT Sample : public Point2D
    {
    public:
      Sample() : Point2D() {_valid=true; _category=0;}
      Sample(double x, double y) : Point2D(x, y) {_valid=true; _category=0;}
      Sample(const Point2D& p) : Point2D(p) {_valid=true; _category=0;}

      void writeData(QTextStream& s) const;
      void writeLine(QTextStream& s) const;
      bool read(const QString& line, int xColumn=0, int yColumn=1, int validColumn=-1);

      bool isValid() const {return _valid;}
      void setValid(bool v) {_valid=v;}

      int category() const {return _category;}
      void setCategory(int c) {_category=c;}

      const QString& line() const {return _line;}
      void setLine(const QString& l) {_line=l;}
    private:
      bool _valid;
      int _category;
      QString _line;
    };

    bool setXSampling(const SamplingParameters& s);
    bool setXSampling(const VectorList<double>& x);
    void setXTitle(const QString& t) {_xTitle=t;}
    void setXTitleInverseScale(const QString& t) {_xTitleInverseScale=t;}
    void setXScaleType(Scale::Type s) {_xScaleType=s;}
    bool setYSampling(const SamplingParameters& s);
    void setYTitle(const QString& t) {_yTitle=t;}
    void setYTitleInverseScale(const QString& t) {_yTitleInverseScale=t;}
    void setYScaleType(Scale::Type s) {_yScaleType=s;}

    void setPeriodic(bool p);
    bool isPeriodic() const {return _periodic;}
    double period() const;

    bool setSamples(const VectorList<Sample>& s);
    void addSample(const Point2D& s);
    int sampleCount() const {return _samples.count();}
    void countSamples() {countSamples(this);}
    VectorList<bool> validSamplesFootprint(int ix) const;
    int validSampleCount(int ix) const;
    const Sample& sample(int index) const {return _samples.at(index);}

    void setValid(const Rect& limits, bool v);
    void setClassification(const Histogram2D * o);

    SamplingOptions xSampling() const {return _xSampling;}
    const QString& xTitle() const {return _xTitle;}
    const QString& xTitleInverseScale() const {return _xTitleInverseScale;}
    Scale::Type xScaleType() const {return _xScaleType;}
    SamplingOptions ySampling() const {return _ySampling;}
    const QString& yTitle() const {return _yTitle;}
    const QString& yTitleInverseScale() const {return _yTitleInverseScale;}
    Scale::Type yScaleType() const {return _yScaleType;}
    SamplingParameters xSamplingParameters() const;
    SamplingParameters ySamplingParameters() const;

    Histogram histogramAt(int index) const;

    RealStatisticalValue mean(int ix) const;
    VectorList<double> sampleValues(int ix) const;
    void removeOutliers(int ix, double factor);

    void meanCurve(CurveProxy * curve) const;
    void medianCurve(CurveProxy * curve) const;
    void modeCurve(CurveProxy * curve) const;
    void validateCurve(Curve<RealStatisticalPoint>& curve, double maxMisfit) const;
    double misfit(double x, const NormalDistribution& gd) const;
    Curve<RealStatisticalPoint> pickAll(double maxMisfit) const;
    Curve<RealStatisticalPoint> pickAll(int maxCount, double exploreRange, double maxError) const;

    void classifySamples(const SampleClassificationParameters& param,
                         CurveProxy * curve1=nullptr, CurveProxy * curve2=nullptr);

    class Invalidate
    {
    public:
      static inline bool accept(Sample& s, const SampleClassificationParameters&) {
        return s.isValid();
      }
      static inline void run(Sample& s, const SampleClassificationParameters&) {
        s.setValid(false);
      }
    };

    class Validate
    {
    public:
      static inline bool accept(Sample& s, const SampleClassificationParameters&) {
        return !s.isValid();
      }
      static inline void run(Sample& s, const SampleClassificationParameters&) {
        s.setValid(true);
      }
    };

    class Categorize
    {
    public:
      static inline bool accept(Sample& s, const SampleClassificationParameters&) {
        return s.isValid();
      }
      static inline void run(Sample& s, const SampleClassificationParameters& param) {
        s.setCategory(param.category());
      }
    };

    class Reset
    {
    public:
      static inline bool accept(Sample&, const SampleClassificationParameters&) {
        return true;
      }
      static inline void run(Sample& s, const SampleClassificationParameters&) {
        s.setCategory(0);
        s.setValid(true);
      }
    };

    template<class Action> void all(const SampleClassificationParameters& param);
    template<class Action> void threshold(const SampleClassificationParameters& param);
    template<class Action> void curveOutsideAbsoluteRange(const CurveProxy * curve,
                                                          const SampleClassificationParameters& param);
    template<class Action> void curveOutsideRelativeRange(const CurveProxy * curve,
                                                          const SampleClassificationParameters& param);
    template<class Action> void curveOutsideSigmaRange(const CurveProxy * curve,
                                                       const SampleClassificationParameters& param);
    template<class Action> void curveInsideAbsoluteRange(const CurveProxy * curve,
                                                         const SampleClassificationParameters& param);
    template<class Action> void curveInsideRelativeRange(const CurveProxy * curve,
                                                         const SampleClassificationParameters& param);
    template<class Action> void curveInsideSigmaRange(const CurveProxy * curve,
                                                      const SampleClassificationParameters& param);
    template<class Action> void aboveCurve(const CurveProxy * curve,
                                           const SampleClassificationParameters& param);
    template<class Action> void belowCurve(const CurveProxy * curve,
                                           const SampleClassificationParameters& param);
    template<class Action> void outsideCurves(const CurveProxy * curve1, const CurveProxy * curve2,
                                              const SampleClassificationParameters& param);
    template<class Action> void insideCurves(const CurveProxy * curve1, const CurveProxy * curve2,
                                             const SampleClassificationParameters& param);
    void filterMisfit(const CurveProxy * curve, double maxMisfit);

    bool save(const QString &fileName, const QString& headerLine=QString(), bool maxFormat=false) const;
    bool load(const QString &fileName);

    void write(QTextStream& s) const;
    void read(QTextStream& s);
  private:
    void setStatistics(CurveProxy * curve, int index,
                       double normFactor, const RealStatisticalValue& val) const;
    double misfit(double x, IrregularGrid2D& grid, const NormalDistribution& gd) const;
    double misfit(int ix, IrregularGrid2D& grid, const NormalDistribution& gd) const;
    void misfit(double x, IrregularGrid2D& grid, const NormalDistribution& gd, double maxMisfit);
    void countSamples(IrregularGrid2D * grid) const;
    void admissibleAbsoluteRange(double x, const CurveProxy * curve, double arg, double& ymin, double& ymax);
    void admissibleRelativeRange(double x, const CurveProxy * curve, double arg, double& ymin, double& ymax);
    void admissibleSigmaRange(double x, const CurveProxy * curve, double arg, double& ymin, double& ymax);
    inline bool isOutside(double val, double min, double max,
                          const double& period, const double& halfPeriod) const;
    inline bool isInside(double val, double min, double max,
                         const double& period, const double& halfPeriod) const;

    SamplingOptions _xSampling;
    QString _xTitle, _xTitleInverseScale;
    Scale::Type _xScaleType;
    SamplingOptions _ySampling;
    QString _yTitle, _yTitleInverseScale;
    Scale::Type _yScaleType;
    bool _periodic;
    VectorList<Sample> _samples;
  };

  inline bool Histogram2D::isOutside(double val, double min, double max,
                                     const double& period, const double& halfPeriod) const
  {
    if(_periodic) {
      PeriodicStatistics::shiftValue(val, 0.5*(min+max), period, halfPeriod);
    }
    return val<min || val>max;
  }

  inline bool Histogram2D::isInside(double val, double min, double max,
                                    const double& period, const double& halfPeriod) const
  {
    if(_periodic) {
      PeriodicStatistics::shiftValue(val, 0.5*(min+max), period, halfPeriod);
    }
    return val>min && val<max;
  }

  template<class Action>
  void Histogram2D::all(const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::threshold(const SampleClassificationParameters& param)
  {
    TRACE;
    int ix, iy;
    int stat=0;
    double t=param.threshold();
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        ix=indexOfX(s.x());
        iy=indexOfY(s.y());
        if(ix>-1 && ix<nx() && iy>-1 && iy<ny()) {
          if(value(ix, iy)<=t && param.matchCategory(s.category())) {
            Action::run(s, param);
            stat++;
          }
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::curveOutsideAbsoluteRange(const CurveProxy * curve,
                                              const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double ymin, ymax;
    double dy=param.absoluteRange();
    double period=Histogram2D::period(); // Not used for non-periodic
    double halfPeriod=0.5*period;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve->isInsideRange(s.x());
          admissibleAbsoluteRange(s.x(), curve, dy, ymin, ymax);
        }
        if((!xInside || isOutside(s.y(), ymin, ymax, period, halfPeriod)) &&
           param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::curveOutsideRelativeRange(const CurveProxy * curve,
                                              const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double ymin, ymax;
    double factor=param.relativeRange();
    double period=Histogram2D::period(); // Not used for non-periodic
    double halfPeriod=0.5*period;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve->isInsideRange(s.x());
          admissibleRelativeRange(s.x(), curve, factor, ymin, ymax);
        }
        if((!xInside || isOutside(s.y(), ymin, ymax, period, halfPeriod)) &&
           param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::curveOutsideSigmaRange(const CurveProxy * curve,
                                           const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double ymin, ymax;
    double factor=param.stddevRange();
    double period=Histogram2D::period(); // Not used for non-periodic
    double halfPeriod=0.5*period;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve->isInsideRange(s.x());
          admissibleSigmaRange(s.x(), curve, factor, ymin, ymax);
        }
        if((!xInside || isOutside(s.y(), ymin, ymax, period, halfPeriod)) &&
           param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::curveInsideAbsoluteRange(const CurveProxy * curve,
                                             const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double ymin, ymax;
    double dy=param.absoluteRange();
    double period=Histogram2D::period(); // Not used for non-periodic
    double halfPeriod=0.5*period;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve->isInsideRange(s.x());
          admissibleAbsoluteRange(s.x(), curve, dy, ymin, ymax);
        }
        if(xInside && isInside(s.y(), ymin, ymax, period, halfPeriod) &&
           param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::curveInsideRelativeRange(const CurveProxy * curve,
                                             const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double ymin, ymax;
    double factor=param.relativeRange();
    double period=Histogram2D::period(); // Not used for non-periodic
    double halfPeriod=0.5*period;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve->isInsideRange(s.x());
          admissibleRelativeRange(s.x(), curve, factor, ymin, ymax);
        }
        if(xInside && isInside(s.y(), ymin, ymax, period, halfPeriod) &&
           param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::curveInsideSigmaRange(const CurveProxy * curve,
                                          const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double ymin, ymax;
    double factor=param.stddevRange();
    double period=Histogram2D::period(); // Not used for non-periodic
    double halfPeriod=0.5*period;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve->isInsideRange(s.x());
          admissibleSigmaRange(s.x(), curve, factor, ymin, ymax);
        }
        if(xInside && isInside(s.y(), ymin, ymax, period, halfPeriod) &&
           param.matchCategory(s.category())) {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::aboveCurve(const CurveProxy * curve,
                               const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double mean, stddev;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    switch(_yScaleType) {
    case Scale::Inverse:
    case Scale::InverseLog:
      for(int i=_samples.count()-1; i>=0; i--) {
        Sample& s=_samples[i];
        if(Action::accept(s, param)) {
          if(lastX!=s.x()) {
            lastX=s.x();
            xInside=curve->isInsideRange(s.x());
            curve->interpole(s.x(), mean, stddev, nullptr, _xSampling, _ySampling);
          }
          if(xInside && s.y()<mean && param.matchCategory(s.category())) {
            Action::run(s, param);
            stat++;
          }
        }
      }
      break;
    case Scale::Linear:
    case Scale::Log:
    case Scale::AbsoluteTime:
    case Scale::RelativeTime:
      for(int i=_samples.count()-1; i>=0; i--) {
        Sample& s=_samples[i];
        if(Action::accept(s, param)) {
          if(lastX!=s.x()) {
            lastX=s.x();
            xInside=curve->isInsideRange(s.x());
            curve->interpole(s.x(), mean, stddev, nullptr, _xSampling, _ySampling);
          }
          if(xInside && s.y()>mean && param.matchCategory(s.category())) {
            Action::run(s, param);
            stat++;
          }
        }
      }
      break;
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::belowCurve(const CurveProxy * curve,
                               const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double mean, stddev;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    switch(_yScaleType) {
    case Scale::Inverse:
    case Scale::InverseLog:
      for(int i=_samples.count()-1; i>=0; i--) {
        Sample& s=_samples[i];
        if(Action::accept(s, param)) {
          if(lastX!=s.x()) {
            lastX=s.x();
            xInside=curve->isInsideRange(s.x());
            curve->interpole(s.x(), mean, stddev, nullptr, _xSampling, _ySampling);
          }
          if(xInside && s.y()>mean && param.matchCategory(s.category())) {
            Action::run(s, param);
            stat++;
          }
        }
      }
      break;
    case Scale::Linear:
    case Scale::Log:
    case Scale::AbsoluteTime:
    case Scale::RelativeTime:
      for(int i=_samples.count()-1; i>=0; i--) {
        Sample& s=_samples[i];
        if(Action::accept(s, param)) {
          if(lastX!=s.x()) {
            lastX=s.x();
            xInside=curve->isInsideRange(s.x());
            curve->interpole(s.x(), mean, stddev, nullptr, _xSampling, _ySampling);
          }
          if(xInside && s.y()<mean && param.matchCategory(s.category())) {
            Action::run(s, param);
            stat++;
          }
        }
      }
      break;
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::outsideCurves(const CurveProxy * curve1,
                                  const CurveProxy * curve2,
                                  const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double mean1, stddev1;
    double mean2, stddev2;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve1->isInsideRange(s.x()) && curve2->isInsideRange(s.x());
          curve1->interpole(s.x(), mean1, stddev1, nullptr, _xSampling, _ySampling);
          curve2->interpole(s.x(), mean2, stddev2, nullptr, _xSampling, _ySampling);
        }
        if(xInside) {
          if(mean1<mean2) {
            if((s.y()<mean1 || mean2<s.y()) && param.matchCategory(s.category())) {
              Action::run(s, param);
              stat++;
            }
          } else {
            if((s.y()<mean2 || mean1<s.y()) && param.matchCategory(s.category())) {
              Action::run(s, param);
              stat++;
            }
          }
        } else {
          Action::run(s, param);
          stat++;
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

  template<class Action>
  void Histogram2D::insideCurves(const CurveProxy * curve1,
                                 const CurveProxy * curve2,
                                 const SampleClassificationParameters& param)
  {
    TRACE;
    int stat=0;
    double mean1, stddev1;
    double mean2, stddev2;
    double lastX=std::numeric_limits<double>::quiet_NaN();
    bool xInside=true;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(Action::accept(s, param)) {
        if(lastX!=s.x()) {
          lastX=s.x();
          xInside=curve1->isInsideRange(s.x()) && curve2->isInsideRange(s.x());
          curve1->interpole(s.x(), mean1, stddev1, nullptr, _xSampling, _ySampling);
          curve2->interpole(s.x(), mean2, stddev2, nullptr, _xSampling, _ySampling);
        }
        if(xInside) {
          if(mean1<mean2) {
            if((mean1<s.y() && s.y()<mean2) && param.matchCategory(s.category())) {
              Action::run(s, param);
              stat++;
            }
          } else {
            if((mean2<s.y() && s.y()<mean1) && param.matchCategory(s.category())) {
              Action::run(s, param);
              stat++;
            }
          }
        }
      }
    }
    countSamples();
    App::log(tr("    %1 samples hit\n").arg(stat));
  }

} // namespace QGpCoreStat

#endif // HISTOGRAM2D_H
