/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  QGpCoreStat is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreStat is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-06-12
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef GAUSSIANMIXTUREDISTRIBUTION_H
#define GAUSSIANMIXTUREDISTRIBUTION_H

#include <QGpCoreMath.h>

#include "QGpCoreStatDLLExport.h"

namespace QGpCoreStat {

  class QGPCORESTAT_EXPORT GaussianMixtureDistribution
  {
  public:
    GaussianMixtureDistribution();
    GaussianMixtureDistribution(int modeCount, int dimensionCount);
    GaussianMixtureDistribution(const GaussianMixtureDistribution& o);
    ~GaussianMixtureDistribution();

    void operator=(const GaussianMixtureDistribution& o);

    int modeCount() const {return _modeCount;}
    int dimensionCount() const {return _modes[0].count();}

    void clearVariance();
    inline void setMode(int mode, int dim, double mean, double stddev);
    const MultivariateNormalDistribution& mode(int mode) const {return _modes[mode];}
    MultivariateNormalDistribution& mode(int mode) {return _modes[mode];}
    inline void setFactor(int mode);

    void setWeight(int mode, double w) {_weights[mode]=w;}
    double weight(int mode) const {return _weights[mode];}

    void sortWeight();
    void sortMean();


    void filterWeight(double min);
    void filterDominantModes(double stddevFactor);
    void excludeMerge();
    void filterRange(int dim, double min, double max);
    void bestWeights(int maxCount);
    void unique();

    double value(int mode, const Vector<double>& x) const;
    double value(const Vector<double>& x) const;
    double value1D(int index, double x) const;
    double cumulativeValue1D(int index, double x) const;

    QString toString(bool varianceMatrix);
  private:
    void allocate(int modeCount, int dimensionCount,
                  double *& weights,
                  MultivariateNormalDistribution *& modes);
    void clear();

    class SortIndex
    {
    public:
      SortIndex() {_i=0; _parent=nullptr;}
      SortIndex(GaussianMixtureDistribution * parent, int i) {_i=i; _parent=parent;}

      int _i;
      GaussianMixtureDistribution * _parent;
    };
    VectorList<SortIndex> sortHelper();
    void commitSort(VectorList<SortIndex>& s);
    static bool lessThanWeight(const SortIndex& i1, const SortIndex& i2);
    static bool lessThanMean(const SortIndex& i1, const SortIndex& i2);
    bool beginFilter(double *& newWeights,
                     MultivariateNormalDistribution *& newModes,
                     int newModeCount);
    void endFilter(double * newWeights,
                   MultivariateNormalDistribution * newModes,
                   int newModeCount);

    int _modeCount;
    MultivariateNormalDistribution * _modes;
    double * _weights;
  };

  inline void GaussianMixtureDistribution::setMode(int mode, int dim, double mean, double stddev)
  {
    _modes[mode].setMean(dim, mean);
    _modes[mode].setStddev(dim, stddev);
  }

  inline void GaussianMixtureDistribution::setFactor(int mode)
  {
    _modes[mode].setFactor();
  }

} // namespace QGpCoreStat

#endif // GAUSSIANMIXTUREDISTRIBUTION_H

