/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-11-18
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "WindowFunctionParameters.h"

namespace QGpCoreMath {

  /*!
    \class WindowFunctionParameters WindowFunctionParameters.h
    \brief Parameters for taper computation

    The details about each window type is given at
    http://en.wikipedia.org/wiki/Window_function
  */

  /*!
    Defaults to Tukey window or tapered cosine window. This was originally
    the only type of window available in Geopsy.
  */
  WindowFunctionParameters::WindowFunctionParameters()
    : AbstractParameters()
  {
    _shape=Tukey;
    _reversed=false;
    _alpha=0.5;
  }

  ENUM_AS_STRING_BEGIN(WindowFunctionParameters, Shape)
  ENUM_AS_STRING_DATA_18(Bartlett, BartlettHann, Blackman, BlackmanHarris, BlackmanNuttall,
                         Cosine, FlatTop, Gaussian, Hamming, Hann, KonnoOhmachi, Lanczos,
                         Nuttall, Parzen, Rectangular, Triangular, Tukey, Welch);
  ENUM_AS_STRING_SYNONYM("Tuckey", Tukey);
  ENUM_AS_STRING_END

  QString WindowFunctionParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=prefix+"WINDOW_TYPE"+suffix+"="+convertShape(_shape)+"\n";
    log+=prefix+"WINDOW_REVERSED"+suffix+"="+(_reversed ? "y" : "n")+"\n";
    switch(_shape) {
    case Tukey:
    case Blackman:
      log+=prefix+"WINDOW_ALPHA"+suffix+"="+QString::number(alpha())+"\n";
      break;
    case Gaussian:
      log+=prefix+"WINDOW_SIGMA"+suffix+"="+QString::number(sigma())+"\n";
      break;
    default:
      break;
    }
    return log;
  }

  int WindowFunctionParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 4+AbstractParameters::keywordCount();
  }

  void WindowFunctionParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=AbstractParameters::keywordCount();
    keywords.add(prefix+"WINDOW_TYPE"+suffix, this, baseIndex);
    keywords.add(prefix+"WINDOW_REVERSED"+suffix, this, baseIndex+1);
    keywords.add(prefix+"WINDOW_SIGMA"+suffix, this, baseIndex+2);
    keywords.add(prefix+"WINDOW_ALPHA"+suffix, this, baseIndex+3);
  }

  bool WindowFunctionParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    bool ok=true;
    switch(index-AbstractParameters::keywordCount()) {
    case 0:
      _shape=convertShape(value, ok);
      return ok;
    case 1:
      setReversed(value=="y");
      return true;
    case 2:
      setSigma(value.toDouble(&ok));
      return ok;
    case 3:
      setAlpha(value.toDouble(&ok));
      return ok;
    default:
      break;
    }
    return AbstractParameters::setValue(index, value, unit, keywords);
  }

  QString WindowFunctionParameters::toShortString() const
  {
    TRACE;
    QString args;
    if(_reversed) {
      args+="reversed ";
    }
    args+=convertShape(shape());
    switch(_shape) {
    case WindowFunctionParameters::Tukey:
    case WindowFunctionParameters::Blackman:
    case WindowFunctionParameters::Gaussian:
      args+=" ";
      args+=QString::number(_alpha);
      break;
    default:
      break;
    }
    return args;
  }

  void WindowFunctionParameters::fromShortString(QString s)
  {
    TRACE;
    // Default value
    _shape=Tukey;
    _alpha=0.2;
    _reversed=false;
    // Remove useless spaces
    s=s.trimmed();
    if(s.startsWith("reversed ")) {
      _reversed=true;
      s=s.mid(9, -1);
    }
    bool ok=true;
    Shape shape=convertShape(s.section(QRegularExpression(" +"), 0, 0), ok);
    if(ok) {
      setShape(shape);
      switch(_shape) {
      case WindowFunctionParameters::Tukey:
      case WindowFunctionParameters::Blackman:
      case WindowFunctionParameters::Gaussian:
        _alpha=s.section(QRegularExpression(" +"), 1, 1).toDouble();
        break;
      default:
        break;
      }
    }
  }

} // namespace QGpCoreMath
