/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-03-04
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "SamplingParameters.h"

namespace QGpCoreMath {
//#define COMPATIBILITY_2_5_0
  /*!
    \class SamplingParameters SamplingParameters.h
    \brief An axis sampling definition

    Full description of class still missing
  */

  SamplingParameters::SamplingParameters()
    : AbstractParameters()
  {
    TRACE;
    _minIndex=0.0;
    _min=0.0;
    _max=0.0;
    _step=0.0;
    _scaleType=Linear;
    _stepType=Step;
    _n=0;
  }

  /*!
    Set the number of samples according to the provided \a step
    Minimum, maximum and the type of sampling must be properly set.
  */
  void SamplingParameters::setStep(double step)
  {
    TRACE;
    _step=step;
    _stepType=Step;
    setCount();
  }

  /*!
    Set the number of samples. They are consistent with reference (0 for linear, 1 for log).
    The first and the last value are not exactly the minimum and the maximum.
  */
  void SamplingParameters::setCount(int n)
  {
    _n=n;
    if(_n<1) {
      _n=1;
    }
    _stepType=Count;
    setCount();
  }

  /*!
    Set the number of samples. They are NOT consistent with reference (0 for linear, 1 for log).
    The first and the last value are exactly the minimum and the maximum.
  */
  void SamplingParameters::setExactCount(int n)
  {
    _n=n;
    if(_n<1) {
      _n=1;
    }
    _stepType=ExactCount;
    setCount();
  }

  void SamplingParameters::setRange(double min, double max)
  {
    _min=min;
    _max=max;
    if(_min==_max) {
      setCount(1);
    } else {
      setCount();
    }
  }

  void SamplingParameters::setCount()
  {
    if(_min<_max) {
      switch(_stepType) {
      case Step:
        if(_step>0.0) {
          double nd=0.0;
          switch(_scaleType) {
          case Linear:
            _minIndex=ceil(_min/_step);
            nd=1.0+floor(_max/_step)-_minIndex;
            break;
          case Inverse:
            _minIndex=floor(1.0/(_min*_step));
            nd=1.0+ceil(_minIndex-1.0/(_max*_step));
            break;
          case Log:
            if(_min>0.0 && _step>1.0) {
              double logStep=std::log(_step);
              _minIndex=ceil(std::log(_min)/logStep);
              nd=1.0+floor(std::log(_max)/logStep-_minIndex);
            }
            break;
          }
          if(nd>INT_MAX/2) {      // Yet a huge number, keep away from INT_MAX to avoid overflow in IrregularGrid::initLookup()
            setCount(INT_MAX/2-1);
          } else {
            _n=nd;
          }
        }
        break;
      case Count:
        if(_n>1) {
          switch(_scaleType) {
          case Linear:
            _step=(_max-_min)/static_cast<double>(_n-1);
#ifndef COMPATIBILITY_2_5_0
            _minIndex=ceil(_min/_step);
            if(value(_n-1)>_max) { // Ensures that effective min and max are inside _min and _max
              _step=(_max-_min)/static_cast<double>(_n);
              _minIndex=ceil(_min/_step);
            }
#endif
            break;
          case Inverse:
            _step=(1.0/_min-1.0/_max)/static_cast<double>(_n-1);
            _minIndex=ceil(1.0/(_min*_step))-1;
            if(value(_n-1)>_max) { //Ensures that effective min and max are inside _min and _max
              _step=(1.0/_min-1.0/_max)/static_cast<double>(_n);
              _minIndex=ceil(1.0/(_min*_step))-1;
            }
            break;
          case Log:
            _step=std::pow(_max/_min, 1.0/static_cast<double>(_n-1));
#ifndef COMPATIBILITY_2_5_0
            _minIndex=ceil(std::log(_min)/std::log(_step));
            if(value(_n-1)>_max) { //Ensures that effective min and max are inside _min and _max
              _step=std::pow(_max/_min, 1.0/static_cast<double>(_n));
              _minIndex=ceil(std::log(_min)/std::log(_step));
            }
#endif
            break;
          }
        } else if(_n==1) {
          _minIndex=1;
          _scaleType=Linear;
          _step=_min;
        }
        break;
      case ExactCount:
        switch(_scaleType) {
        case Linear:
          _step=(_max-_min)/static_cast<double>(_n-1);
          _minIndex=_min/_step;
          break;
        case Inverse:
          _step=(1.0/_min-1.0/_max)/static_cast<double>(_n-1);
          _minIndex=1.0/(_min*_step);
          break;
        case Log:
          _step=std::pow(_max/_min, 1.0/static_cast<double>(_n-1));
          _minIndex=std::log(_min)/std::log(_step);
          break;
        }
        break;
      }
    } else if(_n==1 && _min==_max) {
      _minIndex=1;
      _scaleType=Linear;
      _step=_min;
    } else {
      // Set as null the calculated value
      switch(_stepType) {
      case Count:
      case ExactCount:
        _step=0.0;
        break;
      case Step:
        _n=0;
        break;
      }
      _minIndex=0;
    }
    APP_LOG(3, tr("SamplingParameters::setCount %1\n").arg(toUserString()));
  }

  double SamplingParameters::value(int i) const
  {
    switch(_scaleType) {
    case Linear:
#ifdef COMPATIBILITY_2_5_0
      return _min+_step*i;
#else
      return _step*(i+_minIndex);
#endif
    case Inverse:
      return 1.0/(_step*(_minIndex-i));
    case Log:
      break;
    }
#ifdef COMPATIBILITY_2_5_0
    return _min*std::pow(_step, i);
#else
    return std::pow(_step, i+_minIndex);
#endif
  }

  /*!
    Return the closest index to \a val
  */
  int SamplingParameters::index(double val) const
  {
    int i;
    SAFE_UNINITIALIZED(i, 0);
    switch(_scaleType) {
    case Linear:
#ifdef COMPATIBILITY_2_5_0
      i=qRound(val-_min)/_step);
#else
      i=qRound(val/_step)-_minIndex;
#endif
      break;
    case Inverse:
      i=_minIndex-qRound(1.0/(val*_step));
      break;
    case Log:
#ifdef COMPATIBILITY_2_5_0
      i=qRound(std::log(val/_min)/std::log(_step));
#else
      i=qRound(std::log(val)/std::log(_step))-_minIndex;
#endif
      break;
    }
    if(i<0) {
      i=0;
    } else if(i>=_n) {
      i=_n-1;
    }
    return i;
  }

  VectorList<double> SamplingParameters::values() const
  {
    VectorList<double> v(_n);
    for(int i=0; i<_n; i++) {
      v[i]=value(i);
    }
    return v;
  }

  /*!
    Changes scale type. If \a keepCount is true. keeps the number of samples constant
    even if stepType is Step.
  */
  void SamplingParameters::setScaleType(ScaleType t, bool keepCount)
  {
    if(_scaleType!=t) {
      _scaleType=t;
      if(keepCount && _stepType==Step) {
        _stepType=Count;
        setCount();
        _stepType=Step;
      } else {
        setCount();
      }
    }
  }

  /*!
    Converts SamplingOptions to ScaleType
  */
  SamplingParameters::ScaleType SamplingParameters::scaleType(SamplingOptions s)
  {
    if(s & LogScale) {
      return Log;
    }
    if(s & InverseScale) {
      return Inverse;
    }
    return Linear;
  }

  SamplingOptions SamplingParameters::samplingOptions(ScaleType t)
  {
    TRACE;
    switch(t) {
    case Linear:
      break;
    case Inverse:
      return LinearScale | InverseScale;
    case Log:
      return LogScale;
    }
    return LinearScale;
  }

  ENUM_AS_STRING_BEGIN(SamplingParameters, ScaleType)
  ENUM_AS_STRING_DATA_3(Inverse, Linear, Log);
  ENUM_AS_STRING_SYNONYM("0", Log);
  ENUM_AS_STRING_SYNONYM("1", Linear);
  ENUM_AS_STRING_SYNONYM("Period", Inverse);
  ENUM_AS_STRING_SYNONYM("Frequency", Linear);
  ENUM_AS_STRING_SYNONYM("period", Inverse);   // Used for command line arguments
  ENUM_AS_STRING_SYNONYM("frequency", Linear);
  ENUM_AS_STRING_SYNONYM("log", Log);
  ENUM_AS_STRING_END

  ENUM_AS_STRING_BEGIN(SamplingParameters, StepType)
  ENUM_AS_STRING_DATA_3(Count, ExactCount, Step);
  ENUM_AS_STRING_END

  int SamplingParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 8+AbstractParameters::keywordCount();
  }

  void SamplingParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    int baseIndex=AbstractParameters::keywordCount();
    keywords.add(prefix+"MINIMUM"+suffix, this, baseIndex);
    keywords.add(prefix+"MAXIMUM"+suffix, this, baseIndex+1);
    keywords.add(prefix+"SAMPLING_TYPE"+suffix, this, baseIndex+3); // Kept for compatibility
    keywords.add(prefix+"SCALE_TYPE"+suffix, this, baseIndex+7);
    keywords.add(prefix+"INVERSED"+suffix, this, baseIndex+4);      // Kept for compatibility
    keywords.add(prefix+"STEP_TYPE"+suffix, this, baseIndex+6);
    keywords.add(prefix+"SAMPLES_NUMBER"+suffix, this, baseIndex+2);
    keywords.add(prefix+"STEP"+suffix, this, baseIndex+5);
  }

  bool SamplingParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    bool ok=true;
    switch(index-AbstractParameters::keywordCount()) {
    case 0:
      setMinimum(value.toDouble(&ok));
      return ok;
    case 1:
      setMaximum(value.toDouble(&ok));
      return ok;
    case 2:
      setCount(value.toInt());
      return true;
    case 3:     // Kept for compatibility
    case 7:
      setScaleType(convertScaleType(value, ok));
      setCount();
      return ok;
    case 4:      // Kept for compatibility
      if(value=="y") {
        setScaleType(Inverse);
      }
      return true;
    case 5:
      setStep(value.toDouble(&ok));
      return ok;
    case 6:
      setStepType(convertStepType(value, ok));
      return true;
    default:
      break;
    }
    return AbstractParameters::setValue(index, value, unit, keywords);
  }

  QString SamplingParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    QString log;
    log+=prefix+"MINIMUM"+suffix+"="+QString::number(_min)+"\n";
    log+=prefix+"MAXIMUM"+suffix+"="+QString::number(_max)+"\n";
    log+="# Either 'Linear', 'Log' or 'Inverse'\n";
    log+=prefix+"SCALE_TYPE"+suffix+"="+convertScaleType(_scaleType)+"\n";
    log+="# Number of samples is either set to a fixed value ('Count') or through a step between samples ('Step')'\n";
    log+=prefix+"STEP_TYPE"+suffix+"="+convertStepType(_stepType)+"\n";
    log+=prefix+"SAMPLES_NUMBER"+suffix+"="+QString::number(_n)+"\n";
    log+="# STEP=difference between two successive samples for 'linear' scales\n";
    log+="# STEP=ratio between two successive samples for 'log' scales\n";
    log+="# STEP=difference between two successive inverse samples for 'inverse' scales\n";
    log+=prefix+"STEP"+suffix+"="+QString::number(_step)+"\n";
    return log;
  }

  QString SamplingParameters::toUserString() const
  {
    return tr("from %1 to %2 with %3 %4 samples (step=%5)")
        .arg(_n>1 ? value(0) : _min)
        .arg(_n>1 ? value(_n-1) : _max)
        .arg(_n)
        .arg(convertScaleType(_scaleType))
        .arg(_step);
  }

  bool SamplingParameters::isValid() const
  {
    if(_n>1) {
      return std::isfinite(_min) && std::isfinite(_max) && _min<_max && _step>0.0;
    } else if(_n==1) {
      return std::isfinite(_min);
    } else {
      return false;
    }
  }

  bool SamplingParameters::isValidRange(bool log) const
  {
    if(std::isfinite(_min) && std::isfinite(_max)) {
      switch(_scaleType) {
      case Log:
      case Inverse:
        if(_min<=0.0 || _max<=0.0) {
          if(log) {
            App::log(tr("Null or negative values not allowed for log and inverse scales\n"));
          }
          return false;
        }
        break;
      case SamplingParameters::Linear:
        break;
      }
      return true;
    } else {
      if(log) {
        App::log(tr("Sampling with infinite limits\n"));
      }
      return false;
    }
  }

  /*!
    \a values must be sorted and unique

    TODO check linear and log
  */
  bool SamplingParameters::bestFit(const VectorList<double>& values)
  {
    if(values.count()<3) {
      App::log(tr("Sampling best fit: a minimum of 3 values is required\n") );
      return false;
    }
    double dLin0, dLog0, dInv0, dLin, dLog, dInv;
    int n=values.count();
    // Get the smallest step for each type of sampling
    dLin0=values.at(1)-values.at(0);
    dLog0=values.at(1)/values.at(0);
    dInv0=1.0/values.at(0)-1.0/values.at(1);
    for(int i=2; i<n; i++) {
      dLin=values.at(i)-values.at(i-1);
      if(dLin<dLin0) {
        dLin0=dLin;
      }
      dLog=values.at(i)/values.at(i-1);
      if(dLog<dLog0) {
        dLog0=dLog;
      }
      dInv=1.0/values.at(i-1)-1.0/values.at(i);
      if(dInv<dInv0) {
        dInv0=dInv;
      }
    }
    App::log(1, tr("Minimum steps: lin=%1, log=%2, inv=%3\n")
             .arg(dLin0).arg(dLog0).arg(dInv0));

    double sumLin=0.0, sumLog=0.0, sumInv=0.0;
    double nStepLin, nStepLog, nStepInv;
    for(int i=1; i<n; i++) {
      dLin=values.at(i)-values.at(i-1);
      dLog=values.at(i)/values.at(i-1);
      dInv=1.0/values.at(i-1)-1.0/values.at(i);
      nStepLin=round(dLin/dLin0);
      nStepLog=round(::log(dLog)/::log(dLog0));
      nStepInv=round(dInv/dInv0);
      sumLin+=fabs(nStepLin*dLin0-dLin);
      sumLog+=fabs(::pow(dLog0, nStepLog)-dLog);
      sumInv+=fabs(nStepInv*dInv0-dInv);
    }

    if(n>0) {
      App::log(1, tr("Value range: %1, %2\n").arg(values.first()).arg(values.last()));
      setMinimum(values.first()*(1.0-1e-10));
      setMaximum(values.last()*(1.0+1e-10));
    }

    if(sumLin<=sumLog && sumLin<=sumInv) {
      setScaleType(Linear);
      setStep(dLin0);
      return true;
    } else if(sumLog<=sumInv){
      setScaleType(Log);
      setStep(dLog0);
    } else {
      setScaleType(Inverse);
      setStep(dInv0);
    }
    return isValidRange(true);
  }

  /*!
    Usually we want the sampling to be inside the limits min and max.
    This function ensure the contrary: the limits are inside the sampling range.
  */
  void SamplingParameters::includeLimits()
  {
    if(isValid()) {
      if(_min<value(0)) {
        if(_scaleType==Inverse) {
          _minIndex++;
        } else {
          _minIndex--;
        }
        _n++;
        _min=value(0);
      }
      if(_max>value(_n-1)) {
        if(_scaleType==Inverse) {
          if(_minIndex>_n) { // Avoid division by 0
            _n++;
            _max=value(_n-1);
          }
        } else {
          _n++;
          _max=value(_n-1);
        }
      }
    }
  }

  /*!
    Transforms into an equivalent linear scale.
  */
  void SamplingParameters::linearize()
  {
    switch(_scaleType) {
    case Log:
      _min=log(_min);
      _max=log(_max);
      _stepType=ExactCount;
      setScaleType(Linear);
      break;
    case Inverse:
      _min=1.0/_min;
      _max=1.0/_max;
      qSwap(_min, _max);
      _stepType=ExactCount;
      setScaleType(Linear);
      break;
    case Linear:
      break;
    }
  }

} // namespace QGpCoreMath
