/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-04-03
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "MultivariateStatistics.h"

namespace QGpCoreMath {

  /*!
    \class MultivariateStatistics MultivariateStatistics.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  MultivariateStatistics::MultivariateStatistics(int dim)
    : _sum(dim, 0.0), _sum2(dim, dim)
  {
    _sum2.zero();
    _sumWeight=0.0;
    _sumWeight2=0.0;
  }

  /*!
    Description of destructor still missing
  */
  MultivariateStatistics::~MultivariateStatistics()
  {
  }

  void MultivariateStatistics::add(const Vector<double>& val, double weight)
  {
    int n=_sum.count();
    for(int i=0; i<n; i++) {
      double c=weight*val[i];
      _sum[i]+=c;
      for(int j=0; j<n; j++) {
        _sum2.at(i, j)+=c*val[j];
      }
    }
    _sumWeight+=weight;
    _sumWeight2+=weight*weight;
  }

  void MultivariateStatistics::mean(Vector<double>& m) const
  {
    if(_sumWeight<=0.0) {
      m.init(std::numeric_limits<double>::quiet_NaN());
    } else {
      m=_sum;
      m/=_sumWeight;
    }
  }

  /*!
    Unbiased weighted sample variance
    https://en.wikipedia.org/wiki/Weighted_arithmetic_mean#Related_concepts
    https://en.wikipedia.org/wiki/Variance#Unbiased_sample_variance

    If all N weights are equal to 1, we have
      sigma^2=(_sum2-_sum^2/N)/(N-1)
  */
  DoubleMatrix MultivariateStatistics::variance() const
  {
    DoubleMatrix v(_sum2);
    if(_sumWeight>0.0) {
      double invSumWeight=1.0/_sumWeight;
      double unbiasFactor=_sumWeight-_sumWeight2*invSumWeight;
      if(unbiasFactor>0.0) {
        int n=_sum.count();
        double s2;
        for(int i=0; i<n; i++) {
          v.at(i, i)-=_sum[i]*_sum[i]*invSumWeight;
          for(int j=i+1; j<n; j++) {
            s2=_sum[i]*_sum[j]*invSumWeight;
            v.at(i, j)-=s2;
            v.at(j, i)-=s2;
          }
        }
        v/=unbiasFactor;
      }
    }
    return v;
  }

} // namespace QGpCoreMath

