/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-12-05
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef MULTIVARIATENORMALDISTRIBUTION_H
#define MULTIVARIATENORMALDISTRIBUTION_H

#include "PrivateVector.h"
#include "DoubleMatrix.h"
#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  class MultivariateStatistics;

  class QGPCOREMATH_EXPORT MultivariateNormalDistribution
  {
  public:
    MultivariateNormalDistribution(int dimensionCount=1);
    MultivariateNormalDistribution(const Vector<double>& mean, const DoubleMatrix& covariance);
    MultivariateNormalDistribution(const MultivariateStatistics& stat);
    MultivariateNormalDistribution(const MultivariateNormalDistribution& o)
      : _mean(o._mean), _invVariance(o._invVariance), _factor(o._factor) {}

    void operator=(const MultivariateNormalDistribution& o);
    void init(int dimension);

    bool isSimilar(const MultivariateNormalDistribution& o, double precision);

    int count() const {return _mean.count();}

    const Vector<double>& mean() const {return _mean;}
    double mean(int index) const {return _mean[index];}

    DoubleMatrix variance() const;
    inline double stddev(int index) const;
    void stddev(Vector<double>& s) const;

    void setMean(int index, double m) {_mean[index]=m;}
    void setMean(const Vector<double>& m) {_mean=m;}

    void setVariance(const DoubleMatrix& v);
    void clearVariance();
    void setStddev(int index, double s);
    void setFactor();

    double value(const Vector<double>& x) const;
    double value1D(int index, double x) const;
    double cumulativeValue1D(int index, double x) const;

    bool overlap(const MultivariateNormalDistribution& o) const;
  private:
    PrivateVector<double> _mean;
    DoubleMatrix _invVariance;
    double _factor;
  };

  inline double MultivariateNormalDistribution::stddev(int index) const
  {
    return 1.0/sqrt(_invVariance.at(index, index));
  }

} // namespace QGpCoreMath

#endif // MULTIVARIATENORMALDISTRIBUTION_H

