/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-03-20
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "CubicEquation.h"

namespace QGpCoreMath {

  /*!
    \class CubicEquation CubicEquation.h
    \brief Brief description of class still missing

    Solver for cubic equations with real coefficients
  */

  /*!
    Description of constructor still missing

    https://en.wikipedia.org/wiki/Cubic_equation
  */
  CubicEquation::CubicEquation()
  {
    _third=1.0/3.0;
    _unityRoot=-3.0;
    _unityRoot=sqrt(_unityRoot);
    _unityRoot-=1.0;
    _unityRoot*=0.5;
    _unityRoot2=_unityRoot;
    _unityRoot2.conjugate();  // xi^2
  }

  void CubicEquation::setEquation(double a, double b, double c, double d)
  {
    _a=a;
    _b=b;
    _c=c;
    _d=d;
    // Its roots
    double b2=_b*_b;
    double delta0=b2-3.0*_a*_c;
    double delta1=2.0*b2*_b-9.0*_a*_b*_c+27.0*_a*_a*_d;
    double delta03=delta0*delta0*delta0;
    double delta12=delta1*delta1;
    _delta=(4.0*delta03-delta12)/(27.0*a*a);

    double const third=1.0/3.0;
    Complex delta1203=delta12-4.0*delta03;
    delta1203=sqrt(delta1203);
    Complex deltaC=pow(0.5*(delta1+delta1203),third);
    double inv3a=-1.0/(3.0*a);
    _roots[0]=inv3a*(b+deltaC+delta0/deltaC);
    _roots[1]=inv3a*(b+deltaC*_unityRoot+delta0/(deltaC*_unityRoot));
    _roots[2]=inv3a*(b+deltaC*_unityRoot2+delta0/(deltaC*_unityRoot2));
    if(_delta>0.0) {
      // Sort the three roots
      for(int j=1; j<3; j++) {
        for(int i=1; i<3; i++) {
          if(_roots[i-1].re()>_roots[i].re()) {
            qSwap(_roots[i-1], _roots[i]);
          }
        }
      }
    } // One root, this is always the first one
  }

  double CubicEquation::value(double x) const
  {
    double x2=x*x;
    return _a*x2*x+_b*x2+_c*x+_d;
  }

  Complex CubicEquation::value(const Complex& x) const
  {
    Complex x2=x;
    x2*=x;
    return _a*x2*x+_b*x2+_c*x+_d;
  }

  double CubicEquation::firstDerivative(double x) const
  {
    return 3.0*_a*x*x+2.0*_b*x+_c;
  }

  void CubicEquation::checkRoots() const
  {
    App::log(tr("Root 1 at %1=%2\n")
             .arg(_roots[0].toString())
             .arg(value(_roots[0]).toString()));
    App::log(tr("Root 2 at %1=%2\n")
             .arg(_roots[1].toString())
             .arg(value(_roots[1]).toString()));
    App::log(tr("Root 3 at %1=%2\n")
             .arg(_roots[2].toString())
             .arg(value(_roots[2]).toString()));
  }


} // namespace QGpCoreMath

