/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-05-11
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ColorPaletteModels.h"
#include "ColorPalette.h"
#ifdef Q_OS_DARWIN
#include "QGpCoreMathVersion.h"
#endif

namespace QGpCoreMath {

  /*!
    \class ColorPaletteModels ColorPaletteModels.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  ColorPaletteModels * ColorPaletteModels::_self=nullptr;

  /*!
    Description of constructor still missing
  */
  ColorPaletteModels::ColorPaletteModels()
  {
  }

  /*!
    Description of destructor still missing
  */
  ColorPaletteModels::~ColorPaletteModels()
  {
    QMap<QString, Palette>::iterator it;
    for(it=_palettes.begin(); it!=_palettes.end(); it++) {
      delete it.value()._pal;
    }
    _palettes.clear();
  }

  ColorPaletteModels * ColorPaletteModels::instance()
  {
    if(!_self) {
      _self=new ColorPaletteModels;
      CoreApplication::instance()->addGlobalObject(_self);
    }
    return _self;
  }

  QString ColorPaletteModels::path()
  {
#ifdef Q_OS_DARWIN
    QString path("/Library/Geopsy.org/%1/share");
    QDir d(path.arg(QGPCOREMATH_VERSION_TYPE));
#elif defined(Q_OS_WIN)
    QDir d(QCoreApplication::applicationDirPath());
    d.cdUp();
    d.cd("share");
#else
    const PackageInfo * p=PackageInfo::package("QGpCoreMath");
    QDir d(p->shareDir());
#endif
    d.cd("QGpCoreMath/palettes");
    return d.absolutePath();
  }

  QList<QDir> ColorPaletteModels::sources()
  {
    QDir d(path());
    QStringList sources=d.entryList(QDir::AllDirs | QDir::NoDotAndDotDot);
    QList<QDir> sourceDirs;
    int n=sources.count();
    for(int i=0; i<n; i++) {
      sourceDirs.append(d.absoluteFilePath(sources.at(i)));
      APP_LOG(1, tr("Palette source dir: %1\n").arg(sourceDirs.last().absolutePath()));
    }
    if(n!=instance()->_credits.count()) {
      // Usually the palette directory does not change during execution.
      // This initialization is usually run once at the beginning.
      instance()->_palettes.clear();  // Make sure that credit index is consistent
      instance()->_credits.clear();
      for(int i=0; i<n; i++) {
        QFile f(sourceDirs.at(i).absoluteFilePath("CREDITS"));
        if(f.open(QIODevice::ReadOnly)) {
          instance()->_credits.append(f.readAll());
        } else {
          instance()->_credits.append(QString());
        }
      }
    }
    return sourceDirs;
  }

  ColorPalette * ColorPaletteModels::loadPalette(const QString& fileName)
  {
    QFile f(fileName);
    if(f.open(QIODevice::ReadOnly)) {
      ColorPalette * pal=new ColorPalette;
      pal->fromString(f.readAll());
      if(pal->count()>0) {
        return pal;
      } else {
        delete pal;
        App::log(tr("Palette '%1' is empty\n")
                 .arg(fileName));
        return nullptr;
      }
    } else {
      App::log(tr("Cannot open palette '%1'\n").arg(fileName));
      return nullptr;
    }
  }

  const ColorPalette * ColorPaletteModels::palette(const QString& model)
  {
    QMap<QString, Palette>::iterator it=_palettes.find(model);
    if(it==_palettes.end()) {
      QList<QDir> sList=sources();
      for(int i=sList.count()-1; i>=0; i--) {
        const QDir& d=sList.at(i);
        if(d.exists(model)) {
          ColorPalette * pal=loadPalette(d.absoluteFilePath(model));
          if(pal) {
            _palettes.insert(model, Palette(pal, i));
            return pal;
          }
        }
      }
      App::log(tr("Color palette model '%1' not found in '%2'\n")
               .arg(model)
               .arg(path()));
      if(_palettes.isEmpty()) {
        App::log(tr("  generating a standard gray scale palette...\n"));
        ColorPalette * pal=new ColorPalette;
        pal->sardineGrayScale(0, 255);
        _palettes.insert(model, Palette(pal, -1));
        return pal;
      }
      App::log(tr("  using '%1' as a replacement\n").arg(_palettes.firstKey()));
      return _palettes.first()._pal;
    } else {
      return it.value()._pal;
    }
  }

  /*!
    A successfull call to palette() is required before calling this function.
    It does not query disk.
  */
  QString ColorPaletteModels::credits(const QString& model)
  {
    QMap<QString, Palette>::iterator it=_palettes.find(model);
    if(it!=_palettes.end()) {
      int i=it.value()._creditIndex;
      if(i>=0) {
        return _credits.at(i);
      }
    }
    return QString();
  }

  QStringList ColorPaletteModels::list() const
  {
    QStringList l;
    QList<QDir> sList=sources();
    for(int i=sList.count()-1; i>=0; i--) {
      const QDir& d=sList.at(i);
      QStringList files=d.entryList(QDir::Files);
      for(int j=files.count()-1; j>=0; j--) {
        const QString& f=files.at(j);
        if(f!="CREDITS") {
          l.append(d.absoluteFilePath(f));
        }
      }
    }
    std::sort(l.begin(), l.end());
    return l;
  }

} // namespace QGpCoreMath

