/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-11-08
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ColorPalette.h"
#include "Point.h"
#include "DoubleMatrix.h"
#include "ColorPaletteModels.h"

namespace QGpCoreMath {

  /*!
    \class ColorPalette ColorPalette.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  const QString ColorPalette::defaultSequential="batlow";
  const QString ColorPalette::defaultSequentialWhite="batlowW";
  const QString ColorPalette::defaultDiverging="vik";
  const QString ColorPalette::defaultMultiSequential="bukavu";
  const QString ColorPalette::defaultCyclic="vikO";
  const QString ColorPalette::defaultCategorical="batlowS";

  /*!
    Description of constructor still missing
  */
  ColorPalette::ColorPalette()
  {
    TRACE;
    _n=0;
    _colors=nullptr;
  }

  ColorPalette::ColorPalette(const ColorPalette& o)
  {
    TRACE;
    _colors=nullptr;
    *this=o;
  }

  /*!
    Description of destructor still missing
  */
  ColorPalette::~ColorPalette()
  {
    TRACE;
    delete [] _colors;
  }

  void ColorPalette::operator=(const ColorPalette& o)
  {
    TRACE;
    _n=o._n;
    delete [] _colors;
    _colors=new Color[_n];
    for(int i=0; i<_n; i++) {
      _colors[i]=o._colors[i];
    }
  }

  bool ColorPalette::operator==(const ColorPalette& o) const
  {
    TRACE;
    if(_n!=o._n) {
      return false;
    }
    for(int i=0; i<_n; i++) {
      if(_colors[i]!=o._colors[i]) {
        return false;
      }
    }
    return true;
  }

  /*!
    All current colors are lost.
  */
  void ColorPalette::resize(int n)
  {
    TRACE;
    if(n<1) {
      n=1;
    }
    if(!_colors || n!=_n) {
      if(_colors) {
        delete [] _colors;
      }
      _colors=new Color[n];
      _n=n;
    }
  }

  /*!
    Interpole color
  */
  Color ColorPalette::color(double index) const
  {
    int i0=qFloor(index);
    double di=index-i0;
    const Color& a=_colors[i0];
    const Color& b=_colors[i0+1];
    Color c;
    c.setRed64(a.red64()*(1.0-di)+b.red64()*di);
    c.setGreen64(a.green64()*(1.0-di)+b.green64()*di);
    c.setBlue64(a.blue64()*(1.0-di)+b.blue64()*di);
    return c;
  }

  void ColorPalette::generate(int imin, int imax, const QString& model,
                              Options options, quint16 transparency)
  {
    TRACE;
    if(imin<0) {
      imin=0;
    }
    if(imax>=_n) {
      resize(imax+1);
    }
    const ColorPalette * ref=ColorPaletteModels::instance()->palette(model);
    int lastIndex=ref->count()-1;
    if(options & Categorical) {
      int index;
      if(options & Reversed) {
        for(int i=imin; i<=imax; i++) {
          index=lastIndex-(i-imin);
          while(index>=0) {
            index+=ref->count();
          }
          _colors[i]=ref->color(lastIndex-(i-imin));
          _colors[i].setAlpha64(transparency);
        }
      } else {
        for(int i=imin; i<=imax; i++) {
          index=i-imin;
          while(index>lastIndex) {
            index-=ref->count();
          }
          _colors[i]=ref->color(index);
          _colors[i].setAlpha64(transparency);
        }
      }
    } else {
      double fac=lastIndex;
      if(imax-imin>1) {
        fac/=static_cast<double>(imax-imin);
      } else {
        fac=0.0;
      }
      if(options & Reversed) {
        for(int i=imin; i<=imax; i++) {
          _colors[i]=ref->color(lastIndex-qRound((i-imin)*fac));
          _colors[i].setAlpha64(transparency);
        }
      } else {
        for(int i=imin; i<=imax; i++) {
          _colors[i]=ref->color(qRound((i-imin)*fac));
          _colors[i].setAlpha64(transparency);
        }
      }
    }
  }

  /*!
    if \a b is true, all white colors are set transparent. If false,
    all white colors are set opaque.
  */
  void ColorPalette::setWhiteTransparent(bool b)
  {
    TRACE;
    for(int i=0; i<_n; i++) {
      Color& c=_colors[i];
      if(c.isNearlyWhite()) {
        c.setAlpha64(b ? 0x0000 : 0xFFFF);
      }
    }
  }

  void ColorPalette::toGray()
  {
    TRACE;
    for(int i=0; i<_n; i++) {
      _colors[i].toGray();
    }
  }

  /*!
    Implementation inherited from Sardine (D. Demanet).
    It was the default gray palette under Sardine
  */
  void ColorPalette::sardineGrayScale(int imin, int imax)
  {
    TRACE;
    if(imin<0) {
      imin=0;
    }
    if(imax>=_n) {
      resize(imax+1);
    }
    double x;
    double fac=1.0/static_cast<double>(imax-imin);
    for(int i=imin; i<=imax; i++) {
      x=static_cast<double>(i-imin)*fac;
      x=1.0-x*x;
      _colors[i].setRgba64(qRound(x*COLOR_COMPONENT_MAX),
                           qRound(x*COLOR_COMPONENT_MAX),
                           qRound(x*COLOR_COMPONENT_MAX), COLOR_COMPONENT_MAX);
    }
  }

  void ColorPalette::hsv(int n, int sat, int value, int transparency)
  {
    TRACE;
    resize(n);
    for(int i=0; i<_n; i++) {
      _colors[i].setHsva(qRound(static_cast<double>(i)*65535.0/static_cast<double>(_n)), sat, value, transparency);
    }
  }

  void ColorPalette::rgbInterpole(int imin, int imax)
  {
    TRACE;
    double r, r2, dr, g, g2, dg, b, b2, db, a, a2, da;
    if(_n<=2) {
      return;
    }
    if(imin<0) {
      imin=0;
    }
    if(imax>=_n) {
      imax=_n-1;
    }

    Color& c1=_colors[imin];
    r=c1.red64();
    g=c1.green64();
    b=c1.blue64();
    a=c1.alpha64();

    Color& c2=_colors[imax];
    r2=c2.red64();
    g2=c2.green64();
    b2=c2.blue64();
    a2=c2.alpha64();

    dr=(r2-r)/static_cast<double>(imax-imin+1);
    dg=(g2-g)/static_cast<double>(imax-imin+1);
    db=(b2-b)/static_cast<double>(imax-imin+1);
    da=(a2-a)/static_cast<double>(imax-imin+1);

    for(int i=imin+1; i<imax; i++) {
      r+=dr;
      g+=dg;
      b+=db;
      a+=da;
      _colors[i].setRgba64(qRound(r), qRound(g), qRound(b), qRound(a));
    }
  }

  void ColorPalette::hsvInterpole(int imin, int imax)
  {
    TRACE;
    int h1, h2, s1, s2, v1, v2, a1, a2;
    double h, s, v, a, dh, ds, dv, da;
    if(_n<=2) {
      return;
    }
    if(imin<0) {
      imin=0;
    }
    if(imax>=_n) {
      imax=_n-1;
    }

    Color& c1=_colors[imin];
    c1.hsv(h1, s1, v1);
    a1=c1.alpha64();

    Color& c2=_colors[imax];
    c2.hsv(h2, s2, v2);
    a2=c2.alpha64();

    dh=static_cast<double>(h2-h1)/static_cast<double>(imax-imin+1);
    ds=static_cast<double>(s2-s1)/static_cast<double>(imax-imin+1);
    dv=static_cast<double>(v2-v1)/static_cast<double>(imax-imin+1);
    da=static_cast<double>(a2-a1)/static_cast<double>(imax-imin+1);

    h=h1;
    s=s1;
    v=v1;
    a=a1;
    for(int i=imin+1; i<imax; i++) {
      h+=dh;
      s+=ds;
      v+=dv;
      a+=da;
      _colors[i].setHsva(qRound(h), qRound(s), qRound(v), qRound(a));
    }
  }

  QString ColorPalette::toString() const
  {
    TRACE;
    QString s=_colors[0].toString();
    for(int i=1; i<_n; i++) {
      s+=",";
      s+=_colors[i].toString();
    }
    return s;
  }

  bool ColorPalette::fromString(const QString& s)
  {
    TRACE;
    QStringList l=s.split(",");
    if(l.count()>=2) {
      resize(l.count());
      bool ok=true;
      for(int i=0; i<_n; i++) {
        _colors[i].fromString(l.at(i), ok);
        if(!ok) {
          return false;
        }
      }
      return true;
    } else {
      return false;
    }
  }

  ColorPalette ColorPalette::preference(const QString& name, int n, const QString& model,
                                        Options options, quint16 transparency)
  {
    TRACE;
    QSettings& reg=CoreApplication::instance()->settings();
    reg.beginGroup("ColorPalettes");
    ColorPalette pal;
    if(!reg.contains(name) || !pal.fromString(reg.value(name).toString())) {
      pal.generate(0, n-1, model, options, transparency);
    }
    return pal;
  }

  void ColorPalette::setPreference(const QString& name, const ColorPalette& pal)
  {
    TRACE;
    QSettings& reg=CoreApplication::instance()->settings();
    reg.beginGroup("ColorPalettes");
    reg.setValue(name, pal.toString());
  }

  ENUM_AS_STRING_BEGIN(ColorPalette, Option)
  ENUM_AS_STRING_DATA_3(Categorical, Continuous, Reversed);
  ENUM_AS_STRING_END

} // namespace QGpCoreMath

