/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-10-07
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include "Angle.h"
#include "Point2D.h"

namespace QGpCoreMath {

  /*!
    \fn Angle::Angle()
    Init a 0 radian angle
  */

  /*!
    \fn void Angle::set(double dx, double dy)
    Set cosine and sine with compution of distance (sqrt(dx*dx+dy*dy))
  */

  /*!
    \fn void Angle::set(double dx, double dy, double r)
    Set cosine and sine without recompution of distance (sqrt(dx*dx+dy*dy))
  */

  /*!
    \fn Angle& Angle::operator=(const Angle& o)
  */

  /*!
    \fn void Angle::initDegrees()
    From angle in radians, compute angle in degrees
  */

  /*!
    \fn void Angle::initRadians()
    From cosine and sine of angle, compute angle in radians from -pi/2 to pi/2
  */

  /*!
    \fn void Angle::setDegrees(double degrees)
    Can access angle in degrees, radians, sine or cosine.
  */

  /*!
    \fn void Angle::setRadians(double radians)
    Degrees are not set, only angle in radians, sine or cosine.
    To be able to call degrees(), call initDegrees() before.
  */

  /*!
    \fn void Angle::chSign()
    Reverses angle sign.
  */

  /*!
    \fn void Angle::mirror()
    Rotates angle by 180 degrees
  */

  /*!
    \fn double Angle::geographicToMath(double angle)

    Converts a geographical azimuth in degrees (0 in north direction, clockwise) to
    an angle in radians in mathematical sense (0 in east direction, counter-clockwise).
    Returned angle is between 0 and 2*pi.

    \sa mathToGeographic(double angle)
  */

  /*!
    \fn double Angle::mathToGeographic(double angle)

    Converts an angle in radians in mathematical sense (0 in east direction, counter-clockwise) to
    geographical azimuth in degrees (0 in north direction, clockwise).
    Returned angle is between 0 and 360.

    \sa geographicToMath(double angle)
  */

  /*!
    Converts from degrees to D.MMSSsss... (degrees, minutes, seconds).
  */
  double Angle::degreesToDMS(double angle)
  {
    double sign;
    if(angle<0.0) {
      sign=-1.0;
      angle=-angle;
    } else {
      sign=1.0;
    }
    double d=floor(angle);
    angle=(angle-d)*60.0;
    double m=floor(angle);
    angle=(angle-m)*60.0;
    d+=0.01*m+0.0001*angle;
    return sign*d;
  }

  /*!
    Converts from D.MMSSsss... (degrees, minutes, seconds) to degrees.
  */
  double Angle::DMSToDegrees(double angle)
  {
    double sign;
    if(angle<0.0) {
      sign=-1.0;
      angle=-angle;
    } else {
      sign=1.0;
    }
    double d=floor(angle);
    angle=(angle-d)*100.0;
    double m=floor(angle);
    angle=(angle-m)*100.0;
    d+=m/60.0+angle/3600.0;
    return sign*d;
  }

  /*!
    Converts from degrees to D.MMmm... (degrees, minutes).
  */
  double Angle::degreesToDM(double angle)
  {
    double sign;
    if(angle<0.0) {
      sign=-1.0;
      angle=-angle;
    } else {
      sign=1.0;
    }
    double d=floor(angle);
    angle=(angle-d)*60.0;
    d+=0.01*angle;
    return sign*d;
  }

  /*!
    Converts from D.MMSSsss... (degrees, minutes) to degrees.
  */
  double Angle::DMToDegrees(double angle)
  {
    double sign;
    if(angle<0.0) {
      sign=-1.0;
      angle=-angle;
    } else {
      sign=1.0;
    }
    double d=floor(angle);
    angle=(angle-d)*100.0;
    d+=angle/60.0;
    return sign*d;
  }

  /*!
    Converts an angle string with or without a suffix to a radian value
  */
  double Angle::fromString(QString angle)
  {
    bool fromDegrees=true;
    angle=angle.toLower();
    if(angle.endsWith("deg") || angle.endsWith(tr("deg"))) {
      angle.chop(3);
    } else if(angle.endsWith("°")) {
      angle.chop(1);
    } else if(angle.endsWith("rad") || angle.endsWith(tr("rad"))) {
      angle.chop(3);
      fromDegrees=false;
    }
    bool ok;
    double a=angle.trimmed().toDouble(&ok);
    if(!ok) {
      App::log(tr("Error parsing angle '%1'\n").arg(angle));
      return 0.0;
    } else {
      if(fromDegrees) {
        a=degreesToRadians(a);
      }
      return a;
    }
  }

  /*!
    Maintains degrees and radians between 0 and 360, 0 and 2*pi, respectivelly.
  */
  void Angle::normalize()
  {
    while(_deg>=360.0) _deg-=360.0;
    while(_deg<0.0) _deg+=360.0;
    while(_rad>=2.0*M_PI) _rad-=2.0*M_PI;
    while(_rad<0.0) _rad+=2.0*M_PI;
  }

  void Angle::operator+=(const Angle& a)
  {
    _deg+=a._deg;
    setRadians(_rad+a._rad);
  }

  /*!
    Returned angle is between -M_PI to M_PI.
    \a angle must be inside [0, 2*pi[
  */
  double Angle::aroundZero(double radians)
  {
    if(radians<=0.5*M_PI) {
      return radians;
    } else if(radians<1.5*M_PI) {
      return radians-M_PI;
    } else {
      return radians-2.0*M_PI;
    }
  }

  double Angle::canonicalRadians(double angle)
  {
    TRACE;
    if(angle<0.0) {
      angle+=2.0*M_PI;
      while(angle<0.0) {
        angle+=2.0*M_PI;
      }
    } else if(angle>=2.0*M_PI) {
      angle-=2.0*M_PI;
      while(angle>=2.0*M_PI) {
        angle-=2.0*M_PI;
      }
    }
    return angle;
  }

  double Angle::canonicalDegrees(double angle)
  {
    TRACE;
    if(angle<0.0) {
      angle+=360.0;
      while(angle<0.0) {
        angle+=360.0;
      }
    } else if(angle>=360.0) {
      angle-=360.0;
      while(angle>=360.0) {
        angle-=360.0;
      }
    }
    return angle;
  }

  double Angle::canonicalGradians(double angle)
  {
    TRACE;
    if(angle<0.0) {
      angle+=400.0;
      while(angle<0.0) {
        angle+=400.0;
      }
    } else if(angle>=400.0) {
      angle-=400.0;
      while(angle>=400.0) {
        angle-=400.0;
      }
    }
    return angle;
  }

  /*!
    Return an angle difference \a a1 - \a a2 between -M_PI and M_PI
  */
  double Angle::differenceRadians(double a1, double a2)
  {
    TRACE;
    double diff=a1-a2;
    if(diff<-M_PI) {
      diff+=2.0*M_PI;
    } else if(diff>M_PI) {
      diff-=2.0*M_PI;
    }
    return diff;
  }

  /*!
    Return an angle difference \a a1 - \a a2 between -180 and 180
  */
  double Angle::differenceDegrees(double a1, double a2)
  {
    TRACE;
    double diff=a1-a2;
    if(diff<-180.0) {
      diff+=360.0;
    } else if(diff>180.0) {
      diff-=360.0;
    }
    return diff;
  }
  /*!
    Return an angle difference \a a1 - \a a2 between -200 and 200
  */
  double Angle::differenceGradians(double a1, double a2)
  {
    TRACE;
    double diff=a1-a2;
    if(diff<-200.0) {
      diff+=400.0;
    } else if(diff>200.0) {
      diff-=400.0;
    }
    return diff;
  }

  /*!
    \fn bool Angle::isInside(double angle, double min, double max)

    The difference between \a min and \a max must be less than 2*M_PI, \a min must be less than \a max
    \a angle must be from -pi/2 to pi/2.
    \a min and \a max must be betwen -2*pi and 2*pi.
  */

  void Angle::setValue(Mode m, double v)
  {
    switch(m) {
    case Degrees:
      setDegrees(v);
      break;
    case Radians:
      setRadians(v);
      break;
    case Gradians:
      setGradians(v);
      break;
    }
  }

  void Angle::setGeographicValue(Mode m, double v)
  {
    switch(m) {
    case Degrees:
      setGeographicDegrees(v);
      break;
    case Radians:
      setGeographicRadians(v);
      break;
    case Gradians:
      setGeographicGradians(v);
      break;
    }
  }

  double Angle::value(Mode m) const
  {
    switch(m) {
    case Degrees:
      break;
    case Radians:
      return radians();
    case Gradians:
      return gradians();
    }
    return degrees();
  }

} // namespace QGpCoreMath
