/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "AbstractFunction.h"
#include "FunctionSearch.h"

namespace QGpCoreMath {

  /*!
    \class AbstractFunction AbstractFunction.h
    \brief Abstract 2D function

    This function is used by GridSearch, ConcentricSearch and DirectionalSearch. Values are required for
    gridded (x,y) couples and for free couples. For gridded couples, computation of values can be cached.
  */

  /*!
    \fn AbstractFunction()
    Default constructor
  */

  /*!
    \fn virtual ~AbstractFunction()
    Destructor
  */

  /*!
    Implemement this function to calculate a discrete function.
  */
  double AbstractFunction::value(const Vector<int>& index) const
  {
    Q_UNUSED(index);
    return 0.0;
  }

  /*!
    Implemement this function to calculate a continuous function.
    Some values of \a x might be modified (optimized).
  */
  double AbstractFunction::value(Vector<double>& x) const
  {
    Q_UNUSED(x);
    return 0.0;
  }

  double AbstractFunction::valueFixed(const Vector<double>& x) const
  {
    PrivateVector<double> t(x);
    return value(t);
  }

  /*!
    Reimplement to calculate the gradient at \a x.
  */
  void AbstractFunction::gradient(const Vector<double>& x,
                                  Vector<double>& grad) const
  {
    Q_UNUSED(x);
    grad.setValue(0.0);
  }

  double AbstractFunction::admissibleStep(const Vector<double>& x,
                                          const Vector<double>& step) const
  {
    Q_UNUSED(x);
    Q_UNUSED(step);
    return 1.0;
  }

  /*!
    Reimplement to calculate the step direction at \a x.
  */
  void AbstractFunction::stepDirection(const Vector<double>& x, Vector<double>& dir) const
  {
    Q_UNUSED(x);
    dir.setValue(0.0);
  }

} // namespace QGpCoreMath
