/***************************************************************************
**
**  This file is part of QGpCompatibility.
**
**  QGpCompatibility is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCompatibility is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-10-21
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef COMPATDataPoint_H
#define COMPATDataPoint_H

#include <QGpCoreTools.h>
#include "QGpCompatibilityDLLExport.h"

namespace QGpCompatibility {

class QGPCOMPATIBILITY_EXPORT CompatDataPoint : public XMLClass
{
public:
  CompatDataPoint () {_mean=0.0; _stddev=0.0; _weight=0.0;}
  inline CompatDataPoint (double mean, double stddev, double _weight);

  double& mean() {return _mean;}
  double& stddev() {return _stddev;}
  double& weight() {return _weight;}

  const double& mean() const {return _mean;}
  const double& stddev() const {return _stddev;}
  const double& weight() const {return _weight;}

  inline double misfitOrder1(double value, double invalid);
protected:
  double _mean;
  double _stddev;
  double _weight;
protected:
  virtual const QString& xml_tagName() const {return xmlDataPointTag;}
  static const QString xmlDataPointTag;
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
  virtual bool xml_setProperty(XML_SETPROPERTY_ARGS);
};

class QGPCOMPATIBILITY_EXPORT CompatVDataPoint : public CompatDataPoint
{
public:
  CompatVDataPoint() : CompatDataPoint() {_value=0.0;}
  inline CompatVDataPoint (double mean, double stddev, double _weight);

  inline void operator=(const CompatDataPoint& o);

  void setValue(double val) {_value=val;}
  const double& value() const {return _value;}

  inline double misfitOrder2(int& nMeans, int& nValues, double invalid);
protected:
  double _value;
};

inline CompatDataPoint::CompatDataPoint(double mean, double stddev, double weight)
{
  TRACE;
  _mean=mean;
  _stddev=stddev;
  _weight=weight;
}

inline double CompatDataPoint::misfitOrder1(double value, double invalid)
{
  TRACE;
  if(value!=invalid) {
    double diff;
    if(_stddev)
      diff=(_mean - value)/_stddev;
    else
      diff=(_mean - value)/_mean;
    return (diff > 0) ? diff : -diff;
  }
  return std::numeric_limits<double>::infinity();
}

inline CompatVDataPoint::CompatVDataPoint(double mean, double stddev, double weight)
    : CompatDataPoint(mean, stddev, weight)
{
  TRACE;
  _value=mean;
}

inline void CompatVDataPoint::operator=(const CompatDataPoint& o)
{
  TRACE;
  CompatDataPoint::operator=(o);
  _value=_mean;
}

inline double CompatVDataPoint::misfitOrder2(int& nMeans, int& nValues, double invalid)
{
  TRACE;
  if(_mean!=invalid) {
    if(_value!=invalid) {
      double diff;
      if(_stddev)
        diff=(_mean - _value)/_stddev;
      else
        diff=(_mean - _value)/_mean;
      return diff*diff;
    } else {
      nValues--;
      return 0.0;
    }
  } else {
    nMeans--;
    nValues--;
    return 0.0;
  }
}

QGPCOMPATIBILITY_EXPORT QDataStream& operator<< (QDataStream& s, const CompatDataPoint& p);
QGPCOMPATIBILITY_EXPORT QDataStream& operator>> (QDataStream& s, CompatDataPoint& p);

class QGPCOMPATIBILITY_EXPORT CompatDataPointVector : public VectorList<CompatDataPoint>,
      public XMLClass
{
public:
  CompatDataPointVector() {}
  virtual ~CompatDataPointVector() {}
  VectorList<double> * meanVector() const;
  VectorList<double> * stddevVector() const;
  VectorList<double> * weightVector() const;
protected:
  virtual const QString& xml_tagName() const {return xmlDataPointVectorTag;}
  static const QString xmlDataPointVectorTag;
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
};

class QGPCOMPATIBILITY_EXPORT CompatVDataPointVector : public VectorList<CompatVDataPoint>,
      public XMLClass
{
public:
  CompatVDataPointVector() {}
  virtual ~CompatVDataPointVector() {}
  VectorList<double> * valueVector() const;
  VectorList<double> * meanVector() const;
  VectorList<double> * stddevVector() const;
  VectorList<double> * weightVector() const;
protected:
  virtual const QString& xml_tagName() const {return xmlVDataPointVectorTag;}
  static const QString xmlVDataPointVectorTag;
  virtual void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
  virtual XMLMember xml_member(XML_MEMBER_ARGS);
};

} // namespace QGpCompatibility

#endif // COMPATDATAPOINT_H
