/***************************************************************************
**
**  This file is part of GeopsyPySciFigs.
**
**  GeopsyPySciFigs is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyPySciFigs is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-12-18
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifdef HAS_PYTHON
#include "MethodArgument.h"
#include "GraphicalUserInterface.h"
#include "PythonClassFactory.h"

/*!
  \class MethodArgument MethodArgument.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

QMap<int, int> MethodArgument::_customTypes;

void MethodArgument::initTypes()
{
  ASSERT(_customTypes.isEmpty());
  int customTypeId=0;
  _customTypes.insert(QMetaType::fromType<AxisWindow *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<Axis *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<GraphContents *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<GraphContentsLayer *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<GraphicObject *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<TextEdit *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<LegendWidget *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<ColorMapWidget *>().id(), customTypeId++);
  _customTypes.insert(QMetaType::fromType<ImageWidget *>().id(), customTypeId++);
  /*QMap<int, int>::iterator it;
  for(it=_customTypes.begin(); it!=_customTypes.end(); it++) {
    QMetaType t(it.key());
    printf("Custom type %i --> %i (%s)\n", it.key(), it.value(), t.name());
  }*/
}

bool MethodArgument::isSupported(int type)
{
  switch(type) {
  case QMetaType::Bool:
  case QMetaType::Int:
  case QMetaType::Double:
  case QMetaType::QString:
  case QMetaType::Void:
    return true;
  default:
    return _customTypes.contains(type);
  }
}

bool MethodArgument::unsupportedTypeError() const
{
  QMetaType t(_type);
  GraphicalUserInterface::instance()->setErrorMessage(tr("unsupported type: %1 (%2)")
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
                                                          .arg(t.name()).arg(_type));
#else
                                                          .arg(t.name().data()).arg(_type));
#endif
  return false;
}

#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
bool MethodArgument::setArgument(PyObject * args, int index, const QMetaType& type)
#else
bool MethodArgument::setArgument(PyObject * args, int index, int typeId)
#endif
{
  if(args==NULL) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 is missing")
                                                            .arg(index));

    return false;
  }
  PyObject * arg=PyTuple_GET_ITEM(args, index);
  if(arg==NULL) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 is missing")
                                                            .arg(index));
    return false;
  }
  bool ok=true;
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
  _type=type.id();
#else
  _type=typeId;
#endif
  switch(_type) {
  case QMetaType::Bool:
    _booleanFlag=toBool(arg, ok, index);
    _argument=Q_ARG(bool, _booleanFlag);
    break;
  case QMetaType::Int:
    _integerValue=toInt(arg, ok, index);
    _argument=Q_ARG(int, _integerValue);
    break;
  case QMetaType::Double:
    _floatValue=toDouble(arg, ok, index);
    _argument=Q_ARG(double, _floatValue);
    break;
  case QMetaType::QString:
    _string=toString(arg, ok, index);
    _argument=Q_ARG(QString, _string);
    break;
  default:
    QMap<int, int>::const_iterator it;
    it=_customTypes.find(_type);
    if(it==_customTypes.end()) {
      return unsupportedTypeError();
    }
    _object.qObject=toObjectStar(arg);
    switch(it.value()) {
    case 0:
      if(_object.qObject->metaObject()->inherits(&AxisWindow::staticMetaObject)) {
        _argument=Q_ARG(AxisWindow *, _object.axisWindow);
      } else {
        ok=false;
      }
      break;
    case 1:
      if(_object.qObject->metaObject()->inherits(&Axis::staticMetaObject)) {
        _argument=Q_ARG(Axis *, _object.axis);
      } else {
        ok=false;
      }
      break;
    case 2:
      if(_object.qObject->metaObject()->inherits(&GraphContents::staticMetaObject)) {
        _argument=Q_ARG(GraphContents *, _object.graphContents);
      } else {
        ok=false;
      }
      break;
    case 3:
      if(_object.qObject->metaObject()->inherits(&GraphContentsLayer::staticMetaObject)) {
        _argument=Q_ARG(GraphContentsLayer *, _object.graphContentsLayer);
      } else {
        ok=false;
      }
      break;
    case 4:
      if(_object.qObject->metaObject()->inherits(&GraphicObject::staticMetaObject)) {
        _argument=Q_ARG(GraphicObject *, _object.graphicObject);
      } else {
        ok=false;
      }
      break;
    case 5:
      if(_object.qObject->metaObject()->inherits(&TextEdit::staticMetaObject)) {
        _argument=Q_ARG(TextEdit *, _object.textEdit);
      } else {
        ok=false;
      }
      break;
    case 6:
      if(_object.qObject->metaObject()->inherits(&LegendWidget::staticMetaObject)) {
        _argument=Q_ARG(LegendWidget *, _object.legendWidget);
      } else {
        ok=false;
      }
      break;
    case 7:
      if(_object.qObject->metaObject()->inherits(&ColorMapWidget::staticMetaObject)) {
        _argument=Q_ARG(ColorMapWidget *, _object.colorMapWidget);
      } else {
        ok=false;
      }
      break;
    case 8:
      if(_object.qObject->metaObject()->inherits(&ImageWidget::staticMetaObject)) {
        _argument=Q_ARG(ImageWidget *, _object.imageWidget);
      } else {
        ok=false;
      }
      break;
    default:
      ASSERT(false);
    }
    if(!ok) {
      GraphicalUserInterface::instance()->setErrorMessage(tr("object %1 of type %2 cannot be casted into type %3")
                                                              .arg(PythonClassFactory::objectId(arg))
                                                              .arg(_object.qObject->metaObject()->className())
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
                                                              .arg(type.name()));
#else
                                                              .arg(QMetaType::typeName(typeId)));
#endif
    }
  }
  return ok;
}

#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
bool MethodArgument::setReturnArgument(const QMetaType& type)
{
  _type=type.id();
#else
bool MethodArgument::setReturnArgument(int typeId)
{
  _type=typeId;
#endif
  switch(_type) {
  case QMetaType::Bool:
    _returnArgument=Q_RETURN_ARG(bool, _booleanFlag);
    break;
  case QMetaType::Int:
    _returnArgument=Q_RETURN_ARG(int, _integerValue);
    break;
  case QMetaType::Double:
    _returnArgument=Q_RETURN_ARG(double, _floatValue);
    break;
  case QMetaType::QString:
    _returnArgument=Q_RETURN_ARG(QString, _string);
    break;
  case QMetaType::Void:
    break;
  default:
    QMap<int, int>::const_iterator it;
    it=_customTypes.find(_type);
    if(it==_customTypes.end()) {
      return unsupportedTypeError();
    }
    switch(it.value()) {
    case 0:
      _returnArgument=Q_RETURN_ARG(AxisWindow *, _object.axisWindow);
      break;
    case 1:
      _returnArgument=Q_RETURN_ARG(Axis *, _object.axis);
      break;
    case 2:
      _returnArgument=Q_RETURN_ARG(GraphContents *, _object.graphContents);
      break;
    case 3:
      _returnArgument=Q_RETURN_ARG(GraphContentsLayer *, _object.graphContentsLayer);
      break;
    case 4:
      _returnArgument=Q_RETURN_ARG(GraphicObject *, _object.graphicObject);
      break;
    case 5:
      _returnArgument=Q_RETURN_ARG(TextEdit *, _object.textEdit);
      break;
    case 6:
      _returnArgument=Q_RETURN_ARG(LegendWidget *, _object.legendWidget);
      break;
    case 7:
      _returnArgument=Q_RETURN_ARG(ColorMapWidget *, _object.colorMapWidget);
      break;
    case 8:
      _returnArgument=Q_RETURN_ARG(ImageWidget *, _object.imageWidget);
      break;
    default:
      ASSERT(false);
    }
  }
  return true;
}

int MethodArgument::checkCount(PyObject * args, int expectedCount)
{
  int argc;
  if(args==NULL) {
    argc=0;
  } else if(PyTuple_Check(args)) {
    argc=PyTuple_Size(args);
  } else {
    argc=1;
  }
  if(argc!=expectedCount) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("number of arguments is not correct (%1 instead of %2)")
                                                            .arg(argc).arg(expectedCount));
    return -1;
  }
  return argc;
}

bool MethodArgument::toBool(PyObject * arg, bool& ok, int index)
{
  if(PyBool_Check(arg)) {
    int val=PyLong_AsLong(arg);
    return val!=0;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 must be an integer")
                                                            .arg(index));
  }
  ok=false;
  return true;
}

int MethodArgument::toInt(PyObject * arg, bool& ok, int index)
{
  if(PyLong_Check(arg)) {
    int val=PyLong_AsLong(arg);
    return val;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 must be an integer")
                                                            .arg(index));
  }
  ok=false;
  return 0;
}

double MethodArgument::toDouble(PyObject * arg, bool& ok, int index)
{
  if(PyFloat_Check(arg)) {
    double val=PyFloat_AsDouble(arg);
    return val;
  } else if(PyLong_Check(arg)) {
    int val=PyLong_AsLong(arg);
    return val;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 must be a float")
                                                            .arg(index));
  }
  ok=false;
  return 0.0;
}

QString MethodArgument::toString(PyObject * arg, bool& ok, int index)
{
  if(PyUnicode_Check(arg)) {
    QString val;
    int k=PyUnicode_KIND(arg);
    switch(k) {
    case PyUnicode_1BYTE_KIND:
      val=QString::fromUtf8(reinterpret_cast<char *>(PyUnicode_1BYTE_DATA(arg)));
      return val;
    case PyUnicode_2BYTE_KIND:
      val=QString::fromUtf16(reinterpret_cast<char16_t *>(PyUnicode_2BYTE_DATA(arg)));
      return val;
    case PyUnicode_4BYTE_KIND:
      GraphicalUserInterface::instance()->setErrorMessage(tr("4 byte unicode not supported for argument %1")
                                                              .arg(index));
      break;
    }
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 must be a string")
                                                            .arg(index));
  }
  ok=false;
  return QString();
}

QObject * MethodArgument::toObjectStar(PyObject * arg)
{
  return GraphicalUserInterface::instance()->object(PythonClassFactory::instance()->objectId(arg));
}

/*!
  If the returned value is an object, make sure it is registered.
  Called from the GUI thread.
*/
void MethodArgument::registerObject()
{
  if(_object.qObject) {
    _objectClassName=_object.qObject->metaObject()->className();
    _objectId=GraphicalUserInterface::instance()->objectId(_object.qObject);
    if(_objectId<0) { // Object not yet registered, add it
      _objectId=GraphicalUserInterface::instance()->addObject(_object.qObject);
    }
  }
}

PyObject * MethodArgument::toPyObject() const
{
  switch(_type) {
  case QMetaType::Bool:
    return _booleanFlag ? Py_True : Py_False;
  case QMetaType::Int:
    return PyLong_FromLong(_integerValue);
  case QMetaType::Double:
    return PyFloat_FromDouble(_floatValue);
  case QMetaType::QString:
    return PyUnicode_FromKindAndData(PyUnicode_2BYTE_KIND, _string.data(), _string.size());
  case QMetaType::Void:
    Py_RETURN_NONE;
  default:
    if(_object.qObject) {
      return PythonClassFactory::instance()->create(_objectClassName, _objectId);
    } else {
      Py_RETURN_NONE;
    }
  }
}

#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
QVariant MethodArgument::toVariant(PyObject * args, int index, const QMetaType& type)
#else
QVariant MethodArgument::toVariant(PyObject * args, int index, int typeId)
#endif
{
  QVariant val;
  if(args==NULL) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 is missing")
                                                            .arg(index));
    return val;
  }
  PyObject * arg=PyTuple_GET_ITEM(args, index);
  if(arg==NULL) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("argument %1 is missing")
                                                            .arg(index));
    return val;
  }
  bool ok=true;
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
  switch(type.id()) {
#else
  switch(typeId) {
#endif
  case QMetaType::Bool:
    val=toBool(arg, ok, index);
    break;
  case QMetaType::Int:
    val=toInt(arg, ok, index);
    break;
  case QMetaType::Double:
    val=toDouble(arg, ok, index);
    break;
  case QMetaType::QString:
    val=toString(arg, ok, index);
    break;
  default:
    GraphicalUserInterface::instance()->setErrorMessage(tr("unsupported type %1 for argument %i")
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
                                                            .arg(type.name()).arg(index));
#else
                                                            .arg(QMetaType::typeName(typeId)).arg(index));
#endif
    break;
  }
  if(!ok) {
    val=QVariant();
  }
  return val;
}

#endif // HAS_PYTHON
