/***************************************************************************
**
**  This file is part of GeopsyPySciFigs.
**
**  GeopsyPySciFigs is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyPySciFigs is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-11-17
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifdef HAS_PYTHON
#include "PythonNumPy.h"

#include <SciFigs.h>

#include "GraphicalUserInterface.h"
#include "ActionExecute.h"
#include "ActionRequest.h"

/*!
  \class GraphicalUserInterface GraphicalUserInterface.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

GraphicalUserInterface * GraphicalUserInterface::_self=nullptr;
PyObject * GraphicalUserInterface::_error=NULL;

/*!
  Description of constructor still missing
*/
GraphicalUserInterface::GraphicalUserInterface()
{
  _execute=nullptr;
  _request=nullptr;
  _engineRunning=false;
  _self=this;
  _threadId=0;
}

/*!
  Description of destructor still missing
*/
GraphicalUserInterface::~GraphicalUserInterface()
{
  delete _execute;
  delete _request;
}

bool GraphicalUserInterface::initPythonError(PyObject * module)
{
  _error=PyErr_NewException("GeopsyPySciFigs.error", NULL, NULL);
  Py_XINCREF(_error);
  if (PyModule_AddObject(module, "error", _error)<0) {
    Py_XDECREF(_error);
    Py_CLEAR(_error);
    Py_DECREF(module);
    return false;
  } else {
    return true;
  }
}

void GraphicalUserInterface::setErrorMessage(const QString& msg)
{
  PyErr_SetString(_error, msg.toLocal8Bit().data());
}

void GraphicalUserInterface::engineStarted()
{
  _engineRunning.fetchAndStoreOrdered(true);
}

void GraphicalUserInterface::engineStopped()
{
  printf("GeopsyPySciFigs graphical engine stopped.\n");
  _engineRunning.fetchAndStoreOrdered(false);
}

void GraphicalUserInterface::waitForStarted()
{
  while(_engineRunning.testAndSetOrdered(false,false)) {
    usleep(1000);
  }
}

void GraphicalUserInterface::waitForStopped()
{
  while(_engineRunning.testAndSetOrdered(true,true)) {
    usleep(1000);
  }
}

void * GraphicalUserInterface::run(void * arg)
{
  GraphicalUserInterface * t=static_cast<GraphicalUserInterface *>(arg);
  static int argc=1;
  static char argv1[]="GeopsyPySciFigs";
  static char * argv[]={argv1};
  Application::BugReportOptions options=Application::ReportBugs;
  options|=Application::ReportDirectly;
  options|=Application::NoUserInterruptReport;
  Application * app=new Application(argc, argv, nullptr, options);
  app->setQuitOnLastWindowClosed(false);
  SciFigsGlobal * sg=new SciFigsGlobal(false);
  t->execute();
  t->engineStarted();
  app->exec();
  delete sg;
  delete app;
  t->engineStopped();
  return nullptr;
}

ActionExecute * GraphicalUserInterface::execute()
{
  if(!_execute) {
    _execute=new ActionExecute;
  }
  return _execute;
}

bool GraphicalUserInterface::isQtApplicationAvailable()
{
  if(Application::instance()) {
    return true;
  } else {
    _self->setErrorMessage(tr("GeopsyPyScifigs graphical engine is off."));
    return false;
  }
}

bool GraphicalUserInterface::startEngine()
{
  if(Application::instance()) {
    if(Application::instance()->hasGui()) {
      printf("GeopsyPySciFigs graphical engine already started.\n");
    } else {
      setErrorMessage("Import module GeopsyPySciFigs before non graphical modules.");
      return false;
    }
  } else {
    if(_threadId==0u) {
#ifdef Q_OS_UNIX
      pthread_attr_t attr;
      pthread_attr_init(&attr);
      pthread_create(&_threadId, &attr, run, this);
#else
      fprintf(stderr, "Main thread must be created for this platform.\n");
#endif
      waitForStarted();
      _request=new ActionRequest;
      _request->connectTo(_execute);

      // Found that qInfo() was setting the Python interpreter stream as the main Qt stream.
      printf("GeopsyPySciFigs graphical engine started.\n");
    } else {
      printf("GeopsyPySciFigs graphical engine has been stopped. Restart Python interpreter.\n");
    }
  }
  return true;
}

void GraphicalUserInterface::stopEngine()
{
  if(isQtApplicationAvailable()) {
    _request->quit();
    waitForStopped();
  }
}

int GraphicalUserInterface::createGraphicSheet()
{
  if(isQtApplicationAvailable()) {
    return _request->createGraphicSheet();
  } else {
    return 0;
  }
}

int GraphicalUserInterface::createAxisWindow(int parentId)
{
  if(isQtApplicationAvailable()) {
    return _request->createAxisWindow(parentId);
  } else {
    return 0;
  }
}

void GraphicalUserInterface::addCurves(int parentId, CurveData * d)
{
  if(isQtApplicationAvailable()) {
    return _request->addCurves(parentId, d);
  }
}

QVariant GraphicalUserInterface::propertyValue(int objectId, int propertyId) const
{
  if(isQtApplicationAvailable()) {
    return _request->propertyValue(objectId, propertyId);
  } else {
    return QVariant();
  }
}

bool GraphicalUserInterface::setPropertyValue(int objectId, int propertyId, PyObject * args) const
{
  if(isQtApplicationAvailable()) {
    return _request->setPropertyValue(objectId, propertyId, args);
  } else {
    return false;
  }
}

void GraphicalUserInterface::setVerbosity(int level)
{
  if(isQtApplicationAvailable()) {
    _request->setVerbosity(level);
  }
}

MethodArgument * GraphicalUserInterface::method(int objectId, int methodId, PyObject * args) const
{
  if(isQtApplicationAvailable()) {
    return _request->method(objectId, methodId, args);
  } else {
    return nullptr;
  }
}

int GraphicalUserInterface::objectId(QObject * obj) const
{
  return _execute->find(obj);
}

QObject * GraphicalUserInterface::object(int objectId) const
{
  return _execute->find<QObject>(objectId);
}

int GraphicalUserInterface::addObject(QObject * obj)
{
  return _execute->addObject(obj);
}

#endif // HAS_PYTHON
