/***************************************************************************
**
**  This file is part of GeopsyPySciFigs.
**
**  GeopsyPySciFigs is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyPySciFigs is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-11-21
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifdef HAS_PYTHON
#include "PythonNumPy.h"
#include <SciFigs.h>

#include "ActionExecute.h"
#include "PythonClassFactory.h"
#include "GraphicalUserInterface.h"
#include "CurveData.h"
#include "MethodArgument.h"

/*!
  \class ActionExecute ActionExecute.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
ActionExecute::ActionExecute()
    : QObject(nullptr), _legend(10)
{
  TRACE;
  _acknowledge=nullptr;
  _nextWidgetId=1;
  _legend.generate(0, 20, ColorPalette::defaultCategorical);
}

/*!
  Description of destructor still missing
*/
ActionExecute::~ActionExecute()
{
  TRACE;
}

void ActionExecute::initAcknowledge(QWaitCondition * c)
{
  _acknowledge=c;
}

void ActionExecute::acknowledge()
{
  _acknowledge->wakeAll();
}

/*!
  Adds \a object and all its children. The return ID is from the parent object.
*/
int ActionExecute::addObject(QObject * object)
{
  _objectLists.lock();
  int id=_nextWidgetId++;
  _idsToObjects.insert(id, object);
  _objectsToIds.insert(object, id);
  _objectLists.unlock();
  connect(object, SIGNAL(destroyed(QObject *)), this, SLOT(objectDestroyed(QObject *)));
  QObjectList list=object->children();
  for(int i=0; i<list.size(); i++) {
    QObject * object=list.at(i);
    if(!object->objectName().isEmpty() &&
       !object->inherits("QAction")) {
      addObject(object);
    }
  }
  return id;
}

void ActionExecute::objectDestroyed(QObject * object)
{
  int id=find(object);
  if(id>=0) {
    _objectLists.lock();
    _idsToObjects.remove(id);
    _objectsToIds.remove(object);
    _objectLists.unlock();
  }
}

int ActionExecute::find(QObject * object) const
{
  _objectLists.lock();
  QMap<QObject *, int>::const_iterator it=_objectsToIds.find(object);
  if(it!=_objectsToIds.end()) {
    int id=it.value();
    _objectLists.unlock();
    return id;
  } else {
    _objectLists.unlock();
    return -1;
  }
}

void ActionExecute::createGraphicSheet(int * objectId)
{
  GraphicSheetMenu * w=new GraphicSheetMenu(nullptr, Qt::Window);
  w->resize(QSize(800, 600));
  w->show();
  *objectId=addObject(w->sheet());
  acknowledge();
}

void ActionExecute::createAxisWindow(int parentId, int * objectId)
{
  AxisWindow * w=new AxisWindow(nullptr, Qt::Window);
  if(parentId==-1) {
    w->resize(QSize(800, 600));
    w->show();
  } else {
    GraphicSheet * s=find<GraphicSheet>(parentId);
    if(s) {
      s->addObject(w, true);
    } else {
      GraphicalUserInterface::instance()->setErrorMessage(tr("parent is not available or is not of type GraphicSheet"));
      *objectId=-1;
      return acknowledge();
    }
  }
  *objectId=addObject(w);
  acknowledge();
}

void ActionExecute::addCurves(int parentId, const CurveData * d)
{
  AxisWindow * w=find<AxisWindow>(parentId);
  if(w) {
    QList<LineLayer *> layers=w->graphContents()->findLayers<LineLayer>(QString());
    if(layers.isEmpty()) {
      LineLayer * layer=new LineLayer(w);
      layer->setReferenceLine(new PlotLine);
      layers.append(layer);
    }
    d->addTo(layers.first(), &_legend);
    setLimits(w);
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("parent is not available or is not of type AxisWindow"));
  }
  acknowledge();
}

void ActionExecute::propertyValue(int objectId, int propertyId, QVariant * val)
{
  QObject * object=find<QObject>(objectId);
  if(object) {
    const char * className=object->metaObject()->className();
    int propertyOffset=PythonClassFactory::instance()->propertyOffset(className);
    if(propertyOffset>=0) {
      QMetaProperty p=object->metaObject()->property(propertyOffset+propertyId);
      *val=p.read(object);
    } else {
      *val=QVariant();
    }
  }
  acknowledge();
}

void ActionExecute::setPropertyValue(int objectId, int propertyId, PyObject * args, bool * ok)
{
  QObject * object=find<QObject>(objectId);
  if(!object) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown object %1").arg(objectId));
    *ok=false;
    return acknowledge();
  }
  *ok=true;
  const char * className=object->metaObject()->className();
  int propertyOffset=PythonClassFactory::instance()->propertyOffset(className);
  if(propertyOffset<0) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown class %1").arg(className));
    *ok=false;
    return acknowledge();
  }
  if(MethodArgument::checkCount(args, 1)<0) {
    *ok=false;
    return acknowledge();
  }
  QMetaProperty p=object->metaObject()->property(propertyOffset+propertyId);
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
  QVariant val=MethodArgument::toVariant(args, 0, p.metaType());
#else
  QVariant val=MethodArgument::toVariant(args, 0, p.type());
#endif
  if(val.isValid()) {
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
    object->metaObject()->invokeMethod(object, "setProperty", Qt::DirectConnection, val, p);
#else
    object->metaObject()->invokeMethod(object, "setProperty", Qt::DirectConnection,
                                       Q_ARG(QVariant, val), Q_ARG(QMetaProperty, p));
#endif
  } else {
    *ok=false;
  }
  acknowledge();
}

void ActionExecute::method(int objectId, int methodId, PyObject * args, MethodArgument ** returnValue)
{
  QObject * object=find<QObject>(objectId);
  if(!object) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown object %1").arg(objectId));
    *returnValue=nullptr;
    return acknowledge();
  }
  const QMetaObject * mObj=object->metaObject();
  int methodOffset=PythonClassFactory::instance()->methodOffset(mObj->className());
  if(methodOffset<0) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("unknown class %1").arg(mObj->className()));
    *returnValue=nullptr;
    return acknowledge();
  }
  QMetaMethod m=mObj->method(methodOffset+methodId);
  int argc=MethodArgument::checkCount(args, m.parameterCount());
  if(argc<0) {
    *returnValue=nullptr;
    return acknowledge();
  }
  if(argc>MethodArgument::maximumArgumentCount()) {
    GraphicalUserInterface::instance()->setErrorMessage(tr("number of arguments >%1 unsupported")
                                                            .arg(MethodArgument::maximumArgumentCount()));
    *returnValue=nullptr;
    return acknowledge();
  }
  QList<MethodArgument *> qtArgs;
  for(int i=0; i<argc; i++) {
    MethodArgument * qtArg=new MethodArgument;
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
    if(qtArg->setArgument(args, i, m.parameterMetaType(i))) {
#else
    if(qtArg->setArgument(args, i, m.parameterType(i))) {
#endif
      qtArgs.append(qtArg);
    } else {
      *returnValue=nullptr;
      qDeleteAll(qtArgs);
      return acknowledge();
    }
  }
  MethodArgument * qtReturnArg=new MethodArgument;
#if(QT_VERSION>=QT_VERSION_CHECK(6, 5, 0))
  if(!qtReturnArg->setReturnArgument(m.returnMetaType())) {
#else
  if(!qtReturnArg->setReturnArgument(m.returnType())) {
#endif
    *returnValue=nullptr;
    qDeleteAll(qtArgs);
    return acknowledge();
  }
  bool ok;
  if(qtReturnArg->hasReturnArgument()) {
    switch(qtArgs.count()) {
    case 0:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtReturnArg->returnArgument());
      break;
    case 1:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtReturnArg->returnArgument(),
                    qtArgs.at(0)->argument());
      break;
    case 2:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtReturnArg->returnArgument(),
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument());
      break;
    case 3:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtReturnArg->returnArgument(),
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument(),
                    qtArgs.at(2)->argument());
      break;
    case 4:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtReturnArg->returnArgument(),
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument(),
                    qtArgs.at(2)->argument(),
                    qtArgs.at(3)->argument());
      break;
    case 5:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtReturnArg->returnArgument(),
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument(),
                    qtArgs.at(2)->argument(),
                    qtArgs.at(3)->argument(),
                    qtArgs.at(4)->argument());
      break;
    default:
      ASSERT(false);
    }
  } else {
    switch(qtArgs.count()) {
    case 0:
      ok=m.invoke(object, Qt::DirectConnection);
      break;
    case 1:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtArgs.at(0)->argument());
      break;
    case 2:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument());
      break;
    case 3:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument(),
                    qtArgs.at(2)->argument());
      break;
    case 4:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument(),
                    qtArgs.at(2)->argument(),
                    qtArgs.at(3)->argument());
      break;
    case 5:
      ok=m.invoke(object, Qt::DirectConnection,
                    qtArgs.at(0)->argument(),
                    qtArgs.at(1)->argument(),
                    qtArgs.at(2)->argument(),
                    qtArgs.at(3)->argument(),
                    qtArgs.at(4)->argument());
      break;
    default:
      ASSERT(false);
    }
  }
  if(ok) {
    qtReturnArg->registerObject();
    *returnValue=qtReturnArg;
  } else {
    GraphicalUserInterface::instance()->setErrorMessage(tr("cannot invoke method"));
    delete qtReturnArg;
    *returnValue=nullptr;
  }
  qDeleteAll(qtArgs);
  acknowledge();
}

void ActionExecute::setLimits(AxisWindow * w)
{
  GraphContents * gc=w->graphContents();
  Rect r=gc->boundingRect();
  r.enlarge(0.05, gc->scaleX().sampling(), gc->scaleY().sampling());
  w->xAxis()->setRange(r.x1(), r.x2());
  w->yAxis()->setRange(r.y1(), r.y2());
}

void ActionExecute::quit()
{
  Application::instance()->quit();
  acknowledge();
}

void ActionExecute::setVerbosity(int level)
{
  if(level<0) {
    Application::instance()->freezeStream(true);
  } else {
    AbstractStream::setApplicationVerbosity(level);
  }
  acknowledge();
}

void ActionExecute::dumpIds() const
{
  QMap<int, QObject *>::const_iterator it;
  for(it=_idsToObjects.begin(); it!=_idsToObjects.end(); it++) {
    QObject * obj=it.value();
    printf("%3i %10p %15s %15s\n", it.key(), obj, obj->metaObject()->className(),
           obj->objectName().toLatin1().data());
  }
}

#endif // HAS_PYTHON
