/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-10-20
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiTools.h>

#include "AsciiSignalComponentsWidget.h"

namespace GeopsyGui {

class AsciiSignalComponentItem : public QAbstractTableModel
{
public:
  AsciiSignalComponentItem(QObject * parent) : QAbstractTableModel(parent) {_format=nullptr;}

  void setFormat(AsciiSignalFormat * f);

  virtual int rowCount(const QModelIndex &) const {return _format ? _format->componentCount() : 0;}
  virtual int columnCount(const QModelIndex &) const {return 2;}
  virtual QVariant data(const QModelIndex & index, int role=Qt::DisplayRole) const;
  virtual QVariant headerData(int section, Qt::Orientation orientation, int role=Qt::DisplayRole) const;
  virtual bool setData (const QModelIndex & index, const QVariant & value, int role=Qt::EditRole);
  virtual bool insertRows(int row, int count, const QModelIndex & parent=QModelIndex());
  virtual bool removeRows(int row, int count, const QModelIndex & parent=QModelIndex());
  virtual Qt::ItemFlags flags (const QModelIndex &) const {return Qt::ItemIsEnabled | Qt::ItemIsEditable | Qt::ItemIsSelectable;}
private:
  AsciiSignalFormat * _format;
};

void AsciiSignalComponentItem::setFormat(AsciiSignalFormat * f)
{
  beginResetModel();
  _format=f;
  endResetModel();
}

bool AsciiSignalComponentItem::insertRows(int row, int count, const QModelIndex & parent)
{
  beginInsertRows(parent, row, row+count-1);
  for(int i=0;i<count;i++) {
    _format->insertComponent(row, AsciiSignalFormatComponent());
  }
  endInsertRows();
  return true;
}

bool AsciiSignalComponentItem::removeRows(int row, int count, const QModelIndex & parent)
{
  beginRemoveRows(parent, row, row+count-1);
  for(int i=0;i<count;i++) {
    _format->removeComponent(row);
  }
  endRemoveRows();
  return true;
}

QVariant AsciiSignalComponentItem::data(const QModelIndex & index, int role) const
{
  switch(role) {
  case Qt::DisplayRole:
    if(index.column()==0) {
      return _format->component(index.row()).pattern();
    } else {
      return Signal::userName(_format->component(index.row()).component());
    }
  default:
    return QVariant();
  }
}

QVariant AsciiSignalComponentItem::headerData(int section, Qt::Orientation orientation, int role) const
{
  switch(role) {
  case Qt::DisplayRole:
    if(orientation==Qt::Vertical) {
      return section+1;
    } else {
      if(section==0) {
        return tr("Pattern");
      } else {
        return tr("Component");
      }
    }
  default:
    return QVariant();
  }
}

bool AsciiSignalComponentItem::setData (const QModelIndex & index, const QVariant & value, int role)
{
  switch(role) {
  case Qt::EditRole:
    if(index.column()==0) {
      _format->component(index.row()).setPattern(value.toString());
    } else {
      _format->component(index.row()).setComponent(Signal::userComponent(value.toString()));
    }
    return true;
  default:
    return false;
  }
}

class AsciiSignalComponentDelegate : public ItemDelegate
{
public:
  AsciiSignalComponentDelegate(QObject *parent=0) : ItemDelegate(parent) {}

  QWidget * createEditor(QWidget *parent, const QStyleOptionViewItem &option,
                         const QModelIndex &index) const;
};

QWidget * AsciiSignalComponentDelegate::createEditor(QWidget * parent, const QStyleOptionViewItem &,
                                                     const QModelIndex &index) const
{
  TRACE;
  if(index.column()==0) {
    QLineEdit * w=createLineEdit(parent);
    return w;
  } else {
    QComboBox * w=createComboBox(parent);
    w->addItem(tr("Vertical"));
    w->addItem(tr("North"));
    w->addItem(tr("East"));
    w->addItem(tr("Horizontal"));
    w->addItem(tr("All"));
    w->addItem(tr("Ignore"));
    w->addItem(tr("Time"));
    w->addItem(tr("Undefined"));
    return w;
  }
}

/*!
  \class AsciiSignalComponentsWidget AsciiSignalComponentsWidget.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
AsciiSignalComponentsWidget::AsciiSignalComponentsWidget(QWidget * parent, Qt::WindowFlags f)
    : QWidget(parent, f)
{
  TRACE;
  setupUi(this);
  componentTable->setModel(new AsciiSignalComponentItem(this));
  componentTable->setItemDelegate(new AsciiSignalComponentDelegate(this));
  componentTable->setSelectionMode(QAbstractItemView::SingleSelection);
  Settings::getRect(this, "ComponentTable");
}

/*!
  Description of destructor still missing
*/
AsciiSignalComponentsWidget::~AsciiSignalComponentsWidget()
{
  TRACE;
  Settings::setRect(this, "ComponentTable");
}

void AsciiSignalComponentsWidget::setFormat(AsciiSignalFormat * f)
{
  TRACE;
  static_cast<AsciiSignalComponentItem *>(componentTable->model())->setFormat(f);
}

void AsciiSignalComponentsWidget::on_addButton_clicked()
{
  TRACE;
  componentTable->model()->insertRow(componentTable->model()->rowCount());
}

void AsciiSignalComponentsWidget::on_removeButton_clicked()
{
  TRACE;
  QModelIndexList sel=componentTable->selectionModel()->selectedRows();
  if(!sel.isEmpty()) {
    componentTable->model()->removeRow(sel.first().row());
  }
}

} // namespace GeopsyGui
