/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-11-10
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>
#include "GeopsyPreferences.h"
#include "SignalDatabase.h"

namespace GeopsyCore {

GeopsyPreferences::GeopsyPreferences()
{
  TRACE;
  // Load preferences from registry
  load();
  save();
}

GeopsyPreferences::~GeopsyPreferences()
{
  TRACE;
}

/*!
  Cache all preferences and set default values if not found in registry
*/
void GeopsyPreferences::load()
{
  TRACE;
  QSettings& reg=CoreApplication::instance()->settings();

  // Loading preferences
  reg.beginGroup("DialogOptions");
  reg.beginGroup("Preferences");
  reg.beginGroup("preferenceTab");
  reg.beginGroup("qt_tabwidget_stackedwidget");
  reg.beginGroup("loadTab");

  reg.beginGroup("fileFormatGroup");
  _askForFormat=reg.value("formatAsk", false).toBool();
  QString caption=reg.value("fileFormat", _fileFormat.captionFilter()).toString();
  _fileFormat=SignalFileFormat::fromCaptionFilter(caption);
  reg.endGroup();

  reg.beginGroup("viewerGroup");
  _viewIfFewFiles=reg.value("winDoIt",true).toBool();
  _maxNofFiles=reg.value("winNofFiles",2).toInt();
  _showTable=reg.value("winTable",true).toBool();
  _showGraphic=reg.value("winGraphic",true).toBool();
  _oneWindowPerFile=reg.value("oneWindowPerFile",true).toBool();
  reg.endGroup();

  reg.beginGroup("signalNamesGroup");
  if(reg.value("stationNameRxxx",false).toBool())
    _stationName=StationNameRxxx;
  else
    _stationName=StationNameFile;
  reg.endGroup();
  reg.beginGroup("eventNamesGroup");
  if(reg.value("eventNameFile",false).toBool())
    _eventName=StationNameFile;
  else
    _eventName=EventNameEmpty;
  reg.endGroup();

  reg.endGroup(); // loadTab

  // Memory management
  reg.beginGroup("poolTab");
  // Get phyisical memory size and reserve 80% for cache
  double defaultSize=CoreApplication::getMemorySize()/(1024*1024)*0.8;
  if(defaultSize<=0) {
    defaultSize=512.0;
  }
  _cacheSize=reg.value("cacheSize", defaultSize).toInt();
  _cacheSwapDir=reg.value("cacheSwapDir",QDir::home().absoluteFilePath(".geopsyswap")).toString();
  reg.endGroup();

  // Viewers tab
  reg.beginGroup("viewersTab");
  reg.beginGroup("viewerAppearanceGroup");
  _graphicMakeUp=reg.value( "graphicMakeUpSwitch", false).toBool();
  _graphicMakeUpFile=reg.value( "graphicMakeUpEdit" ).toString();
  _mapMakeUp=reg.value( "mapMakeUpSwitch", false).toBool();
  _mapMakeUpFile=reg.value( "mapMakeUpEdit" ).toString();
  // Table fields in viewers tab
  _tableFields.getSettings();
}

void GeopsyPreferences::save()
{
  TRACE;
  QSettings& reg=CoreApplication::instance()->settings();
  reg.beginGroup("DialogOptions");
  reg.beginGroup("Preferences");
  reg.beginGroup("preferenceTab");
  reg.beginGroup("qt_tabwidget_stackedwidget");

  reg.beginGroup("loadTab");

  reg.beginGroup("fileFormatGroup");
  reg.setValue("formatAsk",askForFormat());
  reg.setValue("formatFixed",!askForFormat());
  reg.setValue("fileFormat", fileFormat().captionFilter());
  reg.endGroup();

  reg.beginGroup("viewerGroup");
  reg.setValue("winDoIt",viewIfFewFiles());
  reg.setValue("winNofFiles",maxNofFiles());
  reg.setValue("winTable",showTable());
  reg.setValue("winGraphic",showGraphic());
  reg.setValue("oneWindowPerFile",oneWindowPerFile());
  reg.endGroup();

  reg.beginGroup("signalNamesGroup");
  reg.setValue("stationNameFile",stationNameFile());
  reg.setValue("stationNameRxxx",stationNameRxxx());
  reg.endGroup();
  reg.beginGroup("eventNamesGroup");
  reg.setValue("eventNameFile",eventNameFile());
  reg.setValue("eventNameEmpty",eventNameEmpty());
  reg.endGroup();

  reg.endGroup();

  reg.beginGroup("poolTab");
  reg.setValue("cacheSize",cacheSize());
  reg.setValue("cacheSwapDir",cacheSwapDir());
  reg.endGroup();

  reg.beginGroup("viewersTab");
  reg.beginGroup("viewerAppearanceGroup");
  reg.setValue("graphicMakeUpSwitch",restoreGraphicMakeUp());
  reg.setValue("graphicMakeUpEdit",graphicMakeUpFile());
  reg.setValue("mapMakeUpSwitch",restoreMapMakeUp());
  reg.setValue("mapMakeUpEdit",mapMakeUpFile());
  saveNoAuto();
}

void GeopsyPreferences::restoreFileDefaults()
{
  TRACE;
  // Remove all preferences from settings
  QSettings& reg=CoreApplication::instance()->settings();
  reg.beginGroup("DialogOptions");
  reg.beginGroup("Preferences");
  reg.beginGroup("preferenceTab");
  reg.beginGroup("qt_tabwidget_stackedwidget");

  reg.beginGroup("loadTab");
  reg.remove("");

  // Load preference to assign the default values
  load();
  // Save back defaults to settings
  save();
}

void GeopsyPreferences::restorePoolDefaults()
{
  TRACE;
  // Remove all preferences from settings
  QSettings& reg=CoreApplication::instance()->settings();
  reg.beginGroup("DialogOptions");
  reg.beginGroup("Preferences");
  reg.beginGroup("preferenceTab");
  reg.beginGroup("qt_tabwidget_stackedwidget");

  reg.beginGroup("poolTab");
  reg.remove("");

  // Load preference to assign the default values
  load();
  // Save back defaults to settings
  save();
}

void GeopsyPreferences::saveNoAuto()
{
  TRACE;
  _tableFields.setSettings();
}

} // namespace GeopsyCore
