/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-05-15
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef FILTERPARAMETERS_H
#define FILTERPARAMETERS_H

#include <QGpCoreMath.h>

#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class GEOPSYCORE_EXPORT FilterParameters: public AbstractParameters
{
public:
  FilterParameters();
  FilterParameters(const FilterParameters& o);

  void operator=(const FilterParameters& o);
  AbstractParameters * clone() const {return new FilterParameters(*this);}

  enum Band {LowPass, HighPass, BandPass, BandReject};
  enum Method {Butterworth, FrequencyWindow};
  ENUM_AS_STRING_DECL(Band)
  ENUM_AS_STRING_DECL(Method)

  void setFrequencyRange(double min, double max=0.0);
  double minimumFrequency() const {return _fmin;}
  double maximumFrequency() const {return _fmax;}

  Band band() const {return _band;}
  void setBand(Band b);

  Method method() const {return _method;}
  void setMethod(Method m) {_method=m;}

  int order() const {return _order;}
  void setOrder(int o) {if(::abs(o)<16777216) _order=o;}

  double rippleFactor() const {return _rippleFactor;}
  void setRippleFactor(double rf) {_rippleFactor=rf;}

  bool causal() const {return _order>0;}
  void setCausal(bool s) {_order=(s ? ::abs(_order) : -::abs(_order));}

  const WindowFunctionParameters& frequencyWindow() const {return _window;}
  WindowFunctionParameters& frequencyWindow() {return _window;}

  double width() const {return _width;}
  void setWidth(double w) {_width=w;}

  QString toString(PARAMETERS_TOSTRING_ARGS_DECL) const;
  void setVersion(PARAMETERS_SETVERSION_ARGS);
  void collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS);
  int totalKeywordCount(PARAMETERS_TOTALKEYWORDCOUNT_ARGS) const;
protected:
  int keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const;
  bool setValue(PARAMETERS_SETVALUE_ARGS);
private:
  Band _band;
  Method _method;
  WindowFunctionParameters _window; // for FrequencyWindow
  int _order;                       // for Butterworth and Chebyshev
  double _rippleFactor;             // for Chebyshev
  double _width;
  double _fmin, _fmax;
};

} // namespace GeopsyCore

#endif // FILTERPARAMETERS_H
