/***************************************************************************
**
**  This file is part of DinverDCCore.
**
**  DinverDCCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-04-02
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Param3DGroundModel.h"
#include "ParamGroundModel.h"
#include "GroundParameter.h"

namespace DinverDCCore {

/*!
  \class Param3DGroundModel Param3DGroundModel.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

const QString Param3DGroundModel::xmlParam3DGroundModelTag="Param3DGroundModel";

/*!
  Description of constructor still missing
*/
Param3DGroundModel::Param3DGroundModel()
{
  TRACE;
}

/*!
  Description of constructor still missing
*/
Param3DGroundModel::Param3DGroundModel(const Param3DGroundModel& o)
  : XMLClass()
{
  TRACE;
  int n=o._models.count();
  for(int i=0; i<n; i++) {
    ParamGroundModel * model=new ParamGroundModel(*o._models.at(i));
    _models.append(model);
    _lookup.insert(model, i);
  }
  _customConditions=o._customConditions;
}

/*!
  Description of destructor still missing
*/
Param3DGroundModel::~Param3DGroundModel()
{
  TRACE;
  qDeleteAll(_models);
}

void Param3DGroundModel::addModel(const ParamGroundModel& gm)
{
  TRACE;
  ParamGroundModel * model=new ParamGroundModel(gm);
  _lookup.insert(model, _models.count());
  _models.append(model);
}

bool Param3DGroundModel::isValid() const
{
  TRACE;
  if(modelCount()!=3) {
    App::log(tr("Support only for 3 ground model interpolations\n") );
    return false;
  }
  // Check if there is gradient layers
  for(VectorList<ParamGroundModel *>::const_iterator it=_models.begin(); it!=_models.end(); it++) {
    if((*it)->hasGradient()) {
      App::log(tr("Gradients are currently not supported\n") );
      return false;
    }
  }
  return true;
}

bool Param3DGroundModel::toParameters(RealSpace& ps) const
{
  TRACE;
  int n=_models.count();
  for(int i=0; i<n; i++) {
    ps.setParameterNamePrefix(QString::number(i)+"_");
    if(!_models.at(i)->toParameters(ps)) {
      return false;
    }
  }
  // Add custom conditions between profiles
  if(!_customConditions.apply(&ps)) {
    return false;
  }

  return true;
}

void Param3DGroundModel::initFinalProfiles()
{
  TRACE;
  for(VectorList<ParamGroundModel *>::const_iterator it=_models.begin(); it!=_models.end(); it++) {
    (*it)->initFinalProfiles();
  }
}

void Param3DGroundModel::updateFinalProfiles(const GroundParameter * from)
{
  TRACE;
  if(from) {
    QHash<const ParamGroundModel *, int>::iterator it=_lookup.find(from->groundModel());
    ASSERT(it!=_lookup.end());
    _models[it.value()]->updateFinalProfiles(from);
  } else {
    for(VectorList<ParamGroundModel *>::iterator it=_models.begin(); it!=_models.end(); it++) {
      (*it)->updateFinalProfiles();
    }
  }
}

void Param3DGroundModel::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  for(VectorList<ParamGroundModel *>::const_iterator it=_models.begin(); it!=_models.end(); it++) {
    (*it)->xml_save(s, context);
  }
  _customConditions.xml_save(s, context);
}

XMLMember Param3DGroundModel::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(context)
  Q_UNUSED(attributes)
  if(tag=="ParamGroundModel" ) {
    addModel(ParamGroundModel());
    return XMLMember(_models.last());
  } else if(tag=="ParamSpaceScript") {
    return XMLMember(&_customConditions);
  }
  return XMLMember(XMLMember::Unknown);
}

void Param3DGroundModel::humanInfo() const
{
  TRACE;
  App::log(tr( "\n---------------------- Positions of reference profiles\n\n"));
  for(VectorList<ParamGroundModel *>::const_iterator it=_models.begin(); it!=_models.end(); it++) {
    App::log((*it)->position().toString()+"\n");
  }
}

} // namespace DinverDCCore
