/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-04-11
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef PDFCURVE_H
#define PDFCURVE_H

#include <QGpCoreMath.h>

#include "ModelSet.h"
#include "ActiveModels.h"
#include "DinverCoreDLLExport.h"

namespace DinverCore {

  class Parameter;
  class NewModel;

  class DINVERCORE_EXPORT PdfPoint
  {
  public:
    PdfPoint() {_x=0.0; _cell.setValue(-1);}
    PdfPoint(double x, const ActiveIndex& cell) {_x=x; _cell=cell;}

    double x() const {return _x;}
    const ActiveIndex& cell() const {return _cell;}

    double value() const {return _value;}
    void setValue(double v) {_value=v;}

    inline void setProbability(const ActiveModels& models, double degreesOfFreedom);
    inline void setAcceptable(const ActiveModels& models, double minimumMisfit);
    inline double misfit(const ActiveModels& models);
  private:
    double _x, _value;
    ActiveIndex _cell;
  };

  inline double PdfPoint::misfit(const ActiveModels& models)
  {
    TargetIndex t;
    return models.misfit(_cell, t);
  }

  inline void PdfPoint::setProbability(const ActiveModels& models, double degreesOfFreedom)
  {
    double m=misfit(models);
    _value=exp(-0.5*degreesOfFreedom*m*m);
  }

  inline void PdfPoint::setAcceptable(const ActiveModels& models, double minimumMisfit)
  {
    _value=misfit(models)<=minimumMisfit ? 1.0 : 0.0;
  }

  class DINVERCORE_EXPORT PdfCurve : private Curve<PdfPoint>
  {
  public:
    void append(double x, const ActiveIndex& cell) {Curve<PdfPoint>::append(PdfPoint(x, cell));}
    void prepend(double x, const ActiveIndex& cell) {Curve<PdfPoint>::prepend(PdfPoint(x, cell));}

    int count() const {return Curve<PdfPoint>::count();}
    const PdfPoint& at(int index) const {return Curve<PdfPoint>::constAt(index);}
    bool compare(const PdfCurve& o, int& index) const;

    void sort();
    void setProbability(const ActiveModels& models, double degreesoffreedom);
    void setAcceptable(const ActiveModels& models, double minimumMisfit);
    void setBorders();
    void setStrictBorders();
    void hits(NewModel * m);

    int generate(double randomValue, ActiveIndex& cellIndex);
    void debugPrint(const ActiveModels * models=nullptr) const;
    void printCumulativeDistribution(const Parameter& p);
  private:
    void setBorderShape(int i1, int i2);
    void setBorderShapeDensity(int i1, int i2);
    void setStrictBorderShapeDensity(int i1, int i2);
    void setBorderShapeCosine(int i1, int i2);

    double * cumulativeDistribution() const;
  };

} // namespace DinverCore

#endif // PDFCURVE_H
