/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-06
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "BestModels.h"

namespace DinverCore {

  /*!
    \class BestModels BestModels.h
    \brief A sorted list (by increasing misfit) of indexes of best models

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  BestModels::BestModels(ActiveModels * models)
      : IncreaseStorage(64)
  {
    TRACE;
    _models=models;
    _nrMax=50;
    _nr2=1;
    _indexes=(ActiveIndex *)allocateVector(sizeof(int));
  }

  BestModels::BestModels(const BestModels& o, ActiveModels * models)
      : IncreaseStorage(64)
  {
    TRACE;
    _models=models;
    _nrMax=o._nrMax;
    _nr2=1;
    _indexes=(ActiveIndex *)allocateVector(sizeof(int));
  }

  BestModels::BestModels(const BestModels& o)
      : IncreaseStorage(o)
  {
    TRACE;
    _models=o._models;
    _nrMax=o._nrMax;
    _nr2=o._nr2;
    _indexes=(ActiveIndex *)allocateVector(sizeof(int));
    memcpy(_indexes, o._indexes, size()*sizeof(int));
  }


  /*!
    Description of destructor still missing
  */
  BestModels::~BestModels()
  {
    TRACE;
    free(_indexes);
  }

  void BestModels::clear()
  {
    free(_indexes);
    IncreaseStorage::clear();
    _indexes=(ActiveIndex *)allocateVector(sizeof(int));
    _nrMax=50;
    _nr2=1;
  }

  void BestModels::reallocate()
  {
    _indexes=static_cast<ActiveIndex *>(reallocateVector((uchar *)_indexes, sizeof(ActiveIndex)));
  }

  /*!

  */
  void BestModels::setNr(int nr)
  {
    TRACE;
    if(nr!=_nrMax && nr>0) {
      if(nr>_nrMax) {
        _nrMax=nr;
        update();
      } else {
        // When reducing the number of best models, make sure that the nr first best models
        // do not have the same misfit. If so leave best model count as it is.
        BestIndex b, bnr;
        bnr.setValue(nr-1);
        if(size()<=nr || misfit(bnr, TargetIndex::first)>misfit(b, TargetIndex::first)) {
          _nrMax=nr;
          if(size()>_nrMax) {
            downSize(_nrMax);
          }
        }
      }
    }
  }

  /*!
    Returns an index vector of size equal to active model count. Indexes are set to
    -1 if they do not correspond to a best model.
  */
  BestIndex * BestModels::activeModelMap() const
  {
    TRACE;
    int activeCount=_models->count();
    NullBestIndex * map=new NullBestIndex[activeCount];
    BestIndex bEnd;
    bEnd.setValue(size());
    for(BestIndex b; b<bEnd; b++) {
      map[activeIndex(b).value()]=b;
    }
    return map;
  }

  /*!
    Returns an index vector of size equal to all model count. Indexes are set to
    -1 if they do not correspond to a best model.
  */
  BestIndex * BestModels::allModelMap() const
  {
    TRACE;
    int allCount=_models->allModels()->size();
    NullBestIndex * map=new NullBestIndex[allCount];
    BestIndex bEnd;
    bEnd.setValue(size());
    for(BestIndex b; b<bEnd; b++) {
      map[_models->at(activeIndex(b)).index().value()]=b;
    }
    return map;
  }

  /*!
    If possible add models to the list of best model until reaching nr==nrMax.
  */
  void BestModels::update()
  {
    TRACE;
    if(size()>=_nrMax) return;
    BestIndex * map=activeModelMap(); // a quick way to check if model is already in current list
    ActiveIndex aEnd;
    aEnd.setValue(_models->count());
    for(ActiveIndex a; a<aEnd; a++) {
      if(!map[a.value()].isValid() && !_models->at(a).isAtomic()) {
        add(a);
      }
    }
    delete map;
  }

  /*!
    Checks if model with \a index can be considered as a best model and if so
    add it to the current list.
  */
  void BestModels::add(const ActiveIndex& index)
  {
    BestIndex i;
    BestIndex end;
    end.setValue(size());
    if(_models->targetCount()==1) { // best models in the classical sense
      double modelMisfit=_models->misfit(index, TargetIndex::first);
      if(size()<_nrMax) {   // Not yet reached the _nrMax best models
        if(size()==0) {     // First best model
          add();
          _indexes[0]=index;
        } else {            // Already some models, maintain them sorted by increasing misfit
          i=indexOf(modelMisfit);
          BestIndex i1(i);
          i1--;
          if(exactIndexOf(index, modelMisfit, i1)==BestIndex::null) {
            add();
            if(i<end) {
              memmove((char *)(_indexes+i.value()+1),
                      (char *)(_indexes+i.value()), sizeof(ActiveIndex)*(end.value()-i.value()));
            }
            _indexes[i.value()]=index;
          }
        }
      } else {
        BestIndex last;
        last.setValue(size()-1);
        double lastMisfit=misfit(last,TargetIndex::first);
        if(modelMisfit<lastMisfit) {  // A new best model to insert
          i=indexOf(modelMisfit);
          BestIndex i1(i);
          i1--;
          if(exactIndexOf(index, modelMisfit, i1)==BestIndex::null) {
            memmove((char *) (_indexes+i.value()+1),
                    (char *) (_indexes+i.value()), sizeof(ActiveIndex)*(size()-i.value()));
            _indexes[i.value()]=index;
            // Less than 5% difference between the very best one and the other best
            // Enlarge the best model ensemble. TODO: eventually reduce nr if the condition is not fulfilled
            /*if(misfit(0)[0]>0.95*misfit(size()-1)[0]) {
              add();
            }*/
          }
        } else if(modelMisfit==lastMisfit) {  // All misfits are equal
          /* Due to clipped misfit (increase the effective nr)
             Do some sort of Monte-Carlo inside the "good" misfit region */
          add();
          printf("Equal misfit: size %i misfit %lf last %lf\n", count(), modelMisfit, lastMisfit);
          _indexes[end.value()]=index;
        }
      }
    } else { // best models contains the Pareto front
    }
  }

  /*!
    In the usual add(), a new good model pushed out of the list the worst model.
    Instead in this function, if the new model is a good model, it replaces the
    closest model.
  */
  void BestModels::addDistinct(const ActiveIndex& index)
  {
    if(size()<_nrMax) {
      add(index);
      return;
    }
    if(_models->targetCount()==1) { // best models in the classical sense
      double modelMisfit=_models->misfit(index, TargetIndex::first);
      BestIndex last;
      last.setValue(size()-1);
      double m=misfit(last,TargetIndex::first);
      if(modelMisfit<m) {  // A new best model to insert
        BestIndex end;
        end.setValue(size());
        double minD=_models->parameterCount()*100;
        BestIndex minIndex=BestIndex::null;
        for(BestIndex i; i<end; i++) {
          double d=_models->distance(index, activeIndex(i));
          if(d<minD) {
            minD=d;
            minIndex=i;
          }
        }
        if(minIndex.isValid()) {
          if(modelMisfit<misfit(minIndex, TargetIndex::first)) {
            _indexes[minIndex.value()]=index;
          }
        } else {
          // Found no similar model back to classical add
          add(index);
        }
      } else if(modelMisfit==m) {  // All misfits are equal
        add(index);
      }
    } else { // best models contains the Pareto front

    }
  }

  /*!
    This function is called whenever a model listed in best models is generating a great number of bad models,
    exceding giveUp ratio or when a cell becomes atomic. This model must be removed from the best models if
    still present.

    It uses the first misfit to quickly locate the \a modelIndex in the list of best models.
  */
  void BestModels::remove(const ActiveIndex& index)
  {
    TRACE;
    // Check if index model is still listed
    double m=_models->misfit(index, TargetIndex::first);
    BestIndex i=indexOf(m);
    i--;
    i=exactIndexOf(index, m, i);
    if(i.isValid()) {
      // Effectively reduce the size by removing the indexBest item
      // _nr is always greater than 0, and indexBest<_nr
      memmove((char *) (_indexes+i.value()),
              (char *) (_indexes+i.value()+1),
              sizeof(ActiveIndex)*(size()-(i.value()+1)));
      downSize(size()-1);
    }
  }

  /*!
    Always return the index of the model having the first misfit just above \a m
    Hence misfit[i-1]<=m<misfit[i].

    This function is always used with exactIindexOf to find models.
  */
  BestIndex BestModels::indexOf(double m) const
  {
    if(m<misfit(BestIndex::first, TargetIndex::first)) {
      return BestIndex::first;
    }
    BestIndex last;
    last.setValue(size()-1);
    if(m>=misfit(last, TargetIndex::first)) {
      last.setValue(size());
      return last;
    }
    BestIndex i, step2;
    i.setValue(_nr2);
    step2=i;
    step2.divideBy2();
    while(step2>BestIndex::first) {
      if(i>last) {
        i-=step2;
      } else if(m<misfit(i, TargetIndex::first)) {
        BestIndex i1(i);
        i1--;
        if(m>=misfit(i1, TargetIndex::first)) {
          break;
        }
        i-=step2;
      } else {
        i+=step2;
      }
      step2.divideBy2();
    }
    return i;
  }

  /*!
    Looks only for first misfit.

    This function is always used with indexOf to find models.
  */
  BestIndex BestModels::exactIndexOf(const ActiveIndex& index, double m, const BestIndex& approxIndex) const
  {
    if(!approxIndex.isValid()) {
      return BestIndex::null; // Misfit of modelIndex is less than all others
    }
    BestIndex end;
    end.setValue(size());
    // approxIndex can be in the middle of similar misfits, first search above
    for(BestIndex i(approxIndex); i<end && misfit(i, TargetIndex::first)==m; i++) {
      if(_indexes[i.value()]==index) {
        return i;
      }
    }
    // Then search below
    for(BestIndex i(approxIndex); i.isValid() && misfit(i, TargetIndex::first)==m; i--) {
      if(model(i)) {
        return i;
      }
    }
    return BestIndex::null;
  }

  void BestModels::print() const
  {
    TRACE;
    printf("#\n#                 Best models\n");
    for(int i=0; i<size();i++) {
      _models->print(_indexes[i]);
    }
  }

  /*!
    Get the _nrMax largest cells. Return randomly one of them.
    Random value must be between 0 and 1, exclusive.
  */
  BestIndex BestModels::randomIndex(double randomValue) const
  {

    BestIndex index;
    //index.setValue(qFloor(size()*randomValue));
    //return activeIndex(index);
    //index.setValue(426);
    //return activeIndex(index);
    if(size()<=_nrMax) {
      index.setValue(qFloor(size()*randomValue));
      return index;
    } else {
      // Navigator hit selection when size()>_nrMax (minimum misfit sampling)
      VectorList<QPair<double, BestIndex>> list(count());
      for(int i=size()-1; i>=0; i--) {
        index.setValue(i);
        ActiveModel& m=_models->at(_indexes[i]);
        list[i]=QPair<int, BestIndex>(m.navigatorHitCount(), index);
        //list[i]=QPair<double, BestIndex>(m.volume(), index);
        //list[i]=QPair<int, BestIndex>(m.limitCount(), index);
      }
      std::sort(list.begin(), list.end());
      // Select in the last nrMax indexes
      //int i=size()-1-qFloor(_nrMax*randomValue);
      int i=qFloor(_nrMax*randomValue);
      return list.at(i).second;
    }
  }

  double BestModels::averageVolume() const
  {
    double vol=0.0;
    for(int i=size()-1; i>=0; i--) {
      vol+=_models->at(_indexes[i]).volume();
    }
    return vol/count();
  }

} // namespace DinverCore
