/***************************************************************************
**
**  This file is part of ArrayGui.
**
**  ArrayGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-09-13
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>

#include "RingEditor.h"

namespace ArrayGui {

  RingEditorModel::RingEditorModel(RingEditor * parent)
    : QAbstractTableModel(parent)
  {
  }

  int RingEditorModel::rowCount(const QModelIndex &) const
  {
    TRACE;
    return _rings.count();
  }

  int RingEditorModel::columnCount(const QModelIndex &) const
  {
    TRACE;
    return 7;
  }

  QVariant RingEditorModel::data(const QModelIndex &index, int role) const
  {
    TRACE;
    const PlotSPACRing& r=_rings.at(index.row());
    switch (role) {
    case Qt::EditRole:
    case Qt::DisplayRole:
      switch(index.column()) {
      case 0: return r.minRadius();
      case 1: return r.maxRadius();
      case 2: return r.count();
      case 3: return r.averageRadius();
      case 4: return r.thickness();
      case 5: return r.maximumAngleStep();
      default:
        break;
      }
      break;
    default:
      break;
    }
    return QVariant();
  }

  QVariant RingEditorModel::headerData(int section, Qt::Orientation orientation,
                                       int role) const
  {
    if(role!=Qt::DisplayRole) return QVariant();
    if(orientation==Qt::Horizontal) {
      switch(section) {
      case 0: return tr("Min");
      case 1: return tr("Max");
      case 2: return tr("Pairs");
      case 3: return tr("Avg radius");
      case 4: return tr("Thickness");
      case 5: return tr("Angle step");
      case 6: return tr("Pen");
      default: return QVariant();
      }
    } else {
      return section+1;
    }
  }

  bool RingEditorModel::setData(const QModelIndex & index, const QVariant & value,
                                int role)
  {
    PlotSPACRing& r=_rings[index.row()];
    switch (role) {
    case Qt::EditRole:
      switch(index.column()) {
      case 0:
        r.setMinRadius(value.toDouble());
        r.setPairs(_pairs);
        emit(dataChanged(RingEditorModel::index(index.row(), 2),
                         RingEditorModel::index(index.row(), 5)));
        return true;
      case 1:
        r.setMaxRadius(value.toDouble());
        r.setPairs(_pairs);
        emit(dataChanged(RingEditorModel::index(index.row(), 2),
                         RingEditorModel::index(index.row(), 5)));
        return true;
      default:
        return true;
        break;
      }
      break;
    default:
      break;
    }
    return false;
  }

  Qt::ItemFlags RingEditorModel::flags (const QModelIndex& index) const
  {
    TRACE;
    if(index.column()<2) {
      return Qt::ItemIsEnabled | Qt::ItemIsEditable | Qt::ItemIsSelectable;
    } else if(index.column()==6) {
      return Qt::ItemIsEnabled;
    } else {
      return Qt::ItemIsEnabled | Qt::ItemIsSelectable;
    }
  }

  void RingEditorModel::clear()
  {
    TRACE;
    beginResetModel();
    _rings.clear();
    endResetModel();
  }

  void RingEditorModel::setArray(const QList<NamedPoint>& array)
  {
    TRACE;
    beginResetModel();
    int nStations=array.count();
    int nPairs=nStations*(nStations-1)/2;
    _pairs.resize(nPairs);
    int iPair=0;
    for(int i=nStations-1; i>=0; i--) {
      for(int j=i-1; j>=0; j--) {
        _pairs[iPair++].setStations(array.at(i), array.at(j));
      }
    }
    std::sort(_pairs.begin(), _pairs.end(), StationPair::lessThan);
    endResetModel();
  }

  void RingEditorModel::setArray(const ArraySelection& array)
  {
    TRACE;
    beginResetModel();
    int nStations=array.count();
    int nPairs=nStations*(nStations-1)/2;
    _pairs.resize(nPairs);
    int iPair=0;
    for(int i=nStations-1; i>=0; i--) {
      for(int j=i-1; j>=0; j--) {
        _pairs[iPair++].setStations(array, i, j);
      }
    }
    std::sort(_pairs.begin(), _pairs.end(), StationPair::lessThan);
    endResetModel();
  }

  void RingEditorModel::addRing(double minR, double maxR, const Pen& pen)
  {
    TRACE;
    beginInsertRows(QModelIndex(), _rings.count(), _rings.count());
    PlotSPACRing r;
    r.setRadii(minR, maxR);
    r.setPairs(_pairs);
    r.setPen(pen);
    _rings.append(r);
    endInsertRows();
  }

  void RingEditorModel::addRing(const Pen& pen)
  {
    TRACE;
    if(_rings.count()>0) {
      beginInsertRows(QModelIndex(), _rings.count(), _rings.count());
      PlotSPACRing r(_rings.last());
      r.setMinRadius(r.maxRadius());
      r.setPen(pen);
      _rings.append(r);
      endInsertRows();
    } else {
      addRing(0.0, 0.0, pen);
    }
  }

  void RingEditorModel::setRing(int index, double minR, double maxR)
  {
    TRACE;
    _rings[index].setRadii(minR, maxR);
    emit(dataChanged(RingEditorModel::index(index, 0),
                     RingEditorModel::index(index, 1)));
  }

  void RingEditorModel::autoRings()
  {
    TRACE;
    VectorList<SPACRing> r=SPACRing::autoRings(_pairs);
    int n=r.count();
    beginResetModel();
    _rings.clear();
    for(int i=0; i<n; i++) {
      _rings.append(r.at(i));
    }
    endResetModel();
  }

  void RingEditorModel::removeRings(QModelIndexList l)
  {
    TRACE;
    QList<int> rows=sortedRows(l);
    for(int i=rows.count()-1; i>=0; i--) {
      int row=rows.at(i);
      beginRemoveRows(QModelIndex(), row, row);
      _rings.removeAt(row);
      endRemoveRows();
    }
  }

  void RingEditorModel::read(QTextStream& s)
  {
    TRACE;
    beginResetModel();
    PlotSPACRing r;
    while(!s.atEnd()) {
      if(r.read(s)) {
        _rings.append(r);
      }
    }
    endResetModel();
  }

  void RingEditorModel::write(QTextStream& s) const
  {
    TRACE;
    int n=_rings.count();
    PlotSPACRing::writeHeader(s);
    for(int i=0; i<n; i++) {
      _rings.at(i).write(s);
    }
  }

  Legend RingEditorModel::legend() const
  {
    int n=_rings.count();
    Legend l(n);
    for(int i=0; i<n; i++) {
      const PlotSPACRing& r=_rings.at(i);
      l.setPen(i, r.pen());
    }
    return l;
  }

  void RingEditorModel::setLegend(const Legend& l)
  {
    int n=_rings.count();
    for(int i=0; i<n; i++) {
      _rings[i].setPen(l.pen(i));
    }
  }

  RingEditorDelegate::RingEditorDelegate(QObject *parent)
      : ItemDelegate(parent)
  {
    TRACE;
  }

  QWidget * RingEditorDelegate::createEditor(QWidget *parent, const QStyleOptionViewItem & option,
                                             const QModelIndex &index) const
  {
    TRACE;
    if(index.column()<2) {
      DoubleSpinBox * w=createDoubleSpinBox(parent);
      connect(w, SIGNAL(valueChanged(double)), this, SLOT(commitWidget()));
      w->setMinimum(0.0);
      w->setMaximum(std::numeric_limits<double>::infinity());
      return w;
    } else {
      return ItemDelegate::createEditor(parent, option, index);
    }
  }

  void RingEditorDelegate::paint(QPainter * painter, const QStyleOptionViewItem & option,
                                 const QModelIndex & index) const
  {
    TRACE;
    if(index.column()==6) {
      const RingEditorModel* m=static_cast<const RingEditorModel *>(index.model());
      LegendItem item(QString(), m->rings().at(index.row()).pen(), Symbol());
      LegendItemDisplay itemDisplay(item);
      drawBackground(painter, option, index);
      itemDisplay.setDisplayText(false);
      itemDisplay.paint(*painter, SciFigsGlobal::screenResolution(), option.rect.x(), option.rect.y(),
                        option.rect.width(), option.rect.height());
    } else {
      ItemDelegate::paint(painter, option, index);
    }
  }

  QSize RingEditorDelegate::sizeHint(const QStyleOptionViewItem & option,
                                     const QModelIndex & index) const
  {
    TRACE;
    if(index.column()==6) {
      const RingEditorModel* m=static_cast<const RingEditorModel *>(index.model());
      LegendItem item(QString(), m->rings().at(index.row()).pen(), Symbol());
      LegendItemDisplay itemDisplay(item);
      itemDisplay.setDisplayText(false);
      return itemDisplay.sizeHint(option.font);
    } else {
      return ItemDelegate::sizeHint(option, index);
    }
  }

  /*
   *  Constructs a RingEditor as a child of 'parent', with the
   *  name 'name' and widget flags set to 'f'.
   */
  RingEditor::RingEditor(QWidget *parent, Qt::WindowFlags f)
      : QWidget(parent, f), _legend(5)
  {
    TRACE;
    setupUi(this);

    ringsTable->setModel(new RingEditorModel(this));
    ringsTable->setItemDelegate (new RingEditorDelegate(this));
    connect(ringsTable->itemDelegate(), SIGNAL(commitData(QWidget*)), this, SLOT(updateGraph()));
    connect(ringsTable->selectionModel(), SIGNAL(currentChanged(const QModelIndex &, const QModelIndex &)),
            this, SLOT(currentChanged()));

    for(int i=0; i<6; i++) {
      ringsTable->resizeColumnToContents(i);
    }
    _circleLayer=nullptr;
    _currentPairs=nullptr;
    _legendEditor=nullptr;

    ColorPalette pal;
    pal.generate(0, 4, "roma");
    QColor c;
    for(int i=0; i<5; i++) {
      guiColor(pal.color(i), c);
      _legend.setPen(i, Pen(c, 0.6, Pen::SolidLine));
    }
  }

  /*
   *  Destroys the object and frees any allocated resources
   */
  RingEditor::~RingEditor()
  {
    TRACE;
    // no need to delete child widgets, Qt does it all for us
  }

  RingEditorModel * RingEditor::model() const
  {
    return static_cast<RingEditorModel *>(ringsTable->model());
  }

  void RingEditor::clear()
  {
    TRACE;
    model()->clear();
    updateGraph();
  }

  void RingEditor::setArray(const ArraySelection& array)
  {
    model()->setArray(array);
  }

  void RingEditor::setArray(const QList<NamedPoint>& array)
  {
    model()->setArray(array);
  }

  void RingEditor::setCoArrayGraph(ArrayMap * w)
  {
    TRACE;
    _circleLayer=new CircleViewer(w);
    _circleLayer->setObjectName("rings");
    _currentPairs=new LineLayer(w);
    _currentPairs->setObjectName("current pairs");
    _currentPairs->setErrorBar(LineLayer::NoBar);
    PlotLine2D * line=new PlotLine2D;
    line->setPen(Pen::NoLine);
    line->setSymbol(Symbol(Symbol::Circle, 2.0, Pen::NoLine,
                    Brush(Qt::red, Qt::SolidPattern)));
    _currentPairs->setReferenceLine(line);
    _currentPairs->addLine();
  }

  void RingEditor::addRing(double minR, double maxR)
  {
    model()->addRing(floor(minR*100.0)*0.01, ceil(maxR*100.0)*0.01,
                     _legend.pen(model()->rings().count()));
    updateGraph();
  }

  void RingEditor::on_addBut_clicked()
  {
    TRACE;
    model()->addRing(_legend.pen(model()->rings().count()));
    updateGraph();
  }

  void RingEditor::on_removeBut_clicked()
  {
    TRACE;
    QItemSelectionModel * sel=ringsTable->selectionModel();
    QModelIndexList l=sel->selectedRows();
    model()->removeRings(l);
    sel->select(ringsTable->currentIndex(), QItemSelectionModel::Select | QItemSelectionModel::Rows);
    updateGraph();
  }

  void RingEditor::updateGraph()
  {
    TRACE;
    if(!_circleLayer) {
      return;
    }
    const VectorList<PlotSPACRing>& rings=model()->rings();
    _circleLayer->resize(2*rings.count());
    for(int i=0; i<rings.count(); i++) {
      const PlotSPACRing& r=rings.at(i);
      _circleLayer->set(2*i, 0, 0, r.minRadius(), r.minRadius(), 0.0, r.pen());
      _circleLayer->set(2*i+1, 0, 0, r.maxRadius(), r.maxRadius(), 0.0, r.pen());
    }
    _circleLayer->deepUpdate();
    currentChanged();
  }

  void RingEditor::on_loadBut_clicked()
  {
    TRACE;
    QString str=Message::getOpenFileName(tr("Open rings file"), tr("Rings file (*.rings)"));
    if(str.length()>0) load(str);
  }

  void RingEditor::load(QString str)
  {
    TRACE;
    QFile f(str);
    f.open(QIODevice::ReadOnly);
    QTextStream s(&f);
    model()->read(s);
    updateGraph();
  }

  void RingEditor::on_saveBut_clicked()
  {
    TRACE;
    QString str=Message::getSaveFileName(tr("Save rings file"), tr("Rings file (*.rings)"));
    if(!str.isEmpty()) {
      QFile f(str);
      f.open(QIODevice::WriteOnly);
      QTextStream s(&f);
      model()->write(s);
    }
  }

  void RingEditor::on_optimizeBut_clicked()
  {
    TRACE;
    QItemSelectionModel * sel=ringsTable->selectionModel();
    QModelIndexList l=sel->selectedRows();
    for(int i=l.count()-1; i>=0; i--) {
      int index=l.at(i).row();
      const PlotSPACRing& r=model()->rings().at(index);
      double minR=r.minRadius();
      double maxR=r.maxRadius();
      static_cast<ArrayMap *>(_circleLayer->graph())->optimizeRing(minR, maxR);
      minR=floor(minR*100.0)*0.01;
      maxR=ceil(maxR*100.0)*0.01;
      model()->setRing(index, minR, maxR);
    }
    updateGraph();
  }

  void RingEditor::on_autoBut_clicked()
  {
    TRACE;
    RingEditorModel * m=model();
    m->autoRings();
    model()->setLegend(_legend);
    updateGraph();
  }

  void RingEditor::on_legendBut_clicked()
  {
    TRACE;
    Legend oldLegend=_legend;
    Dialog * d=new Dialog(this);
    ASSERT(!_legendEditor);
    _legendEditor=new LegendProperties(this);
    _legendEditor->setPropertySections(LegendTable::PenSection);
    d->setMainWidget(_legendEditor, Dialog::OkCancel);
    _legendEditor->setLegend(_legend);
    connect(_legendEditor, SIGNAL(touched()), this, SLOT(commitLegend()));
    Settings::getRect(d, "RingEditor::legend");
    if(d->exec()!=QDialog::Accepted) {
      _legend=oldLegend;
      model()->setLegend(oldLegend);
      updateGraph();
    } else {
      Settings::setRect(d, "RingEditor::legend");
    }
    delete d;
    _legendEditor=nullptr;
    updateGraph();
  }

  void RingEditor::commitLegend()
  {
    TRACE;
    if(_legendEditor) {
      _legend=_legendEditor->legend();
      model()->setLegend(_legend);
      updateGraph();
    }
  }

  void RingEditor::currentChanged()
  {
    TRACE;
    LayerLocker ll(_currentPairs);
    PlotLine2D * l=static_cast<PlotLine2D *>(_currentPairs->line(0));
    Curve<Point2D>& c=l->curve();
    QItemSelectionModel * sel=ringsTable->selectionModel();
    int index=sel->currentIndex().row();
    if(index>=0 && index<model()->rings().count()) {
      const PlotSPACRing& r=model()->rings().at(index);
      const QList<const StationPair *> p=r.pairs();
      c.resize(p.count());
      for(int i=c.count()-1; i>=0; i--) {
        c.at(i)=p.at(i)->coArray();
      }
    } else {
      c.clear();
    }
    _currentPairs->deepUpdate();
  }

} // namespace ArrayGui
