/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-09-08
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef WAVEFIELDVALUES_H
#define WAVEFIELDVALUES_H

#include "ArrayCoreDLLExport.h"
#include "FKPower.h"
#include "FKSteering.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT WavefieldValues
  {
  public:
    WavefieldValues();
    ~WavefieldValues();

    class Observations
    {
    public:
      double xiH, powerH, kConcavityH, xiConcavityH;
      double xiZ, powerZ, kConcavityZ, xiConcavityZ;
      double xiX, powerX, kConcavityX, xiConcavityX;
      double invObsPowerX, invObsKConcavityX, invObsXiConcavityX;
      double invObsPowerH, invObsKConcavityH, invObsXiConcavityH;
      double invObsPowerZ, invObsKConcavityZ, invObsXiConcavityZ;

      void init();
    };

    void setSteering(const FKSteering * s);
    void setCrossSpectrum(const ComplexMatrix * f);
    void setCrossSpectrumDerivative(const ComplexMatrix& df);

    void setWaveNumber(const Complex& k);

    void setSteering();
    void setValues(const Observations& obs);
    double misfit(const Observations& obs);
    QString detailedMisfit(const Observations& obs);
    const FKSteering * steering() const {return _steering;}

    void setDerivativeValues();
    double misfitDerivative(const Observations& obs);
  private:
    PrivateVector<double> _k;
    const FKSteering * _steering;

    class SteeringMode
    {
    public:
      SteeringMode();
      ~SteeringMode();

      void setSteering(const FKSteering * s);
      void setCrossSpectrum(const ComplexMatrix * f);
      void setDerivativeCrossSpectrum(const ComplexMatrix * dfInv);

      FKPower * powerEngine() {return _powerEngine;}

      void setPowerX(const Complex& ell);
      void setPowerH(const Complex& ell);
      void setPowerZ(const Complex& ell);

      void setXiSlopeX(const Complex& ell);
      void setXiSlopeH(const Complex& ell);
      void setXiSlopeZ(const Complex& ell);

      void setXiConcavityX();
      void setXiConcavityH(const Complex& ell);
      void setXiConcavityZ(const Complex& ell);

      void setKConcavity();

      double misfit(const double& obsPower, const double& invObsPower,
                    const double& obsKConcavity, const double& invObsKConcavity);
      QString detailedMisfit(char type) const;

      void setPowerDerivative();
      void setKSlopeDerivative();
      void setKConcavityDerivative();
      void setXiSlopeDerivative();

      double misfitDerivative(const double& invObsPower,
                              const double& invObsKConcavity);
    private:
      FKPower * _powerEngine;
      FKPower * _dpowerEngine;
      double _xiSlope, _xiConcavity, _kConcavity;
      double _diffPower, _diffKConcavity, _diffXiConcavity;
      double _derPower, _derKSlope, _derKConcavity, _derXiSlope, _derXiConcavity;
    };
    SteeringMode _x, _h, _z;
    ComplexMatrix _dfInv;
  };

} // namespace ArrayCore

#endif // WAVEFIELDVALUES_H

