/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-10-25
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SensorOrientationWorker.h"
#include "WaveNumberConverter.h"
#include "SensorOrientationCapon.h"
#include "SensorOrientationTaskManager.h"
#include "CrossSpectrumSplit.h"
#include "SensorOrientationFunction.h"

namespace ArrayCore {

  /*!
    \class SensorOrientationWorker SensorOrientationWorker.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SensorOrientationWorker::SensorOrientationWorker(const ArraySelection * array,
                                                   const FKParameters * param)
    : FKWorker(array, param),
      _optim(array->count()+1),
      _max(array->count()+1, 0.0)
  {
    TRACE;
    int n=array->count();
    // Precision
    PrivateVector<double> p(n+1, 0.0);
    p[0]=crossSpectrum()->parameters()->ellipticityAbsolutePrecision();
    for(int i=n; i>0; i--) {
      p[i]=M_PI/1800.0;
    }
    _optim.setPrecision(p);
    // Range
    _max.at(0)=std::numeric_limits<double>::infinity(); // Ellipticity is never limited (180 deg periodicity)
    for(int i=0; i<n; i++) {
      _max.at(1+i)=crossSpectrum()->parameters()->sensorOrientationMaximumCorrection();
    }
  }

  /*!
    Description of destructor still missing
  */
  SensorOrientationWorker::~SensorOrientationWorker()
  {
    TRACE;
  }

  void SensorOrientationWorker::setOrientationFunction(FKCache * cache)
  {
    SensorOrientationFunction * f=new SensorOrientationFunction(cache);
    f->setInnerStations(taskManager()->selectedStations());
    _optim.setFunction(f);
  }

  void SensorOrientationWorker::process(const VectorList<int>& blocks)
  {
    WaveNumberConverter conv(_crossSpectrum->timeWindows()->frequency().center());
    const TimeWindowList& winList=_crossSpectrum->timeWindows()->list();
    conv.setTime(winList.at(blocks.first()).start());
    APP_LOG(3, tr("Process at time %1 and at %2 Hz\n")
            .arg(conv.time().toString(DateTime::defaultFormat))
            .arg(conv.frequency()))
    if(!setSlownessRange()) {
      return;
    }
    SensorOrientationCapon * function=static_cast<SensorOrientationCapon *>(_search->function());

    function->setTotalLimits(crossSpectrum()->timeWindows()->frequency().squaredOmega());
    function->setAzimuthRange(crossSpectrum()->parameters()->minimumAzimuth(),
                              crossSpectrum()->parameters()->maximumAzimuth());
    if(crossSpectrum()->calculate(blocks, function)) {
      localizeMax();
      if(_search->maximaCount()>0) {
        // We simply split the cross-spectrum into fZZ, fEE, fEN, fNE, fNN, fEZ, fZE
        CrossSpectrumSplit * split=new CrossSpectrumSplit;
        split->setMatrix(*crossSpectrum()->matrix());
#ifdef ONE_INVERSION_PER_FREQUENCY
        taskManager()->addObservation(crossSpectrum()->timeWindows()->frequencyIndex(), split);
#endif
        // Add peaks found on the vertical component
        FunctionMaximaIterator it;
        const FKParameters * param=static_cast<const FKParameters *>(taskManager()->parameters());
        int i=0;
        for(it=_search->begin(); it!=_search->end(); ++it) {
          const FunctionSearchMaximum& m=**it;
          const Vector<double>& k=m.position();
          double slowness=conv.slowness(k);
          if(slowness>param->minimumSlowness() &&
             !function->isOnLimit(k, conv.frequency())) {
            FKPeaks::Value * val=_peaks.add();
            if(val) {
              split->addWaveNumber(k);
              i++;
#ifdef ONE_INVERSION_PER_FREQUENCY
              // Save peaks found on the vertical component for user information
              if(!_peaks.add(k, 0.0, 0.0, m.value(), conv, param->minimumSlowness())) {
                return;
              }
#else
              APP_LOG(3, tr("Optimizing sensor orientations at %1 and %2 Hz with %5 (%3/%4)...\n")
                       .arg(conv.time().toString())
                       .arg(conv.frequency())
                       .arg(i)
                       .arg(_search->maximaCount())
                       .arg(objectName()));
              if(invert(split)) {
                double xi=_optim.position()[0];
                while(xi>M_PI/2.0) {
                  xi-=M_PI;
                }
                while(xi<-M_PI/2.0) {
                  xi+=M_PI;
                }
                val->time=_peaks.time(conv.time());
                val->frequency=conv.frequency();
                val->slowness=slowness;
                val->azimuth=conv.azimuth(k);
                val->valid=true;
                val->extraValues=nullptr;
                val->ellipticity=::tan(xi);
                val->verticalNoise=std::numeric_limits<double>::quiet_NaN();
                val->horizontalNoise=std::numeric_limits<double>::quiet_NaN();
                val->power=_optim.value();
                val->setExtraValues(_optim.position(), 1, _optim.position().count()-1, 180.0/M_PI);
              } else {
                _peaks.removeLast();
              }
              split->clearWavenumbers();
              APP_LOG(3, tr("Finished sensor orientations at %1 and %2 Hz with %5 (%3/%4)...\n")
                       .arg(conv.time().toString())
                       .arg(conv.frequency())
                       .arg(i)
                       .arg(_search->maximaCount())
                       .arg(objectName()));
#endif
            }
          }
        }
#ifndef ONE_INVERSION_PER_FREQUENCY
        delete split;
#endif
      }
    }
  }

  bool SensorOrientationWorker::invert(CrossSpectrumSplit * split)
  {
    SensorOrientationFunction * f=static_cast<SensorOrientationFunction *>(_optim.function());
    VectorList<CrossSpectrumSplit *> splits;
    splits << split;
    f->setCrossSpectra(&splits);
    PrivateVector<double> p0(_optim.position().count());
    p0.setValue(0.0);
    _optim.reset(p0);
    return _optim.minimize(_max, crossSpectrum()->parameters()->sensorOrientationMaximumIterations());
  }

  void SensorOrientationWorker::polish(ParallelTask * t)
  {
    Q_UNUSED(t);
    TODO_WARNING;
#ifdef ONE_INVERSION_PER_FREQUENCY
    //static QMutex debugMutex;
    //debugMutex.lock();
    ArrayTask * at=static_cast<ArrayTask *>(t);
    const ArrayTimeWindows& tw=*at->timeWindows();
    double freq=tw.frequency().center();
    //if(fabs(freq-2.96381)>0.001) {
    //  return;
    //}
    SensorOrientationFunction * f=static_cast<SensorOrientationFunction *>(_optim.function());
    f->setCrossSpectra(taskManager()->observations(tw.frequencyIndex()));
    _optim.reset();
    // Scan only the inner station pairs
    /*int stations[]={4, 5, 7, 9, 10};
    for(int i=0; i<5; i++) {
      for(int j=i+1; j<5; j++) {
        scan2D(k, M_PI/36.0, stations[i], stations[j], "/tmp/scan");
      }
    }
    std::exit(2);*/
    // Full scan of all pairs of stations
    /*for(size_t i=0; i<=n; i++) {
      for(size_t j=i+1; j<=n; j++) {
        f->scan2D(M_PI/36.0, i, j, "/tmp/scan");
      }
    }
    std::exit(2);*/
    if(_optim.maximize(_max, crossSpectrum()->parameters()->maximumSensorOrientationIterations())) {
      taskManager()->addResult(freq, _optim.position());
    } else {
      PrivateVector<double> t(_optim.position().count(), 0.0);
      taskManager()->addResult(freq, t);
    }
    //debugMutex.unlock();
#endif
  }

} // namespace ArrayCore

