/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-04-13
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <DinverDCCore.h>

#include "SPACResults.h"
#include "SPACPeaks.h"

namespace ArrayCore {

  /*!
    \class SPACResults SPACResults.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SPACResults::SPACResults()
    : AbstractArrayResults()
  {
    TRACE;
    _mspacTarget=nullptr;
    _mspacStats=nullptr;
    _esacStats=nullptr;
    for(int i=0; i<3; i++) {
      _rejected[i]=nullptr;
    }

    _nFrequencies=0;
    _nRings=0;
    _nComponents=0;

    _peaks=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  SPACResults::~SPACResults()
  {
    TRACE;
    delete [] _mspacStats;
    delete [] _esacStats;
    for(int i=0; i<3; i++) {
      delete [] _rejected[i];
    }
    delete _mspacTarget;
    delete _peaks;
  }

  void SPACResults::setArray(const ArraySelection * array)
  {
    AbstractArrayResults::setArray(array);
    switch (_array->array()->components()) {
    case StationSignals::AllComponent:
      _nComponents=3;
      break;
    case StationSignals::VerticalComponent:
      _nComponents=1;
      break;
    }
  }

  /*!
    Description of destructor still missing
  */
  void SPACResults::clear()
  {
    TRACE;
  }

  void SPACResults::setParameters(const SPACParameters& param)
  {
    ASSERT(_nComponents>0);
    _nRings=param.ringCount();
    _nFrequencies=param.frequencySampling().count();

    delete _mspacTarget;
    _mspacTarget=nullptr;
    delete [] _mspacStats;
    _mspacStats=nullptr;
    delete [] _esacStats;
    _esacStats=nullptr;
    delete _peaks;
    _peaks=nullptr;

    for(int i=0; i<_nComponents; i++) {
      _rejected[i]=new int[_nFrequencies];
      for(int iFreq=0; iFreq<_nFrequencies; iFreq++) {
        _rejected[i][iFreq]=0;
      }
    }

    switch(param.method()) {
    case SPACParameters::MSPAC: {
        _mspacTarget=new AutocorrCurves;
        _mspacStats=new MSPACStatistics[_nFrequencies];
        for(int i=0; i<_nFrequencies; i++) {
          _mspacStats[i].setParameters(_nRings, _nComponents);
        }

        // Set the correct number of sample directly
        ModalCurve refCurve(param.frequencySampling().count());
        for(int iFreq=0; iFreq<_nFrequencies; iFreq++) {
          refCurve.setX(iFreq, param.frequencySampling().value(iFreq));
        }
        for(int iRing=0; iRing<_nRings; iRing++) {
          _mspacTarget->addRing(param.rings().at(iRing) );
          if(_nComponents==3) {
            _mspacTarget->add(refCurve);
            _mspacTarget->lastCurve().modes().last()=Mode(Mode::Vertical, iRing, 0);
            _mspacTarget->add(refCurve);
            _mspacTarget->lastCurve().modes().last()=Mode(Mode::Radial, iRing, 0);
            _mspacTarget->add(refCurve);
            _mspacTarget->lastCurve().modes().last()=Mode(Mode::Transverse, iRing, 0);
          } else {
            _mspacTarget->add(refCurve);
            _mspacTarget->lastCurve().modes().last()=Mode(Mode::Vertical, iRing, 0);
          }
        }
      }
      break;
    case SPACParameters::ESAC:
      _esacStats=new ESACStatistics[_nFrequencies];
      for(int i=0; i<_nFrequencies; i++) {
        _esacStats[i].setParameters(_nComponents);
      }
      break;
    }
    if(param.dotMaxOutput()) {
      _peaks=new SPACPeaks;
      _peaks->setTimeReference(_array->array()->first()->minTime());
    }
  }

  void SPACResults::addValue(Mode::Polarization polarization,
                             const ValueInfo& info,
                             double value)
  {
    _addLock.lock();
    if(_mspacStats) {
      _mspacStats[info.frequencyIndex].addValue(polarization, info.ringIndex, value);
    }
    if(_esacStats) {
      _esacStats[info.frequencyIndex].addValue(polarization, info.distance, value);
    }
    if(_peaks) {
      _peaks->add(polarization, info, value);
    }
    _addLock.unlock();
  }

  void SPACResults::addRejected(int frequency, int * rejected)
  {
    _addLock.lock();
    for(int i=0; i<_nComponents; i++) {
      _rejected[i][frequency]+=rejected[i];
    }
    _addLock.unlock();
  }

  void SPACResults::commitStatistics()
  {
    if(_mspacTarget && _mspacStats) {
      QList<ModalCurve>& curves=_mspacTarget->curves();
      for(int iFreq=0; iFreq<_nFrequencies; iFreq++) {
        for(int iRing=0; iRing<_nRings; iRing++) {
          int ringOffset=iRing*_nComponents;
          for(int iComp=0; iComp<_nComponents; iComp++){
            const Statistics& s=_mspacStats[iFreq].result(iRing, iComp);
            ModalCurve& curve=curves[ringOffset+iComp];
            FactoryPoint& p=curve.constXAt(iFreq);
            p.setMean(s.mean());
            p.setStddev(s.stddev());
            p.setWeight(s.count());
          }
        }
      }
    }
  }

  void SPACResults::reportRejects()
  {
    if(static_cast<const SPACParameters *>(_parameters)->maximumImaginary()>=1.0) {
      return;
    }
    int nVal, nRej;
    for(int iFreq=0; iFreq<_nFrequencies; iFreq++) {
      if(_nComponents==3) {
        if(_mspacStats) {
          nVal=_mspacStats[iFreq].count(1);
        } else if(_esacStats) {
          nVal=_esacStats[iFreq].count(1);
        } else {
          nVal=0;
        }
        nRej=_rejected[1][iFreq];
        App::log(tr("%1 values, %2 rejected (%3 %) at %4 Hz for radial component.\n")
                 .arg(nVal)
                 .arg(nRej)
                 .arg((100.0*nRej)/(nRej+nVal))
                 .arg(_parameters->frequencySampling().value(iFreq)));
        if(_mspacStats) {
          nVal=_mspacStats[iFreq].count(2);
        } else if(_esacStats) {
          nVal=_esacStats[iFreq].count(2);
        } else {
          nVal=0;
        }
        nRej=_rejected[2][iFreq];
        App::log(tr("%1 values, %2 rejected (%3 %) at %4 Hz for transverse component.\n")
                 .arg(nVal)
                 .arg(nRej)
                 .arg((100.0*nRej)/(nRej+nVal))
                 .arg(_parameters->frequencySampling().value(iFreq)));

      }
      if(_mspacStats) {
        nVal=_mspacStats[iFreq].count(0);
      } else if(_esacStats) {
        nVal=_esacStats[iFreq].count(0);
      } else {
        nVal=0;
      }
      nRej=_rejected[0][iFreq];
      App::log(tr("%1 values, %2 rejected (%3 %) at %4 Hz for vertical component.\n")
               .arg(nVal)
               .arg(nRej)
               .arg((100.0*nRej)/(nRej+nVal))
               .arg(_parameters->frequencySampling().value(iFreq)));
    }
  }

  bool SPACResults::save(QString fileName, const QString& log)
  {
    TRACE;
    QFileInfo fi(fileName);
    QDir d(fi.dir());
    if(_mspacTarget) {
      fileName=d.absoluteFilePath(fi.baseName()+".target");
      TargetList tl;
      tl.autocorrTarget().setCurves(*_mspacTarget);
      XMLHeader hdr(&tl);
      hdr.xml_saveFile(fileName);
      App::log(tr("Results saved in '%1'.\n").arg(fileName));
    }
    if(_esacStats) {
      fileName=d.absoluteFilePath(fi.baseName()+".target");
      TargetList tl;
      tl.dispersionTarget().addCurve(esacDispersionCurve(0));
      if(_nComponents==3) {
        tl.dispersionTarget().addCurve(esacDispersionCurve(1));
        tl.dispersionTarget().addCurve(esacDispersionCurve(2));
      }
      XMLHeader hdr(&tl);
      hdr.xml_saveFile(fileName);
      App::log(tr("Results saved in '%1'.\n").arg(fileName));
    }
    if(_peaks) {
      fileName=d.absoluteFilePath(fi.baseName()+".max");
      _peaks->sort();
      if(!_peaks->save(fileName, *parameters(), *_array, log)) {
        App::log(tr("Error saving results to '%1'.\n").arg(fileName));
        return false;
      }
      App::log(tr("Results saved in '%1'.\n").arg(fileName));
    }
    return true;
  }

  ModalCurve SPACResults::esacDispersionCurve(int iComp)
  {
    ModalCurve curve;
    if(_esacStats) {
      // Take array aperture as a starting wavenumber
      double k=2.0*M_PI/(2.0*_array->array()->radius());
      for(int iFreq=0; iFreq<_nFrequencies; iFreq++) {
        double f=_parameters->frequencySampling().value(iFreq);
        double omega=2.0*M_PI*f;
        k=_esacStats[iFreq].leastSquareWaveNumber(k, iComp);
        curve.append(FactoryPoint(f, k/omega));
      }
    }
    switch(iComp) {
    case 1:
      curve.addMode(Mode(Mode::Phase, Mode::Radial, 0));
      break;
    case 2:
      curve.addMode(Mode(Mode::Phase, Mode::Transverse, 0));
      break;
    default:
      curve.addMode(Mode(Mode::Phase, Mode::Vertical, 0));
      break;
    }
    return curve;
  }

#if 0
  // Export to stmap to be implemented in AutocorrCurves
  if(_loop->outputStmap(iComp)) {
    _loop->lockOutputStmap(iComp);
    const RingPairs& ring=_crossSpectrum->parameters()->ring(iRing);
    QTextStream * s=new QTextStream(_loop->outputStmap(iComp));
    (*s) << iRing << " "
         << index << " "
         << f << " "
         << tsum << " "
         << sqrt(tsum2) << " "
         << ring.minRadius()*0.001 << " "
         << ring.maxRadius()*0.001 << "\n";
    delete s;
    _loop->unlockOutputStmap(iComp);
  }
#endif

} // namespace ArrayCore

