/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-01-04
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "PoggiFKVertical.h"
#include "HRFKRadial.h"

namespace ArrayCore {

  /*!
    \class PoggiFKVertical PoggiFKVertical.h
    \brief Vertical high resolution FK power with ellipticity

    Vertical high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.
    Ellipticity is computed according Poggi et al. (2010) in GJI.
  */

  void PoggiFKVertical::resetCrossSpectrum()
  {
    _crossSpectrum->resetMatrix();
    _crossSpectrum->resetRotatedMatrices();
  }

  void PoggiFKVertical::addCrossSpectrum()
  {
    _crossSpectrum->addVertical();
    _crossSpectrum->addRotatedRadial(steering()->sensorOrientationErrors());
  }

  void PoggiFKVertical::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanMatrix(nBlocks);
    _crossSpectrum->meanRotatedMatrices(nBlocks);
  }

  bool PoggiFKVertical::invertCrossSpectrum()
  {
    return _crossSpectrum->invert() && _crossSpectrum->invertRotatedMatrices();
  }

  double PoggiFKVertical::ellipticity(const Vector<double>& k)
  {
    HRFKRadial radial(cache());
    PrivateVector<double> nck(k);
    return ::sqrt(radial.value(nck)/value(nck));
  }

} // namespace ArrayCore

