/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-09-30
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ORACrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class ORACrossSpectrum ORACrossSpectrum.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ORACrossSpectrum::ORACrossSpectrum(const ArraySelection * array,
                                     const ArrayParameters * param)
    : ArrayCrossSpectrum(array, param)
  {
    TRACE;
    _array=array;
    int n=stationCount();
    // At least half of the matrices will not be used but easier for indexing
    _pairMatrices=new ComplexMatrix[n*n];
    for(int i1=0; i1<n; i1++) {
      for(int i2=i1+1; i2<n; i2++) {
        _pairMatrices[i1+i2*n].resize(3);
      }
    }
    _sigs=new ComplexMatrix[n];
    for(int i=0; i<n; i++) {
      _sigs[i].resize(3, 1);
    }
  }

  /*!
    Description of destructor still missing
  */
  ORACrossSpectrum::~ORACrossSpectrum()
  {
    TRACE;
    delete [] _pairMatrices;
    delete [] _sigs;
  }

  bool ORACrossSpectrum::calculate(const VectorList<int>& blocks)
  {
    TRACE;
    int n=stationCount();
    for(int i1=0; i1<n; i1++) {
      for(int i2=i1+1; i2<n; i2++) {
        _pairMatrices[i1+i2*n].zero();
      }
    }

    // Build the three-component vectors for all stations
    const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
    int nFreq=frequencyFilter.maximumIndex();
    double filter;
    const ArrayStationSignals * stat;
    for(VectorList<int>::const_iterator it=blocks.begin(); it!=blocks.end(); it++) {
      if(lockTimeWindow(_timeWindows->list().at(*it))) {
        for(int i=0; i<n; i++) {
          stat=_stations.at(i);
          ComplexMatrix& sigs=_sigs[i];
          if(parameters()->frequencyBandwidth()==0.0) {
            sigs.at(0, 0)=stat->spectrumValue(_eastIndex);
            sigs.at(1, 0)=stat->spectrumValue(_northIndex);
            sigs.at(2, 0)=stat->spectrumValue(_verticalIndex);
          } else {
            for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
              filter=frequencyFilter.taperValue(iFreq);
              sigs.at(0, 0)=stat->spectrumValue(_eastIndex, iFreq)*filter;
              sigs.at(1, 0)=stat->spectrumValue(_northIndex, iFreq)*filter;
              sigs.at(2, 0)=stat->spectrumValue(_verticalIndex, iFreq)*filter;
            }
          }
        }
        unlockTimeWindow();
      } else {
        return false;
      }
    }

    // Compute the cross products
    for(int i1=0; i1<n; i1++) {
      for(int i2=i1+1; i2<n; i2++) {
        _pairMatrices[i1+i2*n]+=_sigs[i1]*_sigs[i2].conjugate().transposed();
      }
    }
    return true;
  }

  /*!
    Returns the radial direction for the pair \a i1, \a i2.
  */
  double ORACrossSpectrum::defaultAngle(int i1, int i2)
  {
    const VectorList<Point2D>& loc=_array->relativePos();
    return loc.at(i1).azimuthTo(loc.at(i2));
  }

  double ORACrossSpectrum::value(int i1, int i2, double theta1, double theta2) const
  {
    int n=stationCount();
    const ComplexMatrix& covmat=_pairMatrices[i1+i2*n];
    ComplexMatrix rot1(3), rot2(3);
    rot1.identity();
    rot1.at(0, 0)=cos(theta1);
    rot1.at(0, 1)=sin(theta1);
    rot1.at(1, 0)=-rot1.at(0, 1);
    rot1.at(1, 1)=rot1.at(0, 0);
    QTextStream(stdout) << "rot1=" << rot1.toOctaveString() << Qt::endl;
    rot2.identity();
    rot2.at(0, 0)=cos(theta2);
    rot2.at(0, 1)=-sin(theta2);  // transposed
    rot2.at(1, 0)=-rot2.at(0, 1);
    rot2.at(1, 1)=rot2.at(0, 0);
    QTextStream(stdout) << "rot2=" << rot2.toOctaveString() << Qt::endl;
    QTextStream(stdout) << "covmat=" << covmat.toOctaveString() << Qt::endl;
    ComplexMatrix r=rot1;
    r*=covmat;
    r*=rot2;
    QTextStream(stdout) << "r=" << r.toOctaveString() << Qt::endl;
    std::exit(2);
    return r.at(0, 1).abs2()+r.at(1, 0).abs2()+r.at(1, 2).abs2()+r.at(2, 1).abs2();
  }

} // namespace ArrayCore

