/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "HRFKVertical.h"
#include "FKPower.h"

namespace ArrayCore {

  /*!
    \class HRFKVertical HRFKVertical.h
    \brief Vertical high resolution FK power

    Vertical high resolution FK power versus wavenumber (k) function.
  */

// Get a detailled timing report of the processing
// Run with verbosity level to 3 to get the report.
// Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
  static double tgrid=0.0, tref=0.0, tinv=0.0;
  static Mutex globalTimeMutex;
#endif

  HRFKVertical::HRFKVertical(FKCache * cache)
    : ConvFKVertical(cache)
  {
    //_countGrid=0;
    //_countFree=0;
  }

  HRFKVertical::~HRFKVertical()
  {
    //App::log(tr("%1 grid values, %2 free values\n").arg(_countGrid).arg(_countFree));
#ifdef PROCESS_TIMING
    App::log(3, tr("HRFKVertical: %1 %2 %3 (ms; grid,  refine, mat inv)\n")
              .arg(tgrid*1e-6).arg(tref*1e-6).arg(tinv*1e-6));
#endif
  }

  bool HRFKVertical::invertCrossSpectrum()
  {
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    chrono.start();
#endif
    // Keep original before inversion for bright spot removal
    _originalCrossSpectrum=*_crossSpectrum->matrix();
    bool ret=_crossSpectrum->invert();
#ifdef PROCESS_TIMING
    qint64 t=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    tinv+=t;
    globalTimeMutex.unlock();
#endif
    return ret;
  }

  double HRFKVertical::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      //_countGrid++;
      powerEngine()->setHighResolutionValue(oneComponentSteering(index));
#ifdef PROCESS_TIMING
      qint64 t=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      tgrid+=t;
      globalTimeMutex.unlock();
#endif
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }
#if 0
  void HRFKVertical::setGradient(FKPower& p) const
  {
    //_countGrid++;
    p.setHighResolutionValue(_crossSpectrum->matrix());
    p.setOneComponentGradient();
  }

  void HRFKVertical::setHessian(FKPower &p) const
  {
    //_countGrid+=2;
    p.setOneComponentHessian(_crossSpectrum->matrix());
  }
#endif

  double HRFKVertical::value(Vector<double>& k) const
  {
    if(isInsideLimits(k)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      //_countFree++;
      powerEngine()->setHighResolutionValue(oneComponentSteering(k));
      //powerEngine()->setConventionalValue(_crossSpectrum->matrix(), e);
#ifdef PROCESS_TIMING
      qint64 t=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      tref+=t;
      globalTimeMutex.unlock();
#endif
      return powerEngine()->value();
    } else {
      return -1;
    }
  }

  /*!
    Either of the two value() functions must have been run before.
  */
  void HRFKVertical::gradient(const Vector<double>&, Vector<double>& grad) const
  {
    Q_UNUSED(grad);
    TODO_WARNING;
    /*powerEngine()->oneComponentGradient(steering()->xCoordinates(),
                                        steering()->yCoordinates(),
                                        grad);
                                        */
  }

#if 0
  void HRFKVertical::setGradient(const Point& k, FKPower& p) const
  {
    //_countFree++;
    FKSteeringOneComponent * s=static_cast<FKSteeringOneComponent *>(_steering);
    s->initValue(k);
    s->initGradient();
    p.setHighResolutionValue(_crossSpectrum->matrix());
    p.setOneComponentGradient();
  }

  void HRFKVertical::setHessian(const Point&, FKPower& p) const
  {
    FKSteeringOneComponent * s=static_cast<FKSteeringOneComponent *>(_steering);
    s->initHessian();
    p.setOneComponentHessian(_crossSpectrum->matrix());
  }
#endif


  bool HRFKVertical::remove(const Vector<double>& k, FKPeaks::Value& val)
  {
    // Get the cross-spectral matrix due to a plane wave travelling at k
    const Vector<Complex>& q0values=steering()->oneComponent(k, powerEngine());
    //ComplexMatrix incoherentNoise(n);
    //incoherentNoise.identity();
    //incoherentNoise*=noise/10.0;
    ComplexMatrix F=q0values.crossMatrix();
    //F+=incoherentNoise;
    F*=val.power/(1.0+val.verticalNoise);
    _originalCrossSpectrum-=F;
    *_crossSpectrum->matrix()=_originalCrossSpectrum;
    return invertCrossSpectrum();
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double HRFKVertical::wavenumberNormalizedConcavity(const Vector<double>& k) const
  {
    powerEngine()->setHighResolutionValue(oneComponentSteering(k));
    double p=powerEngine()->value();
    steering()->oneComponentSensorRadialProjections(k, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->steeringVector());
    return powerEngine()->highResolutionKRadialSecondDerivative()/p;
  }

  /*!
    Analyse with

    cat /tmp/kderivatives | awk '{print $1, $2}' | gpcurve -derivative | gpcurve -derivative | figue -c
    cat /tmp/kderivatives | figue -c -y-columns 1,2
  */
  void HRFKVertical::testKDerivatives(const Vector<double>& k) const
  {
    QFile f("/tmp/kderivatives");
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);

    PrivateVector<double> k1(k);
    k1/=k1.length(0, 1);
    for(double k=0; k<0.5; k+=0.001) {
      PrivateVector<double> kv(k1);
      kv*=k;
      const Vector<Complex>& e0=oneComponentSteering(kv);
      powerEngine()->setHighResolutionValue(e0);
      double p=powerEngine()->value();
      steering()->oneComponentSensorRadialProjections(kv, powerEngine());
      double dp=powerEngine()->highResolutionKRadialFirstDerivative(e0);
      double ddp=powerEngine()->highResolutionKRadialSecondDerivative();
      s << k << " " << p << " " << dp << " " << ddp << Qt::endl;
    }
    f.close();
    exit(0);
  }

} // namespace ArrayCore
