/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-11-17
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRayleighSquared.h"
#include "FKPower.h"
#include "HREllipticityEngine.h"

namespace ArrayCore {

//#define NEWTON_EVALUATION_STATS
#ifdef NEWTON_EVALUATION_STATS
  static qint64 nEvaluations[]={0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  static Mutex evaluationMutex;
#endif

  /*!
    \class HRFKRayleighSquared HRFKRayleighSquared.h
    \brief Rayleigh high resolution FK power

    Rayleigh high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.
  */

// Run with verbosity level to 3 to get the report.
// Level 3 also activate pre-loading of signals in AbstractTool.
// Configure with '-D PROCESS_TIMING'

#ifdef PROCESS_TIMING
  static double tgrid=0.0, tref=0.0, tell=0.0;
  static Mutex globalTimeMutex;
#endif

  HRFKRayleighSquared::HRFKRayleighSquared(FKCache * cache)
    : HRFKRayleigh(cache)
  {
    // We cannot start iterating from 0 and +/-pi/2 because they are minima
    _xi1.x=M_PI/4.0;
    _xi2.x=M_PI/4.0;
  }

  void HRFKRayleighSquared::initialize(const FKParameters& param)
  {
    HRFKRayleigh::initialize(param);
    _ellEngine=new HREllipticityEngine(cache(), param);
  }

  void HRFKRayleighSquared::showProcessingTime()
  {
#ifdef NEWTON_EVALUATION_STATS
    App::log(tr("Number of Newton attribute evaluations:\n"));
    qint64 sum=0;
    for(int i=2; i<=20; i++) {
      sum+=nEvaluations[i];
    }
    qint64 cum=0;
    for(int i=2; i<=20; i++) {
      cum+=nEvaluations[i];
      App::log(tr("%1 %2 % %3 % %4\n")
               .arg(i)
               .arg(100.0*nEvaluations[i]/sum)
               .arg(100.0*cum/sum)
               .arg(nEvaluations[i]));
    }
#endif
#ifdef PROCESS_TIMING
    App::log(3, tr("HRFKRayleighSquared (grid %1 ms,  refine %2 ms, ell %3 ms)\n")
              .arg(tgrid*1e-6).arg(tref*1e-6).arg(tell*1e-6));
#endif
  }

  inline void HRFKRayleighSquared::maximumEllipticity() const
  {
    _ellOptimization.maximize(_xi1, this, M_PI/8.0);
    canonicalEllipticity(_xi1.x);
    // The two peaks are theoretically at the same absolute value.
    _xi2.x=-_xi1.x;
    _ellOptimization.maximize(_xi2, this, M_PI/8.0);
    // Verification that the two peaks are at the same absolute value and opposite
    double dxi=fabs(_xi1.x+_xi2.x);
    while(dxi>M_PI/2.0) {
      dxi=fabs(dxi-M_PI);
    }
    if(dxi>M_PI/180.0) {
      _ellOptimization.curves<HRFKRayleighSquared, Attributes>(this, -M_PI, M_PI,  M_PI/180.0);
      _ellOptimization.step<HRFKRayleighSquared, Attributes>(this, -M_PI, M_PI,  M_PI/180.0, M_PI/8.0);
      fprintf(stderr, "xi1 and xi2 are too different: %lf and %lf\n", _xi1.x, _xi2.x);
      std::abort();
    }
#ifdef NEWTON_EVALUATION_STATS
    evaluationMutex.lock();
    if(_xi1.evaluationCount>=20) {
      nEvaluations[20]++;
    } else {
      nEvaluations[_xi1.evaluationCount]++;
    }
    if(_xi2.evaluationCount>=20) {
      nEvaluations[20]++;
    } else {
      nEvaluations[_xi2.evaluationCount]++;
    }
    evaluationMutex.unlock();
#endif
  }

  double HRFKRayleighSquared::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      steering()->oneComponent(index, powerEngine());
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      maximumEllipticity();
#ifdef PROCESS_TIMING
      qint64 t=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      tgrid+=t;
      globalTimeMutex.unlock();
#endif
      if(_xi1.value>_xi2.value) {
        return _xi1.value;
      } else {
        return _xi2.value;
      }
    } else {
      return -1.0;
    }
  }

  double HRFKRayleighSquared::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      steering()->oneComponent(kell, powerEngine());
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      maximumEllipticity();
#ifdef PROCESS_TIMING
    qint64 t=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    tref+=t;
    globalTimeMutex.unlock();
#endif
      if(_xi1.value>_xi2.value) {
        kell[2]=_xi1.x;
        return _xi1.value;
      } else {
        kell[2]=_xi2.x;
        return _xi2.value;
      }
    } else {
      return -1.0;
    }
  }

  double HRFKRayleighSquared::valueFixed(const Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      steering()->oneComponent(kell, powerEngine());
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      _xi1.x=kell[2];
      setFunctionValue(_xi1);
      return _xi1.value;
    } else {
      return -1.0;
    }
  }

  void HRFKRayleighSquared::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    // Get f=P_{p_h}
    const Vector<Complex>& e=FKSteering::twoComponentRayleighRadial(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.valueH=powerEngine()->value();
    a.valueH2=a.valueH*a.valueH;
    a.tan=a.ell.im()/a.ell.re();
    a.tan2=a.tan*a.tan;
    a.value=a.tan2*a.valueH2;
  }

  void HRFKRayleighSquared::setFunctionDerivatives(Attributes& a) const
  {
    // Get first derivative of P_{p_h}
    const Vector<Complex>& de=FKSteering::twoComponentRayleighRadialGradient(a.ell, powerEngine());
    a.slope=-2.0*a.valueH2*powerEngine()->ellipticityGradient(de);
    // Get second derivative of P_{p_h}
    double slope2=a.slope*a.slope;
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(slope2/a.valueH+a.tan*a.slope-a.valueH2*a.concavity);
    // Get second derivative of P_{p_s}
    double dtan=1.0/(a.ell.re()*a.ell.re());
    a.concavity=2.0*(dtan*(dtan+2.0*a.tan2)*a.valueH2
                     +4.0*a.tan*dtan*a.valueH*a.slope
                     +a.tan2*(slope2+a.valueH*a.concavity));
    // Get first derivative of P_{p_s}
    a.slope=2.0*(a.tan*dtan*a.valueH2+a.tan2*a.valueH*a.slope);
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double HRFKRayleighSquared::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    steering()->oneComponent(kell, powerEngine());
    powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));

    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::twoComponentRayleighRadial(ell, powerEngine());
    powerEngine()->setHighResolutionValue(e0);
    double p=powerEngine()->value();
    steering()->twoComponentSensorRadialProjections(kell, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    double tan=ell.im()/ell.re();
    return powerEngine()->highResolutionKRadialSecondDerivative()/(2.0*tan*tan*p*p);
  }

  void HRFKRayleighSquared::setAssociatedResults(const Vector<double>& kell, double power,
                                                 FKPeaks::Value& val) const
  {
    val.power=sqrt(power)/tan(kell[2]);
    rawEllipticity(kell, val);
  }

} // namespace ArrayCore

