/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-11-17
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRadial.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"

namespace ArrayCore {

  /*!
    \class HRFKRadial HRFKRadial.h
    \brief Radial high resolution FK power

    Radial high resolution FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.
  */

  bool HRFKRadial::invertCrossSpectrum()
  {
    return _crossSpectrum->invertRotatedMatrices();
  }

  double HRFKRadial::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      const Vector<Complex>& e=steering()->oneComponent(index, powerEngine());
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      powerEngine()->setHighResolutionValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }

  double HRFKRadial::value(Vector<double>& k) const
  {
    if(isInsideLimits(k)) {
      const Vector<Complex>& e=steering()->oneComponent(k, powerEngine());
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(k));
      powerEngine()->setHighResolutionValue(e);
      return powerEngine()->value();
    } else {
      return -1.0;
    }
  }


  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double HRFKRadial::wavenumberNormalizedConcavity(const Vector<double>& k) const
  {
    powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(k));
    powerEngine()->setHighResolutionValue(steering()->oneComponent(k, powerEngine()));
    double p=powerEngine()->value();
    steering()->oneComponentSensorRadialProjections(k, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->steeringVector());
    return powerEngine()->highResolutionKRadialSecondDerivative()/p;
  }


} // namespace ArrayCore

