/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-09-10
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRayleighAll.h"
#include "HRDirectEllipticityEngine.h"
#include "FKCrossSpectrum.h"

namespace ArrayCore {

// Get a detailled timing report of the processing
// Run with verbosity level to 3 to get the report.
// Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
  static double tgrid=0.0, tref=0.0;
  static Mutex globalTimeMutex;
#endif

//#define NEWTON_EVALUATION_STATS
#ifdef NEWTON_EVALUATION_STATS
  static qint64 nEvaluations[]={0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  static Mutex evaluationMutex;
#endif
//#define ELLIPTICITY_GRID_OPTIMIZATION

  /*!
    \class HRFKDirectRayleighAll HRFKDirectRayleighAll.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of destructor still missing
  */
  HRFKDirectRayleighAll::~HRFKDirectRayleighAll()
  {
#ifdef NEWTON_EVALUATION_STATS
    App::log(tr("Number of Newton attribute evaluations:\n"));
    qint64 sum=0;
    Statistics stat;
    for(int i=2; i<=20; i++) {
      stat.add(i, nEvaluations[i]);
      sum+=nEvaluations[i];
    }
    qint64 cum=0;
    int ev;
    for(int i=2; i<=20; i++) {
      cum+=nEvaluations[i];
#ifdef ELLIPTICITY_GRID_OPTIMIZATION
      ev=i*4;
#else
      ev=i;
#endif
      App::log(tr("%1 %2 % %3 % %4\n")
               .arg(ev)
               .arg(100.0*nEvaluations[i]/sum)
               .arg(100.0*cum/sum)
               .arg(nEvaluations[i]));
    }
    App::log(tr("Average %1+/-%2\n").arg(stat.mean()).arg(stat.stddev()));
#endif
#ifdef PROCESS_TIMING
    App::log(3, tr("HRFKDirectRayleighAll: %1 %2 %3 (ms; grid,  refine)\n")
              .arg(tgrid*1e-6).arg(tref*1e-6));
#endif
  }

  inline void HRFKDirectRayleighAll::commitEvaluationStat() const
  {
#ifdef NEWTON_EVALUATION_STATS
    evaluationMutex.lock();
    if(_xi.evaluationCount>=20) {
      nEvaluations[20]++;
    } else {
      nEvaluations[_xi.evaluationCount]++;
    }
    evaluationMutex.unlock();
#endif
  }


  void HRFKDirectRayleighAll::initialize(const FKParameters& param)
  {
    HRFKDirectRayleigh::initialize(param);
    _ellEngine=new HRDirectEllipticityEngine(cache(), param);
  }

  inline void HRFKDirectRayleighAll::maximumEllipticity() const
  {
    // We admit values ouside -pi/2 to p/2 to avoid being trapped right at +/- pi/2
    // This happens if _xi is between the limits and the minimum
    double xi0=_xi.x;
    if(!_ellOptimization.maximize(_xi, this, M_PI/8.0)) {
      App::log(tr("Error maximizing beampower for ellipticity (cross)\n"));
      // to debug optimization errors
      _ellOptimization.curves<HRFKDirectRayleighAll, Attributes>(this, -M_PI/2, M_PI/2, M_PI/1800.0);
      _xi.x=xi0;
      _ellOptimization.maximize(_xi, this, M_PI/8.0);
    }
    commitEvaluationStat();
  }

  inline void HRFKDirectRayleighAll::gridMaximumEllipticity() const
  {
    _xi.evaluationCount=0;
    double step=M_PI/9.0;
    double maxValue=0.0, maxX=0.0;
    for(_xi.x=-M_PI/2.0; _xi.x<M_PI/2.0; _xi.x+=step) {
      _xi.evaluationCount++;
      setFunctionValue(_xi);
      if(_xi.value>maxValue) {
        maxValue=_xi.value;
        maxX=_xi.x;
      }
    }
    // Refine
    double bestX=maxX;
    while(step>_ellOptimization.absolutePrecision()) {
      step*=0.5;
      _xi.x=bestX-step;
      setFunctionValue(_xi);
      _xi.evaluationCount++;
      if(_xi.value>maxValue) {
        maxValue=_xi.value;
        maxX=_xi.x;
      }
      _xi.x=bestX+step;
      setFunctionValue(_xi);
      _xi.evaluationCount++;
      if(_xi.value>maxValue) {
        maxValue=_xi.value;
        maxX=_xi.x;
      }
      bestX=maxX;
    }
    _xi.evaluationCount/=4;  // Keep results on the same range as the derivative search (max=20)
    _xi.x=bestX;
    _xi.value=maxValue;
    commitEvaluationStat();
  }

  double HRFKDirectRayleighAll::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      steering()->threeComponentRayleigh(index, powerEngine());
#ifdef ELLIPTICITY_GRID_OPTIMIZATION
      gridMaximumEllipticity();
#else
      maximumEllipticity();
#endif
#ifdef PROCESS_TIMING
      qint64 t=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      tgrid+=t;
      globalTimeMutex.unlock();
#endif
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRayleighAll::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
#ifdef PROCESS_TIMING
      QElapsedTimer chrono;
      chrono.start();
#endif
      steering()->threeComponentRayleigh(kell, powerEngine());
      //_ellOptimization.curves<HRFKDirectRayleighAll, Attributes>(this, -M_PI, M_PI,  M_PI/180.0);
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
#ifdef PROCESS_TIMING
      qint64 t=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      tref+=t;
      globalTimeMutex.unlock();
#endif
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRayleighAll::valueFixed(const Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      steering()->threeComponentRayleigh(kell, powerEngine());
      _xi.x=kell[2];
      setFunctionValue(_xi);
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void HRFKDirectRayleighAll::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::threeComponentRayleighCross(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.value=powerEngine()->value();
  }

  void HRFKDirectRayleighAll::setFunctionDerivatives(Attributes& a) const
  {
    double value2=a.value*a.value;
    // Get first derivative
    const Vector<Complex>& de=FKSteering::threeComponentRayleighCrossGradient(a.ell, powerEngine());
    a.slope=-2.0*value2*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.slope*a.slope/a.value+a.value-value2*a.concavity);
  }

  bool HRFKDirectRayleighAll::remove(const Vector<double>& kell, FKPeaks::Value& val)
  {
    // Get the cross-spectral matrix due to a plane wave travelling at k and with ell.
    const Vector<Complex>& q0values=steering()->oneComponent(kell, powerEngine());
    Angle theta0(kell[0], kell[1]);
    Complex ell0;
    ell0.setUnitExp(atan(kell[2]));
    int n=_crossSpectrum->array().count();
    int n2=n*2;
    ComplexMatrix V0(n2, 1);
    ComplexMatrix q0(n, 1);
    ComplexMatrix q0ct(1, n);
    for(int i=0; i<n; i++) {
      V0.at(i, 0)=theta0.cos()*q0values[i];
      V0.at(i+n, 0)=theta0.sin()*q0values[i];
      q0.at(i, 0)=q0values[i];
      q0ct.at(0, i)=conjugate(q0values[i]);
    }
    ComplexMatrix Fh=V0*V0.conjugate().transposed();
    Fh*=ell0.im()*ell0.im();
    ComplexMatrix Fz=q0*q0ct;
    Fz*=ell0.re()*ell0.re();
    ComplexMatrix Fx=V0*q0ct;
    Fx*=Complex(0.0, -1.0)*ell0.re()*ell0.im();
    ComplexMatrix F(n*3);
    F.copyAt(0, 0, Fh);
    F.copyAt(0, n2, Fx);
    Fx=Fx.conjugate();
    Fx.transpose();
    F.copyAt(n2, 0, Fx);
    F.copyAt(n2, n2, Fz);
    F*=val.power/(1.0+val.verticalNoise/static_cast<double>(n));
    //_originalCrossSpectrum-=F;
    //_crossSpectrum->matrix()=_originalCrossSpectrum;
    return invertCrossSpectrum();
  }

  /*!
    Return the ratio of beampower over its the second derivative.
  */
  double HRFKDirectRayleighAll::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    steering()->threeComponentRayleigh(kell, powerEngine());

    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::threeComponentRayleighCross(ell, powerEngine());
    powerEngine()->setHighResolutionValue(e0);
    double p=powerEngine()->value();
    steering()->threeComponentSensorRadialProjections(kell, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    return powerEngine()->highResolutionKRadialSecondDerivative()/p;
  }

  /*!
    value() or valueFixed() must be called before with the same \a kell.
    If \a axis is 0, radial wavenumber concavity is returned.
    If \a axis is 2, ellipticity concavity is returned.
  */
  double HRFKDirectRayleighAll::concavity(const Vector<double>& kell, int axis) const
  {
    // Recompute fe, altered by setFunctionDerivatives()
    setFunctionValue(_xi);
    switch(axis) {
    case 2:
      setFunctionDerivatives(_xi); // Required if valueFixed() is called before
      return _xi.concavity;
    default:
      steering()->threeComponentSensorRadialProjections(kell, powerEngine());
      powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
      return powerEngine()->highResolutionKRadialSecondDerivative();
    }

  }

} // namespace ArrayCore

