/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-04-12
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef FKPOWER_H
#define FKPOWER_H

#include <QGpCoreMath.h>

#include "ArrayCoreDLLExport.h"

namespace ArrayCore {

  class ArraySelection;

  class ARRAYCORE_EXPORT FKPower
  {
  public:
    FKPower(int stationCount, int matrixSizeFactor);
    ~FKPower();

    void setCrossSpectrum(const ComplexMatrix * f) {_f=f;}
    const ComplexMatrix * crossSpectrum() const {return _f;}

    void setOneComponentSteeringVector(const Vector<Complex> * q) {_q=q;}
    const Vector<Complex>& oneComponentSteeringVector() {return *_q;}

    int stationCount() const {return _eGrid.count();}
    Vector<Complex>& steeringVector() {return _e;}
    Vector<Complex>& ellipticitySteeringVector() {return _exi;}
    Vector<Complex>& ellipticityDerivativeSteeringVector() {return _edxi;}
    Vector<Complex>& steeringVectorGrid() {return _eGrid;}
    const Vector<Complex>& fe() const {return _fe;}
    Vector<double>& sensorProjections() {return _r;}
    double factorN() const {return _factorN;}

    inline Complex steering(const Vector<Complex>& e);
    inline void setConventionalValue(const Vector<Complex>& e);
    inline void setHighResolutionValue(const Vector<Complex>& e);
    inline void setRayleighTwoComponentValue(double ell, const Vector<Complex>& e);
    inline void setRayleighThreeComponentValue(double ell, const Vector<Complex>& e);

    inline double ellipticityGradient(const Vector<Complex>& de) const;
    inline double orientationGradient(const ComplexMatrix& df);

    inline double conventionalKRadialFirstDerivative(const Vector<Complex>& e);
    inline double conventionalKRadialSecondDerivative();

    inline double highResolutionKRadialFirstDerivative(const Vector<Complex>& e);
    double highResolutionKRadialFirstDerivative() const {return _dp;}
    inline double highResolutionKRadialSecondDerivative();
    inline double xiSecondDerivative();
    /*
    inline void setThreeComponentGradient();
    inline void setOneComponentHessian(const ComplexMatrix& crossSpectrum);
    inline void setThreeComponentHessian(const ComplexMatrix& crossSpectrum);*/

    double value() const {return _p.re();}
    void setValue(double p) {_p=p;}
    double value2() const {return _p2;}
    //const Point& gradient() const {return _gradp;}
    //const DoubleMatrix& hessian() const {return _hessian;}
    //inline double concavity(const Matrix3x3 * scales=nullptr) const;
    //inline Point newtonStep() const;

    QString toString(char format='g', int precision=6) const;
    void toOctaveFile(const QString& fileName) const;
  private:
    const ComplexMatrix * _f;
    PrivateVector<Complex> _fe, _ddfe, _e, _exi, _edxi, _re, _re2;
    PrivateVector<double> _r;
    const Vector<Complex> * _q;
    Vector<Complex> _eGrid;
    Complex _p;
    double _p2, _dp, _factorN;
  };

  inline Complex FKPower::steering(const Vector<Complex>& e)
  {
    _fe.multiply(*_f, e);
    return _fe.scalarProductConjugate(e);
  }

  inline void FKPower::setHighResolutionValue(const Vector<Complex>& e)
  {
    _p=steering(e);
    // Even if imaginary part is not null and even not negligible compared to the
    // real part, the real part remains correctly estimated. Imaginary part should be
    // null in all cases but numerical errors for ill-posed cross spectral matrices
    // lead to non null imaginary.
    // Compared to previous versions, the imaginary part is not inversed. This modification
    // implies very minor changes (vertical Capon).
#ifdef COMPATIBILITY_3_4_1
    _p=1.0/_p;
#else
    _p.setRe(1.0/_p.re());
#endif
  }

  inline void FKPower::setConventionalValue(const Vector<Complex>& e)
  {
    _p=steering(e);
    _p*=_factorN;
  }

  inline void FKPower::setRayleighTwoComponentValue(double ell, const Vector<Complex>& e)
  {
    setHighResolutionValue(e);
    double ps=_p.re()*_p.re()*ell*ell;
    _p=ps;
  }

  inline void FKPower::setRayleighThreeComponentValue(double ell, const Vector<Complex>& e)
  {
    setHighResolutionValue(e);
    double ps=pow(_p.re(), 1.5)*ell*ell;
    //_3pps=3.0*_p*ps;
    _p=ps;
  }

  inline double FKPower::conventionalKRadialFirstDerivative(const Vector<Complex>& e)
  {
    _re.dotMultiply(_r, e);
    _dp=-2.0*_factorN*_fe.scalarProductConjugate(_re).im();
    return _dp;
  }

  inline double FKPower::conventionalKRadialSecondDerivative()
  {
    _re2.dotMultiply(_r, _re);
    double dp2=_fe.scalarProductConjugate(_re2).re();
    _fe.multiply(*_f, _re);
    dp2-=_fe.scalarProductConjugate(_re).re();
    return -2.0*_factorN*dp2;
  }

  inline double FKPower::highResolutionKRadialFirstDerivative(const Vector<Complex>& e)
  {
    _p2=2.0*_p.re()*_p.re();
    _re.dotMultiply(_r, e);
    _dp=_p2*_fe.scalarProductConjugate(_re).im();
    return _dp;
  }

  inline double FKPower::highResolutionKRadialSecondDerivative()
  {
    double dp2=2.0*_dp*_dp/_p.re();
    _re2.dotMultiply(_r, _re);
    dp2+=_p2*_fe.scalarProductConjugate(_re2).re();
    _ddfe.multiply(*_f, _re);
    dp2-=_p2*_ddfe.scalarProductConjugate(_re).re();
    return dp2;
  }

  inline double FKPower::xiSecondDerivative()
  {
    _ddfe.multiply(*_f, _edxi);
    return _ddfe.scalarProductConjugate(_edxi).re();
  }

  inline double FKPower::ellipticityGradient(const Vector<Complex>& de) const
  {
    return _fe.scalarProductConjugate(de).re();
  }

  inline double FKPower::orientationGradient(const ComplexMatrix& df)
  {
    _exi.multiply(df, _fe);
    return _exi.scalarProductConjugate(_fe).re();
  }

  /*inline void FKPower::initOneComponent()
  {
    //_hessian.zero();
    //_gradp.setZ(0.0);
  }*/

  /*

  inline void FKPower::setThreeComponentGradient()
  {
    _thfe=(_steering->_th*_fe).at(0, 0).im();
    _uhfe=(_steering->_uh*_fe).at(0, 0).im();
    _gradp.setX(_3pps*((_steering->_ehrx*_fe).at(0, 0).im()
                        -_steering->_esink*_thfe));
    _gradp.setY(_3pps*((_steering->_ehry*_fe).at(0, 0).im()
                        +_steering->_ecosk*_thfe));
    _gradp.setZ((_3pps*_uhfe+_steering->_2inve*_p)*_steering->_ellFactor);
  }

  inline void FKPower::setOneComponentHessian(const ComplexMatrix& crossSpectrum)
  {
    ComplexMatrix ferx=crossSpectrum*_steering->_rxe;
    ComplexMatrix fery=crossSpectrum*_steering->_rye;
    double inv2p=2.0/_p;
    _hessian.at(0, 0)=inv2p*_gradp.x()*_gradp.x()+_2p2*((_steering->_ehrxrx*_fe).at(0, 0).re()
                                                       -(_steering->_ehrx*ferx).at(0, 0).re());
    _hessian.at(0, 1)=inv2p*_gradp.x()*_gradp.y()+_2p2*((_steering->_ehrxry*_fe).at(0, 0).re()
                                                       -(_steering->_ehrx*fery).at(0, 0).re());
    _hessian.at(1, 0)=_hessian.at(0, 1);
    _hessian.at(1, 1)=inv2p*_gradp.y()*_gradp.y()+_2p2*((_steering->_ehryry*_fe).at(0, 0).re()
                                                       -(_steering->_ehry*fery).at(0, 0).re());
  }

  inline void FKPower::setThreeComponentHessian(const ComplexMatrix& crossSpectrum)
  {
    double inv53ps=5.0/3.0/_p;
    ComplexMatrix ferx=crossSpectrum*_steering->_rxe;
    ComplexMatrix fery=crossSpectrum*_steering->_rye;
    ComplexMatrix ft=crossSpectrum*_steering->_t;

    double thrxfe=(_steering->_thrx*_fe).at(0, 0).re();
    double thryfe=(_steering->_thry*_fe).at(0, 0).re();
    double thft=(_steering->_th*ft).at(0, 0).re();
    double thferx=(_steering->_th*ferx).at(0, 0).re();
    double thfery=(_steering->_th*fery).at(0, 0).re();
    double dthfex=thferx-thrxfe;
    double dthfey=thfery-thryfe;
    double ecossink2thfe=2.0*_steering->_ecossink2*_thfe;
    double uhft=(_steering->_uh*ft).at(0, 0).re();

    _hessian.at(0, 0)=inv53ps*_gradp.x()*_gradp.x()+_3pps*( (_steering->_ehrxrx*_fe).at(0, 0).re()
                                                           -(_steering->_ehrx*ferx).at(0, 0).re()
                                                           +2.0*_steering->_esink*dthfex
                                                           +ecossink2thfe
                                                           -_steering->_esin2k2*_uhfe
                                                           -_steering->_e2sin2k2*thft);
    _hessian.at(0, 1)=inv53ps*_gradp.x()*_gradp.y()+_3pps*( (_steering->_ehrxry*_fe).at(0, 0).re()
                                                           -(_steering->_ehrx*fery).at(0, 0).re()
                                                           -_steering->_ecosk*dthfex
                                                           +_steering->_esink*dthfey
                                                           +(_steering->_esin2k2-_steering->_ecos2k2)*_thfe
                                                           +_steering->_ecossink2*_uhfe
                                                           +_steering->_e2cossink2*thft);
    _hessian.at(1, 1)=inv53ps*_gradp.y()*_gradp.y()+_3pps*( (_steering->_ehryry*_fe).at(0, 0).re()
                                                           -(_steering->_ehry*fery).at(0, 0).re()
                                                           -2.0*_steering->_ecosk*dthfey
                                                           -ecossink2thfe
                                                           -_steering->_ecos2k2*_uhfe
                                                           -_steering->_e2cos2k2*thft);
    _hessian.at(0, 2)=( inv53ps*_gradp.x()*_gradp.z()
                       -_steering->_4inv3e*_gradp.x()+_3pps*( (_steering->_uhrx*_fe).at(0, 0).re()
                                                             -(_steering->_uh*ferx).at(0, 0).re()
                                                             -_steering->_sink*_thfe
                                                             +_steering->_esink*uhft))*_steering->_ellFactor;
    _hessian.at(1, 2)=( inv53ps*_gradp.y()*_gradp.z()
                       -_steering->_4inv3e*_gradp.y()+_3pps*( (_steering->_uhry*_fe).at(0, 0).re()
                                                              -(_steering->_uh*fery).at(0, 0).re()
                                                              +_steering->_cosk*_thfe
                                                              -_steering->_ecosk*uhft))*_steering->_ellFactor;
    _hessian.at(2, 2)=( inv53ps*_gradp.z()*_gradp.z()
                       -_steering->_8inv3e*_gradp.z()
                       +_steering->_2inv3e2*_p
                       -_3pps*(_steering->_uh*crossSpectrum*_steering->_u).at(0, 0).re())*_steering->_ellFactor2
                       +_steering->_2ell*_gradp.z();

    _hessian.at(1, 0)=_hessian.at(0, 1);
    _hessian.at(2, 0)=_hessian.at(0, 2);
    _hessian.at(2, 1)=_hessian.at(1, 2);
  }

  inline double FKPower::concavity(const Matrix3x3 * scales) const
  {
    bool ok=true;
    VectorList<double> eigenValues;
    if(scales) {
      Matrix3x3 sh=_hessian.dotMultiply(*scales);
      eigenValues=sh.symetricEigenValues(ok);
    } else {
      eigenValues=_hessian.symetricEigenValues(ok);
    }
    if(ok) {
      for(int i=eigenValues.count()-1; i>=0; i--) {
        if(eigenValues.at(i)>0.0) {
          return 1.0;
        }
      }
      return -1.0;
    } else {
      return 0.0;
    }
  }

  inline Point FKPower::newtonStep() const
  {
    Matrix3x3 hinv=_hessian;
    if(hinv.invert()) {           // Spending a bit of time for OneComponent is not so dramatic
                                  // ThreeComponent is more critical
                                  // newtonStep is used exclusively when concavity is negative
                                  // the Hessian is always invertible.
      return -(hinv*_gradp);
    } else {
      Matrix2x2 hinv=_hessian.subMatrix(0, 0, 1, 1);
      if(hinv.invert()) {
        return -(hinv*_gradp);
      } else {
        return Point();
      }
    }
  }*/

} // namespace ArrayCore

#endif // FKPOWER_H

