/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-08
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "FKParameters.h"

namespace ArrayCore {

  /*!
    \class FKParameters FKParameters.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  FKParameters::FKParameters()
    : ArrayParameters()
  {
    _processType=Undefined;
    _ellipticityCorrection=None;
    _inversionMethod=RefinedGrid;
    setDefaultValues();
    _cacheGridStepFactor=0.05;
    _cacheGridStep=0.0;
    _gridStep=0.0;
    _gridSize=0.0;
    _kmin=0.0;
    _kmax=0.0;
    _smin=1.0/3500.0;
    _smax=1.0/50.0;
    _minAzimuth=0.0;
    _maxAzimuth=0.0;
    _sourceGridStep=1.0;
    _sourceGridSize=0.0;
    // dV/V=-dk/k, hence a relative precision in wavenumber is approximately the
    // same as a relative precision in velocity or slowness (dV/V=-ds/s)
    _velocityRelativePrecision=1e-3;
    _ellipticityAbsolutePrecision=M_PI/180000.0;

    _sensorOrientationApertureAngle=M_PI;
    _sensorOrientationMaximumRadius=0.0;
    _sensorOrientationMaximumCorrection=M_PI/9.0;
    _sensorOrientationMaximumIterations=100;

    _brightSpots=0;
  }

  /*!
    Set default processing type according to mode \a m if it is not
    already set. Other values are also set to their defaults.
  */
  void FKParameters::setDefaultValues(ArrayStations::Mode m)
  {
    if(_processType==Undefined) {
      switch(m) {
      case ArrayStations::Vertical:
        _processType=Capon;
        break;
      case ArrayStations::Horizontal:
        _processType=CaponRadial;
        break;
      case ArrayStations::ThreeComponents:
        _processType=ARDS;
        break;
      }
    }
    setDefaultValues();
  }

  /*!
    Process type and inversion mthod must be already set.
  */
  void FKParameters::setDefaultValues()
  {
    _maxPeakCount=0;
    _maxPeakCountFactor=1.0;
    _absoluteThreshold=0.0;
    _relativeThreshold=0.9;

    _saveUndefinedEllipticities=true;
    _sortResultValues=true;
    _exportAllFKGrids=false;
    _damping=0.0;
    _rotateStepCount=72;
    setfrequencyBandwidth(0.0);
    blockAveraging().setStatisticCount(50);
    blockAveraging().setStatisticMaxOverlap(0.0);

    switch(_processType) {
    case Undefined:
    FKPARAMETERS_PASSIVE_PROCESS_TYPES
      windowing().setPeriodCount(50.0);
      setOversamplingFactor(1.0);
      _minimumDistance=0.0;
      _maximumDistance=std::numeric_limits<double>::infinity();
      windowing().setSeismicEventTrigger(false);
      setSelectDurationFactor(0.0);
      break;
    FKPARAMETERS_ACTIVE_PROCESS_TYPES
      windowing().setLength(5.0);
      setOversamplingFactor(5.0);
      _minimumDistance=10.0;
      _maximumDistance=100.0;
      windowing().setSeismicEventTrigger(true);
      windowing().setSeismicEventDelay(-0.1);
      setSelectDurationFactor(0.0); // required for multi-shot with huge gaps
      break;
    }

    switch(_processType) {
    case ARDS:
    case LDS3:
    case SensorOrientation:
    case RDSSingle:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(6.0);
      _relativeThreshold=0.01;
      _saveUndefinedEllipticities=false;
      break;
    case Undefined:
    case Capon:
    case CaponRadial:
    case CaponTransverse:
    case Omni:
    case PoggiRadial:
    case PoggiVertical:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(2.0);
      break;
    case RTBF:
    case ARTBF:
    case RTBFFixedEll:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(4.0);
      _relativeThreshold=0.01;
      _inversionMethod=RefinedGrid; // Gradient not supported for RTBF
      _saveUndefinedEllipticities=false;
      break;
    case LDS2:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(4.0);
      break;
    case Conventional:
    case ConventionalRadial:
    case ConventionalTransverse:
    case ConventionalRayleigh:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(0.5);
      break;
    case ActiveConventionalRayleigh:
      blockAveraging().setCount(0);
      blockAveraging().setCountFactor(0.5);
      setMaximumPeakCount(1);
      break;
    case ActiveConventional:
      blockAveraging().setCount(1);
      setMaximumPeakCount(1);
      break;
    case ActiveRTBF:
    case ActiveCapon:
      blockAveraging().setCount(3);
      break;
    }
    // Ellipticity correction
    switch(_processType) {
    case ARDS:
    case ARTBF:
      _ellipticityCorrection=ThreeComponent;
      break;
    default:
      _ellipticityCorrection=None;
      break;
    }
    _gridStepFactor=defaultGridStepFactor(_processType, _inversionMethod);
    _gridSizeFactor=2.0;
    // Force default grid step and size if they are not yet set
    setKmin(_kmin);
    setKmax(_kmax);
  }

  double FKParameters::defaultGridStepFactor(ProcessType t, InversionMethod m)
  {
    switch(t) {
    case Undefined:
    case Capon:
    case Omni:
    case CaponRadial:
    case CaponTransverse:
    case RTBF:
    case ARTBF:
    case RTBFFixedEll:
    case ARDS:
    case SensorOrientation:
    case RDSSingle:
    case PoggiVertical:
    case PoggiRadial:
    case LDS2:
    case LDS3:
    case ActiveCapon:
    case ActiveRTBF:
      switch(m) {
      case Gradient:
        return 0.5;
      case RefinedGrid:
        return 0.1;
      }
    case Conventional:
    case ConventionalRadial:
    case ConventionalTransverse:
    case ConventionalRayleigh:
    case ActiveConventional:
    case ActiveConventionalRayleigh:
      break;
    }
    switch(m) {
    case Gradient:
      break;
    case RefinedGrid:
      return 0.25;
    }
    return 1.0;
  }

  double FKParameters::cacheGridStep() const
  {
    if(_cacheGridStep==0.0) {
      return _kmin*_cacheGridStepFactor;
    } else {
      return _cacheGridStep;
    }
  }

  double FKParameters::gridStep() const
  {
    if(_gridStep==0.0) {
      return _kmin*_gridStepFactor;
    } else {
      return _gridStep;
    }
  }

  double FKParameters::gridSize() const
  {
    if(_gridSize==0.0) {
      return _kmax*_gridSizeFactor;
    } else {
      return _gridSize;
    }
  }

  void FKParameters::setKmin(double k)
  {
    _kmin=k;
  }

  void FKParameters::setKmax(double k)
  {
    _kmax=k;
  }

  int FKParameters::maximumPeakCount(int sensorCount) const
  {
    if(_maxPeakCount<1) {
      int n=qRound(_maxPeakCountFactor*sensorCount);
      if(n<1) {
        n=1;
      }
      return n;
    } else {
      return _maxPeakCount;
    }
  }

  /*!
    Limit grid size to fall inside the frequency-slowness area of interest.
  */
  double FKParameters::effectiveGridSize() const
  {
    double ksmax=2.0*M_PI*frequencySampling().maximum()*_smax;
    double gs=gridSize();
    // For the GUI frequency sampling maximum is usually null
    if(ksmax<gs && ksmax>0.0) {
      return ksmax;
    } else {
      return gs;
    }
  }

  /*!
    Convenience function to get the effective grid size at \a frequnecy,
    taking into account the maximum slowness.
  */
  double FKParameters::effectiveGridSize(double frequency) const
  {
    TRACE;
    double ksmax=2.0*M_PI*frequency*_smax;
    double gs=gridSize();
    if(ksmax<gs) {
      return ksmax;
    } else {
      return gs;
    }
  }

  void FKParameters::clearCurveRefine()
  {
    TRACE;
    _refineSlowMin.clear();
    _refineSlowMax.clear();
  }

  void FKParameters::addCurveRefine(double f, double minSlow, double maxSlow)
  {
    TRACE;
    _refineSlowMin.append(Point2D(f, minSlow));
    _refineSlowMax.append(Point2D(f, maxSlow));
  }

  int FKParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 41+ArrayParameters::keywordCount();
  }

  void FKParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=ArrayParameters::keywordCount();
    ArrayParameters::collectKeywords(keywords, prefix, suffix);
    keywords.add(prefix+"INVERSION_METHOD"+suffix, this, baseIndex+19);
    keywords.add(prefix+"CACHE_GRID_STEP"+suffix, this, baseIndex+20);
    keywords.add(prefix+"CACHE_GRID_STEP_FACTOR"+suffix, this, baseIndex+21);
    keywords.add(prefix+"GRID_STEP"+suffix, this, baseIndex+16);
    keywords.add(prefix+"GRID_STEP_FACTOR"+suffix, this, baseIndex+18);
    keywords.add(prefix+"GRID_SIZE"+suffix, this, baseIndex+17);
    keywords.add(prefix+"GRID_SIZE_FACTOR"+suffix, this, baseIndex+28);
    keywords.add(prefix+"K_MIN"+suffix, this, baseIndex);
    keywords.add(prefix+"K_MAX"+suffix, this, baseIndex+1);
    keywords.add(prefix+"MIN_V"+suffix, this, baseIndex+2);
    keywords.add(prefix+"MAX_V"+suffix, this, baseIndex+24);
    keywords.add(prefix+"MIN_AZIMUTH"+suffix, this, baseIndex+25);
    keywords.add(prefix+"MAX_AZIMUTH"+suffix, this, baseIndex+26);
    keywords.add(prefix+"N_MAXIMA"+suffix, this, baseIndex+3);
    keywords.add(prefix+"N_MAXIMA_FACTOR"+suffix, this, baseIndex+40);
    keywords.add(prefix+"ABSOLUTE_THRESHOLD"+suffix, this, baseIndex+4);
    keywords.add(prefix+"RELATIVE_THRESHOLD"+suffix, this, baseIndex+5);
    keywords.add(prefix+"EXPORT_ALL_FK_GRIDS"+suffix, this, baseIndex+6);
    keywords.add(prefix+"SAVE_UNDEFINED_ELLIPTICITIES"+suffix, this, baseIndex+27);
    keywords.add(prefix+"SORT_RESULT_VALUES"+suffix, this, baseIndex+29);
    keywords.add(prefix+"DAMPING_FACTOR"+suffix, this, baseIndex+7);
    keywords.add(prefix+"ROTATE_STEP_COUNT"+suffix, this, baseIndex+8);
    keywords.add(prefix+"PROCESS_TYPE"+suffix, this, baseIndex+9);
    keywords.add(prefix+"ARRAY"+suffix, this, baseIndex+10);
    keywords.add(prefix+"SKIP_LOVE"+suffix, this, baseIndex+11);        // For compatibility
    keywords.add(prefix+"FIXED_ELLIPTICITY_FILE_NAME"+suffix, this, baseIndex+12);
    keywords.add(prefix+"MINIMUM_DISTANCE"+suffix, this, baseIndex+13);
    keywords.add(prefix+"MAXIMUM_DISTANCE"+suffix, this, baseIndex+14);
    keywords.add(prefix+"DO_DAMPING_FACTOR"+suffix, this, baseIndex+15); // For compatibility
    keywords.add(prefix+"SOURCE_GRID_STEP"+suffix, this, baseIndex+22);
    keywords.add(prefix+"SOURCE_GRID_SIZE"+suffix, this, baseIndex+23);
    keywords.add(prefix+"SENSOR_ORIENTATION_ERRORS"+suffix, this, baseIndex+30);
    keywords.add(prefix+"SENSOR_POSITION_ERRORS"+suffix, this, baseIndex+31);
    keywords.add(prefix+"VELOCITY_RELATIVE_PRECISION"+suffix, this, baseIndex+32);
    keywords.add(prefix+"ELLIPTICITY_ABSOLUTE_PRECISION"+suffix, this, baseIndex+33);
    keywords.add(prefix+"SENSOR_ORIENTATION_APERTURE_ANGLE"+suffix, this, baseIndex+36);
    keywords.add(prefix+"SENSOR_ORIENTATION_MAXIMUM_RADIUS"+suffix, this, baseIndex+37);
    keywords.add(prefix+"SENSOR_ORIENTATION_MAXIMUM_CORRECTION"+suffix, this, baseIndex+34);
    keywords.add(prefix+"SENSOR_ORIENTATION_MAXIMUM_ITERATIONS"+suffix, this, baseIndex+35);
    keywords.add(prefix+"BRIGHT_SPOTS"+suffix, this, baseIndex+38);
    keywords.add(prefix+"ELLIPTICITY_CORRECTION"+suffix, this, baseIndex+39);
  }

  QString FKParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+=ArrayParameters::toString(prefix, suffix);
    log+="#\n"
         "#\n"
         "#     FK method\n"
         "#\n"
         "#\n";
    log+="# Process types:\n"
         "#  [Comp is the required components]\n"
         "#  Keyword                  Comp Comments\n"
         "#  Conventional             Z    Conventional FK processing.\n"
         "#  ConventionalRadial       EN   Conventional FK processing for radial projections.\n"
         "#  ConventionalTransverse   EN   Conventional FK processing for transverse projections.\n"
         "#  ConventionalRayleigh     ENZ  Conventional FK processing for radial projections.\n"
         "#  Capon                    Z    High resolution FK processing (Capon, 1969).\n"
         "#  CaponRadial              EN   High resolution FK processing (Capon, 1969) for radial projections.\n"
         "#  CaponTransverse          EN   High resolution FK processing (Capon, 1969) for transverse projections.\n"
         "#  RTBF                     ENZ  According to Wathelet et al (2018).\n"
         "#                                Cross spectrum made of radial projections and vertical.\n"
         "#                                Product of radial and vertical ellipticity steering.\n"
         "#  ARTBF                    ENZ  RTBF with all-component ellipticity steering.\n"
         "#                                Cross spectrum made of radial projections and vertical.\n"
         "#  PoggiVertical            ENZ  According Poggi et al. (2010)\n"
         "#                                k picked from vertical processing.\n"
         "#  PoggiRadial              ENZ  According Poggi et al. (2010)\n"
         "#                                k picked from radial processing.\n"
         "#  ARDS                     ENZ  Rayleigh Direct Steering with All-component ellipticity steerting.\n"
         "#                                Cross spectrum made of raw components E, N and Z.\n"
         "#                                Radial projections included in the steering matrix.\n"
         "#  LDS3                     ENZ  Love Direct Steering.\n"
         "#                                Cross spectrum made of raw components E, N and Z.\n"
         "#                                Transverse projections included in the steering matrix.\n"
         "#  LDS2                     EN   Love Direct Steering.\n"
         "#                                Cross spectrum made of raw components E and N.\n"
         "#                                Transverse projections included in the steering matrix.\n"
         "#  Experimental modes:\n"
         "#  RTBFFixedEll             ENZ  Same as RTBF but ellipticity is fixed.\n"
         "#                                FIXED_ELLIPTICITY_FILE_NAME must be provided.\n"
         "#  ActiveConventional       Z    Conventional FK processing for active sources.\n"
         "#  ActiveRTBF               ENZ  RTBF for active sources\n"
         "#                                Cross spectrum made of radial and transverse projections.\n";
    log+=prefix+"PROCESS_TYPE"+suffix+"="+convertProcessType(_processType)+"\n";
    switch(_processType) {
    case RTBF:
    case ARTBF:
    case RTBFFixedEll:
    case PoggiVertical:
    case PoggiRadial:
    case ConventionalRadial:
    case ConventionalTransverse:
    case ConventionalRayleigh:
    case CaponRadial:
    case CaponTransverse:
      log+="# Number of steps for the computation of radial/transverse projections\n";
      log+=prefix+"ROTATE_STEP_COUNT"+suffix+"="+QString::number(_rotateStepCount)+"\n";
      break;
    default:
      break;
    }
    log+="# Correction can be: None, OneComponent, ThreeComponent or Mixed\n";
    log+=prefix+"ELLIPTICITY_CORRECTION"+suffix+"="+convertEllipticityCorrection(_ellipticityCorrection)+"\n";
    log+=prefix+"DAMPING_FACTOR"+suffix+"="+QString::number(_damping)+"\n";
    log+="# Iterative bright spot removal: the cross-spectrum for the best peak\n"
         "# is reconstructed and removed from the cross-spectral matrix. Another best\n"
         "# peak is searched with the modified cross-spectral matrix. The process is\n"
         "# repeated until removing the expected number of bright spots.\n";
    log+=prefix+"BRIGHT_SPOTS"+suffix+"="+QString::number(_brightSpots)+"\n";
    log+="# If provided and PROCESS_TYPE==RTBF, ARTBF or ARDS, the ellipticity is forced to the provided curve.\n"
         "# The file must contain two columns: frequency and signed ellipticity.\n"
         "# Provided sampling must not necessarily match the processing sampling frequency, linear interpolation is used.\n"
         "# Better for precision if the two sampling match.\n"
         "# To generate a synthetic curve: gpell M2.1.model -one-mode -R 1 -min 0.5 -max 50 -n 187 > curve.txt\n";
    log+=prefix+"FIXED_ELLIPTICITY_FILE_NAME"+suffix+"="+_fixedEllipticityFileName+"\n";
    log+="# Minimum distance between source and receiver (for active source only)\n";
    log+=prefix+"MINIMUM_DISTANCE"+suffix+"="+QString::number(_minimumDistance)+"\n";
    log+="# Maximum distance between source and receiver (for active source only)\n";
    log+=prefix+"MAXIMUM_DISTANCE"+suffix+"="+QString::number(_maximumDistance)+"\n";
    log+=prefix+"SOURCE_GRID_STEP"+suffix+"="+QString::number(_sourceGridStep)+"\n";
    log+=prefix+"SOURCE_GRID_SIZE"+suffix+"="+QString::number(_sourceGridSize)+"\n";
    log+="# Introduces errors in sensor orientation to test the robustness of methods.\n"
         "# Provides a list of names and values like: STATION_NAME1:error,STATION_NAME2:error,... in degrees.\n"
         "# To set random perturbations to each sensor lower than a threshold value, use '@RANDOM:error' keyword.\n"
         "# To set a fixed value for all station use '@ALL:error'.\n";
    log+=prefix+"SENSOR_ORIENTATION_ERRORS"+suffix+" (deg)="+sensorOrientationErrorsToString()+"\n";
    log+="# Introduces errors in sensor position to test the robustness of methods.\n"
         "# Provides a list of names and values like: STATION_NAME1:dx dy,STATION_NAME2:dx dy,....\n"
         "# To set random perturbations to each sensor lower than a threshold value, use '@RANDOM:dx dy' keyword.\n"
         "# To set a fixed value for all station use '@ALL:dx dy'.\n";
    log+=prefix+"SENSOR_POSITION_ERRORS"+suffix+" (m)="+sensorPositionErrorsToString()+"\n";
    for(QList<QStringList>::const_iterator it=_arrays.begin(); it!=_arrays.end(); it++) {
      log+=prefix+"ARRAY"+suffix+"="+it->join(',')+"\n";
    }
    log+="# Experimental join processing of several arrays\n";
    log+="# Several ARRAY can be defined with a list of station names\n";
    for(QList<QStringList>::const_iterator it=_arrays.begin(); it!=_arrays.end(); it++) {
      log+=prefix+"ARRAY"+suffix+"="+it->join(',')+"\n";
    }
    log+="#\n"
         "#\n"
         "#     Wavenumber grid\n"
         "#\n"
         "#\n";
    log+="# Wavenumber fine gridding used as a cache for the FK maps\n";
    log+=prefix+"CACHE_GRID_STEP"+suffix+" (rad/m)="+QString::number(_cacheGridStep)+"\n";
    log+="# If CACHE_GRID_STEP is null, GRID_STEP is computed from K_MIN*CACHE_GRID_STEP_FACTOR.\n";
    log+=prefix+"CACHE_GRID_STEP_FACTOR"+suffix+"="+QString::number(_cacheGridStepFactor)+"\n";

    log+="# Wavenumber coarse gridding used for searching maxima of the FK maps\n";
    log+=prefix+"GRID_STEP"+suffix+" (rad/m)="+QString::number(_gridStep)+"\n";
    log+="# If GRID_STEP is null, GRID_STEP is computed from K_MIN*GRID_STEP_FACTOR.\n";
    log+=prefix+"GRID_STEP_FACTOR"+suffix+"="+QString::number(_gridStepFactor)+"\n";
    log+=prefix+"GRID_SIZE"+suffix+" (rad/m)="+QString::number(_gridSize)+"\n";
    log+="# If GRID_SIZE is null, GRID_SIZE is computed from K_MAX*GRID_SIZE_FACTOR.\n";
    log+=prefix+"GRID_SIZE_FACTOR"+suffix+"="+QString::number(_gridSizeFactor)+"\n";
    log+="# Effective GRID_STEP is "+QString::number(gridStep())+".\n";
    log+="# Effective GRID_SIZE is "+QString::number(gridSize())+".\n";

    log+="# Minimum velocity of the searched maxima of the FK map\n";
    log+=prefix+"MIN_V"+suffix+" (m/s)="+QString::number(1.0/_smax)+"\n";
    log+="# Maximum velocity of the searched maxima of the FK map\n";
    log+=prefix+"MAX_V"+suffix+" (m/s)="+QString::number(1.0/_smin)+"\n";

    log+="# Minimum azimuth of the searched maxima of the FK map (clockwise from North)\n";
    log+=prefix+"MIN_AZIMUTH"+suffix+" (deg.)="+QString::number(_minAzimuth)+"\n";
    log+="# Maximum azimith of the searched maxima of the FK map (clockwise from North)\n";
    log+=prefix+"MAX_AZIMUTH"+suffix+" (deg.)="+QString::number(_maxAzimuth)+"\n";

    log+="# Theoretical Kmin and Kmax computed from array geometry\n";
    log+="# Used only for post-processing\n";
    log+=prefix+"K_MIN"+suffix+" (rad/m)="+QString::number(_kmin)+"\n";
    log+=prefix+"K_MAX"+suffix+" (rad/m)="+QString::number(_kmax)+"\n";
    log+="#\n"
         "#\n"
         "#     Peak picking\n"
         "#\n"
         "#\n";
    log+="# Inversion method used for getting FK peaks: Gradient or RefinedGrid\n";
    log+=prefix+"INVERSION_METHOD"+suffix+"="+convertInversionMethod(_inversionMethod)+"\n";
    log+="# When refining a peak these two precision thresholds define the end conditions.\n";
    log+=prefix+"VELOCITY_RELATIVE_PRECISION"+suffix+"="+QString::number(velocityRelativePrecision())+"\n";
    log+=prefix+"ELLIPTICITY_ABSOLUTE_PRECISION"+suffix+" (deg)="+QString::number(Angle::radiansToDegrees(ellipticityAbsolutePrecision()))+"\n";
    log+=prefix+"N_MAXIMA"+suffix+"="+QString::number(_maxPeakCount)+"\n";
    log+=prefix+"N_MAXIMA_FACTOR"+suffix+"="+QString::number(_maxPeakCountFactor)+"\n";
    log+=prefix+"ABSOLUTE_THRESHOLD"+suffix+"="+QString::number(_absoluteThreshold)+"\n";
    log+=prefix+"RELATIVE_THRESHOLD"+suffix+" (%)="+QString::number(_relativeThreshold*100.0)+"\n";
    log+=prefix+"SAVE_UNDEFINED_ELLIPTICITIES"+suffix+"="+(_saveUndefinedEllipticities ? "y" : "n")+"\n";
    log+=prefix+"SORT_RESULT_VALUES"+suffix+"="+(_sortResultValues ? "y" : "n")+"\n";
    log+=prefix+"EXPORT_ALL_FK_GRIDS"+suffix+"="+(_exportAllFKGrids ? "y" : "n")+"\n";
    log+="#\n"
         "#\n"
         "#     Sensor orientation (only for PROCESS_TYPE=SensorOrientation)\n"
         "#\n"
         "#\n";
    log+=prefix+"SENSOR_ORIENTATION_APERTURE_ANGLE"+suffix+"="+QString::number(Angle::radiansToDegrees(_sensorOrientationApertureAngle))+"\n";
    log+=prefix+"SENSOR_ORIENTATION_MAXIMUM_RADIUS"+suffix+"="+QString::number(_sensorOrientationMaximumRadius)+"\n";
    log+=prefix+"SENSOR_ORIENTATION_MAXIMUM_CORRECTION"+suffix+"="+QString::number(Angle::radiansToDegrees(_sensorOrientationMaximumCorrection))+"\n";
    log+=prefix+"SENSOR_ORIENTATION_MAXIMUM_ITERATIONS"+suffix+"="+QString::number(_sensorOrientationMaximumIterations)+"\n";
    return log;
  }

  bool FKParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    bool ok=true;
    switch(index-ArrayParameters::keywordCount()) {
    case 19:
      setInversionMethod(convertInversionMethod(value, ok));
      return ok;
    case 20:
      _cacheGridStep=value.toDouble(&ok);
      return ok;
    case 21:
      _cacheGridStepFactor=value.toDouble(&ok);
      return ok;
    case 0:
      if(version()<1) {
        _gridStep=value.toDouble(&ok); // Old release were saving grid_step instead of kmin
        _kmin=0.0;                     // Force automatic computation from array geometry
      } else {
        _kmin=value.toDouble(&ok);
      }
      return ok;
    case 1:
      if(version()<1) {
        _gridSize=value.toDouble(&ok); // Old release were saving grid_size instead of kmax
        _kmax=0.0;                     // Force automatic computation from array geometry
      } else {
        _kmax=value.toDouble(&ok);
      }
      return ok;
    case 16:
      _gridStep=value.toDouble(&ok);
      return ok;
    case 17:
      _gridSize=value.toDouble(&ok);
      return ok;
    case 18:
      _gridStepFactor=value.toDouble(&ok);
      return ok;
    case 28:
      _gridSizeFactor=value.toDouble(&ok);
      return ok;
    case 24:
      _smin=1.0/value.toDouble(&ok);
      return ok;
    case 2:
      _smax=1.0/value.toDouble(&ok);
      return ok;
    case 25:
      _minAzimuth=value.toDouble(&ok);
      return ok;
    case 26:
      _maxAzimuth=value.toDouble(&ok);
      return ok;
    case 3:
      _maxPeakCount=value.toInt(&ok);
      return ok;
    case 4:
      _absoluteThreshold=value.toDouble(&ok);
      return ok;
    case 5:
      _relativeThreshold=value.toDouble(&ok)*0.01;
      return ok;
    case 6:
      _exportAllFKGrids=(value=="y");
      return true;
    case 27:
      _saveUndefinedEllipticities=(value=="y");
      return ok;
    case 7:
      if(_damping>=0.0) {
        _damping=value.toDouble(&ok);
      }
      return ok;
    case 8:
      _rotateStepCount=value.toInt(&ok);
      return ok;
    case 9:
      _processType=convertProcessType(value, ok);
      return ok;
    case 10:
      _arrays.append(value.split(','));
      return true;
    case 11:  // For compatibility
      obsoleteKeyword(keywords, 11);
      return true;
    case 12:
      _fixedEllipticityFileName=value;
      return true;
    case 13:
      _minimumDistance=value.toDouble(&ok);
      return ok;
    case 14:
      _maximumDistance=value.toDouble(&ok);
      return ok;
    case 15: // For compatibility
      obsoleteKeyword(keywords, 15);
      if(value=="n") {
        _damping=-1.0;
      }
      return true;
    case 22:
      _sourceGridStep=value.toDouble(&ok);
      return ok;
    case 23:
      _sourceGridSize=value.toDouble(&ok);
      return ok;
    case 29:
      _sortResultValues=(value=="y");
      return ok;
    case 30:
      sensorOrientationErrorsFromString(value, ok);
      return ok;
    case 31:
      sensorPositionErrorsFromString(value, ok);
      return ok;
    case 32:
      _velocityRelativePrecision=value.toDouble(&ok);
      return ok;
    case 33:
      _ellipticityAbsolutePrecision=Angle::degreesToRadians(value.toDouble(&ok));
      return ok;
    case 34:
      _sensorOrientationMaximumCorrection=Angle::degreesToRadians(value.toDouble(&ok));
      return ok;
    case 35:
      _sensorOrientationMaximumIterations=value.toInt(&ok);
      return ok && _sensorOrientationMaximumIterations>1;
    case 36:
      _sensorOrientationApertureAngle=Angle::degreesToRadians(value.toDouble(&ok));
      return ok;
    case 37:
      _sensorOrientationMaximumRadius=value.toDouble(&ok);
      return ok && _sensorOrientationMaximumRadius>=0.0;
    case 38:
      _brightSpots=value.toInt(&ok);
      return ok;
    case 39:
      _ellipticityCorrection=convertEllipticityCorrection(value, ok);
      return ok;
    case 40:
      _maxPeakCountFactor=value.toDouble(&ok);
      return ok;
    default:
      break;
    }
    return ArrayParameters::setValue(index, value, unit, keywords);
  }

  QString FKParameters::sensorOrientationErrorsToString() const
  {
    if(_sensorOrientationErrors.isEmpty()) {
      return "@ALL:0";
    }
    QMap<QString, double>::const_iterator it;
    QString str;
    if(_sensorPositionErrors.count()>1) {
      str+="\\\n  ";
    }
    for(it=_sensorOrientationErrors.begin(); it!=_sensorOrientationErrors.end(); it++) {
      if(!str.isEmpty()) {
        str+=", \\\n";
      }
      str+=it.key();
      str+=":";
      str+=QString::number(it.value(), 'g', 20);
    }
    return str;
  }

  void FKParameters::sensorOrientationErrorsFromString(const QString& values, bool& ok)
  {
    _sensorOrientationErrors.clear();
    QStringList list=values.split(",");
    for(QStringList::const_iterator it=list.begin(); it!=list.end(); it++) {
      const QString& item=*it;
      if(!item.trimmed().isEmpty()) {
        _sensorOrientationErrors.insert(item.section(':', 0, 0).trimmed(), item.section(':', 1, 1).toDouble(&ok));
        if(!ok) {
          _sensorOrientationErrors.clear();
          break;
        }
      }
    }
  }

  double FKParameters::sensorOrientationError(const QString& name) const
  {
     QMap<QString, double>::const_iterator it=_sensorOrientationErrors.find(name);
     if(it==_sensorOrientationErrors.end()) {
       return 0.0;
     } else {
       return it.value();
     }
  }

  QString FKParameters::sensorPositionErrorsToString() const
  {
    if(_sensorPositionErrors.isEmpty()) {
      return "@ALL:0 0";
    }
    QMap<QString, Point2D>::const_iterator it;
    QString str;
    if(_sensorPositionErrors.count()>1) {
      str+="\\\n  ";
    }
    for(it=_sensorPositionErrors.begin(); it!=_sensorPositionErrors.end(); it++) {
      if(!str.isEmpty()) {
        str+=", \\\n  ";
      }
      str+=it.key();
      str+=":";
      str+=it.value().toString('g', 20);
    }
    return str;

  }

  void FKParameters::sensorPositionErrorsFromString(const QString& values, bool& ok)
  {
    _sensorPositionErrors.clear();
    QStringList list=values.split(",");
    for(QStringList::const_iterator it=list.begin(); it!=list.end(); it++) {
      const QString& item=*it;
      if(!item.trimmed().isEmpty()) {
        Point2D p;
        ok=p.fromString(item.section(':', 1, 1));
        if(ok) {
          _sensorPositionErrors.insert(item.section(':', 0, 0).trimmed(), p);
        } else {
          _sensorPositionErrors.clear();
          break;
        }
      }
    }
  }

  Point2D FKParameters::sensorPositionError(const QString& name) const
  {
     QMap<QString, Point2D>::const_iterator it=_sensorPositionErrors.find(name);
     if(it==_sensorPositionErrors.end()) {
       return Point2D();
     } else {
       return it.value();
     }
  }

  ENUM_AS_STRING_BEGIN(FKParameters, ProcessType)
  ENUM_AS_STRING_DATA_23(ActiveCapon, ActiveConventional, ActiveConventionalRayleigh,
                         ActiveRTBF, ARDS, ARTBF, Capon, CaponRadial, CaponTransverse,
                         Conventional, ConventionalRadial, ConventionalRayleigh,
                         ConventionalTransverse, LDS2, LDS3, Omni, PoggiRadial,
                         PoggiVertical, RDSSingle, RTBF, RTBFFixedEll,
                         SensorOrientation, Undefined);
  ENUM_AS_STRING_SYNONYM("Projections", RTBF);
  ENUM_AS_STRING_DEFAULT_VALUE(Undefined);
  ENUM_AS_STRING_END

  ENUM_AS_STRING_BEGIN(FKParameters, InversionMethod)
  ENUM_AS_STRING_DATA_2(Gradient, RefinedGrid);
  ENUM_AS_STRING_END

  ENUM_AS_STRING_BEGIN(FKParameters, EllipticityCorrection)
  ENUM_AS_STRING_DATA_2(None, ThreeComponent);
  ENUM_AS_STRING_END

  QStringList FKParameters::processTypes()
  {
    QStringList l;
    for(int i=1; i<=RDSSingle; i++) {
      l.append(convertProcessType(static_cast<ProcessType>(i)));
    }
    return l;
  }

  bool FKParameters::isActiveProcessType() const
  {
    switch(_processType) {
    case Undefined:
    FKPARAMETERS_PASSIVE_PROCESS_TYPES
      return false;
    FKPARAMETERS_ACTIVE_PROCESS_TYPES
      break;
    }
    return true;
  }

  QString FKParameters::outputName(const QString& groupName) const
  {
    if(ArrayParameters::outputBaseName().endsWith(".max")) {
      return ArrayParameters::outputBaseName();
    } else if(ArrayParameters::outputBaseName().isEmpty()) {
      return QString(convertProcessType(_processType)).toLower()+"-"+groupName+".max";
    } else {
      QString baseName;
      if(!ArrayParameters::outputBaseName().endsWith("-") &&
         !ArrayParameters::outputBaseName().endsWith("_")) {
        baseName=ArrayParameters::outputBaseName()+"-";
      } else {
        baseName=ArrayParameters::outputBaseName();
      }
      QString p=convertProcessType(_processType);
      p=p.toLower();
      if(baseName.contains(p)) {
        return baseName+groupName+".max";
      } else {
        return baseName+p+"-"+groupName+".max";
      }
    }
  }

} // namespace ArrayCore
