/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-12-02
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreWave.h>

#include "FKCrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class FKCrossSpectrum FKCrossSpectrum.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  FKCrossSpectrum::FKCrossSpectrum(const ArraySelection * array,
                                   const FKParameters * param)
    : ArrayCrossSpectrum(array, param)
  {
    TRACE;
    _rotateStepCount=param->rotateStepCount();
    if(_rotateStepCount<1) {
      _rotateStepCount=1;
    }
    _rotateStep=2.0*M_PI/static_cast<double>(_rotateStepCount);
    _invRotateStep=1.0/_rotateStep;

    _rotatedMatrices=nullptr;
    _rotatedRadialMatrices=nullptr;
    _verticalMatrix=nullptr;
    _horizontalMatrix=nullptr;
    _source=nullptr;
    resize();
  }

  /*!
    Description of destructor still missing
  */
  FKCrossSpectrum::~FKCrossSpectrum()
  {
    TRACE;
    delete [] _rotatedMatrices;
    delete [] _rotatedRadialMatrices;
    delete _verticalMatrix;
    delete _horizontalMatrix;
  }

  /*!
    Must be run whenever the process type changes in parameters.
  */
  void FKCrossSpectrum::resize()
  {
    // Clear rotated matrices and one-component splits
    delete [] _rotatedMatrices;
    delete [] _rotatedRadialMatrices;
    delete _verticalMatrix;
    delete _horizontalMatrix;
    delete _matrix;
    _rotatedMatrices=nullptr;
    _rotatedRadialMatrices=nullptr;
    _verticalMatrix=nullptr;
    _horizontalMatrix=nullptr;
    _matrix=nullptr;

    int stationCount=_array.count();
    FKParameters::ProcessType processType;
    processType=static_cast<const FKParameters *>(_parameters)->processType();
    switch(processType) {
    case FKParameters::Undefined:
    case FKParameters::Conventional:
    case FKParameters::Capon:
    case FKParameters::Omni:
    case FKParameters::ActiveConventional:
    case FKParameters::ActiveCapon:
      _matrix=new ComplexMatrix(stationCount);
      break;
    case FKParameters::PoggiVertical:
    case FKParameters::PoggiRadial:
    case FKParameters::CaponRadial:
    case FKParameters::CaponTransverse:
    case FKParameters::ConventionalRadial:
    case FKParameters::ConventionalTransverse:
      _rotatedMatrices=new ComplexMatrix[_rotateStepCount];
      for(int i=0; i<_rotateStepCount; i++) {
        _rotatedMatrices[i].resize(stationCount);
      }
      break;
    case FKParameters::ConventionalRayleigh:
    case FKParameters::RTBF:
    case FKParameters::ARTBF:
    case FKParameters::RTBFFixedEll:
      _rotatedMatrices=new ComplexMatrix[_rotateStepCount];
      _rotatedRadialMatrices=new ComplexMatrix[_rotateStepCount];
      for(int i=0; i<_rotateStepCount; i++) {
        _rotatedMatrices[i].resize(2*stationCount);
        _rotatedRadialMatrices[i].resize(stationCount);
      }
      _verticalMatrix=new ComplexMatrix(stationCount);
      break;
    case FKParameters::LDS2:
    case FKParameters::ActiveRTBF:
    case FKParameters::ActiveConventionalRayleigh:
      _matrix=new ComplexMatrix(2*stationCount);
      break;
    case FKParameters::ARDS:
      _matrix=new ComplexMatrix(3*stationCount);
      _verticalMatrix=new ComplexMatrix(stationCount);
      _horizontalMatrix=new ComplexMatrix(2*stationCount);
      break;
    case FKParameters::LDS3:
    case FKParameters::SensorOrientation:
    case FKParameters::RDSSingle:
      _matrix=new ComplexMatrix(3*stationCount);
      break;
    }
  }

  /*!
    Get the average source location from all active sources during blockset.
    Stations are then selected according to min and max distance parameters.
  */
  Point FKCrossSpectrum::setActiveSource(const VectorList<int>& blocks, bool& ok)
  {
    const FKParameters * param=parameters();
    Point src;
    if(_source) {  // Only one source is forced
      src=_source->position();
    } else {
      const SeismicEventTable * activeSources;
      activeSources=_array.array()->first()->originals(0).database()->seismicEvents();

      // Get the list of all events recorded by all receivers
      QMap<const SeismicEvent *, int> eventList;
      QMap<const SeismicEvent *, int>::iterator eventIt;
      QList<const SeismicEvent *> srcList;
      // Total number of receivers independently of the current selections
      int nRec=_array.array()->count();
      for(int iRec=0; iRec<nRec; iRec++) {
        for(VectorList<int>::const_iterator it=blocks.begin(); it!=blocks.end(); it++) {
          const TimeRange& win=_timeWindows->list().at(*it);
          srcList=activeSources->events(win, _array.array()->at(iRec)->coordinates(),
                                        param->minimumSlowness(), param->maximumSlowness());
          for(QList<const SeismicEvent *>::iterator it=srcList.begin(); it!=srcList.end(); it++) {
            eventIt=eventList.find(*it);
            if(eventIt!=eventList.end()) {
              eventIt.value()++;
            } else {
              eventList.insert(*it, 1);
            }
          }
        }
      }
      if(eventList.isEmpty()) {
        ok=false;
        App::log(tr("  No source found, aborting.\n"));
        return Point();
      }

      // Check that the sources are not too far away
      eventIt=eventList.begin();
      Point ref=eventIt.key()->position();
      for(eventIt++; eventIt!=eventList.end(); eventIt++) {
        if(eventIt.key()->position().distanceTo(ref)>1.0) {
          ok=false;
          App::log(tr("Skip blockset because it contains a mix of too distant sources.\n"));
          return Point();
        }
      }

      // Average the position of sources recorded by at least half of the receivers
      int nSrc=0;
      nRec/=2;
      for(eventIt=eventList.begin(); eventIt!=eventList.end(); eventIt++) {
        if(eventIt.value()>nRec) {
          src+=eventIt.key()->position();
          nSrc++;
        } else {
          App::log(tr("Event %1 probably not recorded by all receivers, skipped.\n").arg(eventIt.key()->name()));
        }
      }
      if(nSrc==0) {
        ok=false;
        App::log(tr("  No source found, aborting.\n"));
        return Point();
      } else {
        App::log(tr("  Found %1 active sources.\n").arg(nSrc));
      }
      src/=static_cast<double>(nSrc);
    }
    ok=selectStations(src);
    return src;
  }

  bool FKCrossSpectrum::selectStations(const Point& src)
  {
    const FKParameters * param=parameters();
    _array.clear();
    APP_LOG(1, tr("Checking distances from source (%1)...\n").arg(src.toString('f', 2)))
    int n=_array.count();
    _array.select(src, param->minimumDistance(), param->maximumDistance());
    if(_array.count()<n) {
      APP_LOG(1, tr("Selected %1 stations:\n%2")
                       .arg(_array.count())
                       .arg(_array.toString(ArraySelection::Absolute)));
    }
    if(!parameters()->selectStations(_array)) {
      return false;
    }

    if(_array.count()<2) {
      App::log(tr("Less than 2 stations selected, aborting\n"));
      return false;
    } else {
      setStations(&_array);
      resize();
      return true;
    }
  }

  bool FKCrossSpectrum::calculate(const VectorList<int>& blocks, AbstractFKFunction * function)
  {
    TRACE;
#ifdef CROSS_SPECTRUM_TIMING
    QTime chrono;
    int t1, t2, t3, t4, t5, tBeforeLock;
    int tLock=0;
    chrono.start();
#endif
    // For active process, estimate the average source location
    if(!function->setSource(blocks)) {
      return false;
    }

#ifdef CROSS_SPECTRUM_TIMING
    t1=chrono.elapsed();
#endif
    function->resetCrossSpectrum();

#ifdef CROSS_SPECTRUM_TIMING
    t2=chrono.elapsed();
#endif
    for(VectorList<int>::const_iterator it=blocks.begin(); it!=blocks.end(); it++) {
#ifdef CROSS_SPECTRUM_TIMING
    tBeforeLock=chrono.elapsed();
#endif
      if(lockTimeWindow(_timeWindows->list().at(*it))) {
#ifdef CROSS_SPECTRUM_TIMING
    tLock+=chrono.elapsed()-tBeforeLock;
#endif
        function->addCrossSpectrum();
        unlockTimeWindow();
      } else {
        return false;
      }
    }
#ifdef CROSS_SPECTRUM_TIMING
    t3=chrono.elapsed();
#endif

    function->meanCrossSpectrum(blocks.count());
    if(!function->setFixedEllipticity(_timeWindows->frequency().center())) {
      return false;
    }

    // Experimental, mask built once with station name lists (1 or 0)
    if(!_arrayMask.isEmpty()) {
      *_matrix=_matrix->dotMultiply(_arrayMask);
      APP_LOG(4, _matrix->toOctaveString()+"\n")
    }
#ifdef CROSS_SPECTRUM_TIMING
    t4=chrono.elapsed();
#endif

    // Invert average cross spectra
    // Using Cholesky does not change speed for Vertical
    // Using Cholesky does change speed for 3-Component: from ~11' to 3'20" for a typical case.
    if(!function->invertCrossSpectrum()) {
      return false;
    }
#ifdef CROSS_SPECTRUM_TIMING
    t5=chrono.elapsed();
    App::log(3, tr("Cross spectrum computation took %1 ms (block %2 ms, reset %3 ms, lock %7 ms, add %4 ms, mean %5 ms, inv %6 ms)\n")
             .arg(t5).arg(t1).arg(t2-t1).arg(t3-t2).arg(t4-t3).arg(t5-t4).arg(tLock));
#endif
    return true;
  }

  void FKCrossSpectrum::resetRotatedMatrices()
  {
    for(int i=0; i<_rotateStepCount; i++) {
      _rotatedMatrices[i].zero();
    }
  }

  void FKCrossSpectrum::addRotatedRadial(const Complex * sensorRotations)
  {
    for(int i=0; i<_rotateStepCount; i++) {
      addRotatedRadial(i, sensorRotations);
    }
  }

  void FKCrossSpectrum::addRotatedTransverse(const Complex * sensorRotations)
  {
    for(int i=0; i<_rotateStepCount; i++) {
      addRotatedTransverse(i, sensorRotations);
    }
  }

  void FKCrossSpectrum::addRotatedRayleigh(const Complex * sensorRotations)
  {
    for(int i=0; i<_rotateStepCount; i++) {
      addRotatedRayleigh(i, sensorRotations);
    }
  }

  void FKCrossSpectrum::meanRotatedMatrices(int nBlocks)
  {
    double invBlockCount=1.0/static_cast<double>(nBlocks);
    for(int i=0; i<_rotateStepCount; i++) {
      _rotatedMatrices[i]*=invBlockCount;
    }
  }

  bool FKCrossSpectrum::invert()
  {
    if(_horizontalMatrix && _verticalMatrix) {
      int n=_array.count();
      int n2=2*n;
      int n21=2*n-1;
      int n31=3*n-1;
      *_verticalMatrix=_matrix->subMatrix(n2, n2, n31, n31);
      if(!invert(*_verticalMatrix)) {
        App::log(tr("Cannot invert cross spectrum for vertical sub matrix\n"));
        return false;
      }
      *_horizontalMatrix=_matrix->subMatrix(0, 0, n21, n21);
      if(!invert(*_horizontalMatrix)) {
        App::log(tr("Cannot invert cross spectrum for horizontal sub matrix\n"));
        return false;
      }
    }
    if(!invert(*_matrix)) {
      App::log(tr("Cannot invert cross spectrum\n"));
      return false;
    }
    return true;
  }

  bool FKCrossSpectrum::invertRotatedMatrices()
  {
    if(_rotatedRadialMatrices && _verticalMatrix) {
      int n=_array.count();
      int n1=n-1;
      int n2=2*n-1;
      *_verticalMatrix=_rotatedMatrices[0].subMatrix(n, n, n2, n2);
      if(!invert(*_verticalMatrix)) {
        App::log(tr("Cannot invert cross spectrum for vertical sub matrix\n"));
        return false;
      }
      for(int i=0; i<_rotateStepCount; i++) {
        _rotatedRadialMatrices[i]=_rotatedMatrices[i].subMatrix(0, 0, n1, n1);
        if(!invert(_rotatedRadialMatrices[i])) {
          App::log(tr("Cannot invert cross spectrum for radial sub matrix in direction %1 deg\n")
                   .arg(Angle::radiansToDegrees(i*_rotateStep)));
          return false;
        }
      }
    }
    for(int i=0; i<_rotateStepCount; i++) {
      if(!invert(_rotatedMatrices[i])) {
        App::log(tr("Cannot invert cross spectrum for direction %1 deg\n")
                 .arg(Angle::radiansToDegrees(i*_rotateStep)));
        return false;
      }
    }
    return true;
  }

  void FKCrossSpectrum::addRotatedRadial(int rotationIndex, const Complex * sensorRotations)
  {
    int stationCount=_stations.count();

    const ArrayStationSignals * stat;
    ComplexMatrix& covmat=_rotatedMatrices[rotationIndex];
    ComplexMatrix sig(stationCount, 1);
    Angle rotation;
    rotation.setRadians(_rotateStep*rotationIndex);
    Complex rsv;
    // Calculate orientation of each station
    // TODO: take that value from station data itself
    /*Angle * orientations=new Angle[stationCount];
    for(int is=0; is<stationCount; is++) {
      Point2D p=_stations.at(is)->originalSignals()->coordinates();
      p-=Point2D(2000, 2000);
      Angle& orientation=orientations[is];
      orientation.set(p.x(), p.y());
      orientation.chSign();
    }*/
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(is, 0);
        const Complex& rot=sensorRotations[is];
        rsv=stat->spectrumValue(_eastIndex)*rot.re();
        rsv-=stat->spectrumValue(_northIndex)*rot.im();
        s=rotation.cos()*rsv;   // Rotated east
        rsv=stat->spectrumValue(_northIndex)*rot.re();
        rsv+=stat->spectrumValue(_eastIndex)*rot.im();
        s+=rotation.sin()*rsv;  // Rotated north
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      //Complex north, east;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(is, 0);
          const Complex& rot=sensorRotations[is];
          rsv=stat->spectrumValue(_eastIndex, iFreq)*rot.re();
          rsv-=stat->spectrumValue(_northIndex, iFreq)*rot.im();
          s=rotation.cos()*rsv;   // Rotated east
          rsv=stat->spectrumValue(_northIndex, iFreq)*rot.re();
          rsv+=stat->spectrumValue(_eastIndex, iFreq)*rot.im();
          s+=rotation.sin()*rsv;  // Rotated north
          /* Test optical fiber array
          north=stat->spectrumValue(_northIndex, iFreq);
          if(is%5==0) {
            east=stat->spectrumValue(_eastIndex, iFreq);
          } else {
            east.set(0.0, 0.0);
          }
          s=rotation.sin()*(north*orientations[is].cos()+east*orientations[is].sin());
          s+=rotation.cos()*(east*orientations[is].cos()-north*orientations[is].sin());*/
          s*=filter;
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  void FKCrossSpectrum::addRotatedTransverse(int rotationIndex, const Complex * sensorRotations)
  {
    int stationCount=_stations.count();

    // Calculate orientation of each station
    // TODO: take that value from station data itself
    /*Angle * orientations=new Angle[stationCount];
    for(int is=0; is<stationCount; is++) {
      Point2D p=_stations.at(is)->originalSignals()->coordinates();
      p-=Point2D(2000, 2000);
      Angle& orientation=orientations[is];
      orientation.set(p.x(), p.y());
      orientation.chSign();
    }*/

    const ArrayStationSignals * stat;
    ComplexMatrix& covmat=_rotatedMatrices[rotationIndex];
    ComplexMatrix sig(stationCount, 1);
    Angle rotation;
    rotation.setRadians(_rotateStep*rotationIndex);
    Complex rsv;
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(is, 0);
        const Complex& rot=sensorRotations[is];
        rsv=stat->spectrumValue(_northIndex)*rot.re();
        rsv+=stat->spectrumValue(_eastIndex)*rot.im();
        s=rotation.cos()*rsv;    // Rotated north
        rsv=stat->spectrumValue(_eastIndex)*rot.re();
        rsv-=stat->spectrumValue(_northIndex)*rot.im();
        s-=rotation.sin()*rsv;   // Rotated east
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      //Complex north, east;
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(is, 0);
          const Complex& rot=sensorRotations[is];
          rsv=stat->spectrumValue(_northIndex, iFreq)*rot.re();
          rsv+=stat->spectrumValue(_eastIndex, iFreq)*rot.im();
          s=rotation.cos()*rsv;    // Rotated north
          rsv=stat->spectrumValue(_eastIndex, iFreq)*rot.re();
          rsv-=stat->spectrumValue(_northIndex, iFreq)*rot.im();
          s-=rotation.sin()*rsv;   // Rotated east
          /* Test optical fiber array
          north=stat->spectrumValue(_northIndex, iFreq);
          if(is%5==0) {
            east=stat->spectrumValue(_eastIndex, iFreq);
          } else {
            east.set(0.0, 0.0);
          }
          s=rotation.cos()*(north*orientations[is].cos()+east*orientations[is].sin());
          s-=rotation.sin()*(east*orientations[is].cos()-north*orientations[is].sin());*/
          s*=filter;
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  void FKCrossSpectrum::addRotatedRayleigh(int rotationIndex, const Complex * sensorRotations)
  {
    int stationCount=_stations.count();

    const ArrayStationSignals * stat;
    ComplexMatrix& covmat=_rotatedMatrices[rotationIndex];
    ComplexMatrix sig(2*stationCount, 1);
    Angle rotation;
    rotation.setRadians(_rotateStep*rotationIndex);
    Complex rsv;
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(is, 0);
        const Complex& rot=sensorRotations[is];
        rsv=stat->spectrumValue(_eastIndex)*rot.re();
        rsv-=stat->spectrumValue(_northIndex)*rot.im();
        s=rotation.cos()*rsv;   // Rotated east
        rsv=stat->spectrumValue(_northIndex)*rot.re();
        rsv+=stat->spectrumValue(_eastIndex)*rot.im();
        s+=rotation.sin()*rsv;  // Rotated north
      }
      for(int is=0; is<stationCount; is++) {
        sig.at(stationCount+is, 0)=_stations.at(is)->spectrumValue(_verticalIndex);
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(is, 0);
          const Complex& rot=sensorRotations[is];
          rsv=stat->spectrumValue(_eastIndex, iFreq)*rot.re();
          rsv-=stat->spectrumValue(_northIndex, iFreq)*rot.im();
          s=rotation.cos()*rsv;   // Rotated east
          rsv=stat->spectrumValue(_northIndex, iFreq)*rot.re();
          rsv+=stat->spectrumValue(_eastIndex, iFreq)*rot.im();
          s+=rotation.sin()*rsv;  // Rotated north
          s*=filter;
        }
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(stationCount+is, 0);
          s=_stations.at(is)->spectrumValue(_verticalIndex, iFreq);
          s*=filter;
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  void FKCrossSpectrum::addThreeComponent(const Complex * sensorRotations)
  {
    int stationCount=_stations.count();
    int stationCount2=2*stationCount;

    const ArrayStationSignals * stat;
    ComplexMatrix& covmat=*_matrix;
    ComplexMatrix sig(3*stationCount, 1);
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(is, 0);
        const Complex& rot=sensorRotations[is];
         s=stat->spectrumValue(_eastIndex)*rot.re();
         s-=stat->spectrumValue(_northIndex)*rot.im();
      }
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(stationCount+is, 0);
        const Complex& rot=sensorRotations[is];
        s=stat->spectrumValue(_northIndex)*rot.re();
        s+=stat->spectrumValue(_eastIndex)*rot.im();
      }
      for(int is=0; is<stationCount; is++) {
        sig.at(stationCount2+is, 0)=_stations.at(is)->spectrumValue(_verticalIndex);
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(is, 0);
          const Complex& rot=sensorRotations[is];
          s=stat->spectrumValue(_eastIndex, iFreq)*rot.re();
          s-=stat->spectrumValue(_northIndex, iFreq)*rot.im();
          s*=filter;
        }
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(stationCount+is, 0);
          const Complex& rot=sensorRotations[is];
          s=stat->spectrumValue(_northIndex, iFreq)*rot.re();
          s+=stat->spectrumValue(_eastIndex, iFreq)*rot.im();
          s*=filter;
        }
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(stationCount2+is, 0);
          s=_stations.at(is)->spectrumValue(_verticalIndex, iFreq);
          s*=filter;
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  // Test of optical fiber arrays
  /*void FKCrossSpectrum::addHorizontalRotated(ComplexMatrix& covmat)
  {
    int stationCount=_stations.count();
    const GaussianFrequencyBand& frequencyFilter=_timeWindows->frequencyFilter();

    ComplexMatrix sig(2*stationCount, 1);

    // Calculate orientation of each station
    // TODO: take that value from station data itself
    Angle * orientations=new Angle[stationCount];
    for(int is=0; is<stationCount; is++) {
      Point2D p=_stations.at(is)->originalSignals()->coordinates();
      p-=Point2D(2000, 2000);
      Angle& orientation=orientations[is];
      orientation.set(p.x(), p.y());
      orientation.chSign();
    }

    if(_singleValueSpectrum) {
      for(int is=0; is<stationCount; is++) {
        sig.at(is, 0)=_stations.at(is)->spectrumValue(_eastIndex)*orientations[is].cos()
                     -_stations.at(is)->spectrumValue(_northIndex)*orientations[is].sin();
      }
      for(int is=0; is<stationCount; is++) {
        sig.at(stationCount+is, 0)=_stations.at(is)->spectrumValue(_northIndex)*orientations[is].cos()
                                  +_stations.at(is)->spectrumValue(_eastIndex)*orientations[is].sin();
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(is, 0);
          s=_stations.at(is)->spectrumValue(_eastIndex, iFreq)*orientations[is].cos()
           -_stations.at(is)->spectrumValue(_northIndex, iFreq)*orientations[is].sin();
          s*=filter;
        }
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(stationCount+is, 0);
          s=_stations.at(is)->spectrumValue(_northIndex, iFreq)*orientations[is].cos()
           +_stations.at(is)->spectrumValue(_eastIndex, iFreq)*orientations[is].sin();
          s*=filter;
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
    delete [] orientations;
  }*/

  /*!
    Signals are corrected by factor 1/sqrt(distance to source)
  */
  void FKCrossSpectrum::addActiveVertical(const ActiveFKSteering& geom)
  {
    int stationCount=_stations.count();

    ComplexMatrix& covmat=*_matrix;
    ComplexMatrix sig(stationCount, 1);
#ifndef COMPATIBILITY_2_5_0
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        Complex& s=sig.at(is, 0);
        s=_stations.at(is)->spectrumValue(_verticalIndex);
        s*=geom.amplitudeFactor(is);
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
#endif
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(is, 0);
          s=_stations.at(is)->spectrumValue(_verticalIndex, iFreq);
#ifdef COMPATIBILITY_2_5_0
          s*=filter;
#else
          s*=filter*geom.amplitudeFactor(is);
#endif
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
#ifndef COMPATIBILITY_2_5_0
    }
#endif
  }

  void FKCrossSpectrum::addActiveRayleigh(const ActiveFKSteering& geom)
  {
    int stationCount=_stations.count();

    const ArrayStationSignals * stat;
    ComplexMatrix& covmat=*_matrix;
    ComplexMatrix sig(2*stationCount, 1);
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(is, 0);
        s=geom.radialRotation(is).sin()*stat->spectrumValue(_northIndex);
        s+=geom.radialRotation(is).cos()*stat->spectrumValue(_eastIndex);
        s*=geom.amplitudeFactor(is);
      }
      for(int is=0; is<stationCount; is++) {
        Complex& s=sig.at(stationCount+is, 0);
        s=_stations.at(is)->spectrumValue(_verticalIndex);
        s*=geom.amplitudeFactor(is);
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(is, 0);
          s=geom.radialRotation(is).sin()*stat->spectrumValue(_northIndex, iFreq);
          s+=geom.radialRotation(is).cos()*stat->spectrumValue(_eastIndex, iFreq);
          s*=filter*geom.amplitudeFactor(is);
        }
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(stationCount+is, 0);
          s=_stations.at(is)->spectrumValue(_verticalIndex, iFreq);
          s*=filter*geom.amplitudeFactor(is);
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  void FKCrossSpectrum::addActiveTransverse(const ActiveFKSteering& geom)
  {
    int stationCount=_stations.count();

    const ArrayStationSignals * stat;
    ComplexMatrix& covmat=*_matrix;
    ComplexMatrix sig(stationCount, 1);
    if(parameters()->frequencyBandwidth()==0.0) {
      for(int is=0; is<stationCount; is++) {
        stat=_stations.at(is);
        Complex& s=sig.at(is, 0);
        s=geom.transverseRotation(is).sin()*stat->spectrumValue(_northIndex);
        s+=geom.transverseRotation(is).cos()*stat->spectrumValue(_eastIndex);
        s*=geom.amplitudeFactor(is);
      }
      covmat+=sig*sig.conjugateTransposedVector();
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          stat=_stations.at(is);
          Complex& s=sig.at(is, 0);
          s=geom.transverseRotation(is).sin()*stat->spectrumValue(_northIndex, iFreq);
          s+=geom.transverseRotation(is).cos()*stat->spectrumValue(_eastIndex, iFreq);
          s*=filter*geom.amplitudeFactor(is);
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  void FKCrossSpectrum::addActiveThreeComponent(const ActiveFKSteering& geom)
  {
    int stationCount=_stations.count();
    int stationCount2=2*stationCount;

    ComplexMatrix& covmat=*_matrix;
    ComplexMatrix sig(3*stationCount, 1);
    if(parameters()->frequencyBandwidth()==0.0) {
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(is, 0);
          s=_stations.at(is)->spectrumValue(_eastIndex, iFreq);
          s*=filter*geom.amplitudeFactor(is);
        }
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(stationCount+is, 0);
          s=_stations.at(is)->spectrumValue(_northIndex, iFreq);
          s*=filter*geom.amplitudeFactor(is);
        }
        for(int is=0; is<stationCount; is++) {
          Complex& s=sig.at(stationCount2+is, 0);
          s=_stations.at(is)->spectrumValue(_verticalIndex, iFreq);
          s*=filter*geom.amplitudeFactor(is);
        }
        covmat+=sig*sig.conjugateTransposedVector();
      }
    }
  }

  /*!
  */
  bool FKCrossSpectrum::invert(ComplexMatrix& covmat)
  {
    ComplexMatrix u, vt;
    VectorList<double> sigma;
    int n=covmat.rowCount();

    /*covmat.singularValue(sigma, u, vt);
    ComplexMatrix sigma1(sigma.count());
    sigma1.zero();
    for(int i=n-2; i>=0; i--) {
      sigma1.at(i, i)=sigma.at(i);
    }
    covmat=u;
    covmat*=sigma1;
    covmat*=vt;*/

    if(parameters()->damping()>0.0) {
      // The total power
      double d=covmat.trace(0.0).re();
      //App::log(tr("Trace %1\n").arg(d));
      d*=parameters()->damping();
      for(int ir=0; ir<n; ir++) {
        covmat.at(ir, ir)+=d;
      }
    }
    covmat.singularValue(sigma, u, vt);
    double sRef=1e-12*sigma.first();
    for(int row=0; row<n ;row++ ) {
      if(sigma.at(row)<sRef) {
        if(_timeWindows) {
          App::log(tr("%1 Hz, effective rank of cross-spectrum matrix: %2\n")
                           .arg(_timeWindows->frequency().center())
                           .arg(row));
        } else {
          App::log(tr("Effective rank of cross-spectrum matrix: %1\n")
                           .arg(row));
        }
        return false;
      }
    }
    //App::log(tr("Largest sigma %1 smallest sigma %2\n").arg(sigma.first()).arg(sigma.last()));
    /*ComplexMatrix sigma1(sigma.count());
    sigma1.zero();
    int index=3;
    sigma1.at(index, index)=sigma.at(index);
    covmat=u;
    covmat*=sigma1;
    covmat*=vt;
    return true;*/

    //ComplexMatrix original=covmat;
    if(!covmat.invert(u, sigma, vt)) {
      return false;
    }
    /*original*=covmat;
    // Quantify distance from identity
    double sum=0.0;
    for(int i=original.rowCount()-1; i>=0; i--) {
      for(int j=original.columnCount()-1; j>i; j--) {
        sum+=original.at(i, j).abs();
      }
      sum+=original.at(i, i).abs()-1.0;
      for(int j=i-1; j>=0; j--) {
        sum+=original.at(i, j).abs();
      }
    }
    App::log(tr("Identity misfit %1\n").arg(sum));*/
    return true;
  }

  bool FKCrossSpectrum::setMask()
  {
    TRACE;
    FKParameters::ProcessType processType;
    processType=static_cast<const FKParameters *>(_parameters)->processType();
    switch(_mode) {
    case ArrayStations::Vertical:
      switch(processType) {
      case FKParameters::Conventional:
      case FKParameters::Capon:
        return setOneComponentArrayMask();
      default:
        break;
      }
      break;
    case ArrayStations::Horizontal:
      break;
    case ArrayStations::ThreeComponents:
      switch(processType) {
      case FKParameters::ARDS:
        return setRayleighArrayMask();
      default:
        break;
      }
      break;
    }
    return false;
  }

  bool FKCrossSpectrum::setRayleighArrayMask()
  {
    const QList<QStringList>& arrays=static_cast<const FKParameters *>(_parameters)->arrays();
    if(arrays.isEmpty()) {
      return true;
    }
    // Map station names to indexes
    QHash<QString, int> stationIndex;
    int n=_stations.count();
    for(int i=n-1; i>=0; i--) {
      stationIndex.insert(_stations.at(i)->originalSignals()->name(), i);
    }
    _arrayMask.resize(3*n);
    _arrayMask.zero();
    QList<QStringList>::const_iterator ita;
    APP_LOG(1, tr("Mapping station names to index...\n"))
    for(ita=arrays.begin(); ita!=arrays.end(); ita++) {
      VectorList<int> index;
      index.reserve(ita->count());
      for(QStringList::const_iterator it=ita->begin(); it!=ita->end(); it++) {
        int i=stationIndex.value(*it, -1);
        if(i==-1) {
          App::log(tr("unknwon station name '%1'\n").arg(*it));
          return false;
        }
        index.append(i);
        APP_LOG(1, tr("'%1' ---> %2\n").arg(*it).arg(i))
      }
      for(VectorList<int>::const_iterator itr=index.begin(); itr!=index.end(); itr++) {
        int ir0=*itr;
        for(VectorList<int>::const_iterator itc=index.begin(); itc!=index.end(); itc++) {
          int ic0=*itc;
          for(int ir=2*n+ir0; ir>=ir0; ir-=n) {
            for(int ic=2*n+ic0; ic>=ic0; ic-=n) {
              double& v=_arrayMask.at(ir, ic);
              if(v==0.0) {
                v=1;
              }
            }
          }
        }
      }
    }
    APP_LOG(4, _arrayMask.toUserString())
    return true;
  }

  bool FKCrossSpectrum::setOneComponentArrayMask()
  {
    const QList<QStringList>& arrays=static_cast<const FKParameters *>(_parameters)->arrays();
    if(arrays.isEmpty()) {
      return true;
    }
    // Map station names to indexes
    QHash<QString, int> stationIndex;
    int n=_stations.count();
    for(int i=n-1; i>=0; i--) {
      stationIndex.insert(_stations.at(i)->originalSignals()->name(), i);
    }
    _arrayMask.resize(n);
    _arrayMask.zero();
    QList<QStringList>::const_iterator ita;
    APP_LOG(1, tr("Mapping station names to index...\n"))
    for(ita=arrays.begin(); ita!=arrays.end(); ita++) {
      VectorList<int> index;
      index.reserve(ita->count());
      APP_LOG(1, tr("  Array containing %1 stations\n").arg(ita->count()))
      for(QStringList::const_iterator it=ita->begin(); it!=ita->end(); it++) {
        int i=stationIndex.value(*it, -1);
        if(i==-1) {
          App::log(tr("    unknown station name '%1'\n").arg(*it));
          return false;
        }
        index.append(i);
        APP_LOG(1, tr("    '%1' ---> %2\n").arg(*it).arg(i))
      }
      for(VectorList<int>::const_iterator itr=index.begin(); itr!=index.end(); itr++) {
        for(VectorList<int>::const_iterator itc=index.begin(); itc!=index.end(); itc++) {
          double& v=_arrayMask.at(*itr, *itc);
          if(v==0.0) {
            v=1;
          }
        }
      }
    }
    APP_LOG(4, _arrayMask.toOctaveString()+"\n")
    return true;
  }

  void FKCrossSpectrum::addRayleighSingle()
  {
    Complex east, north, vertical;
    ASSERT(_stations.count()==1);
    if(parameters()->frequencyBandwidth()==0.0) {
      east=_stations.at(0)->spectrumValue(_eastIndex);
      north=_stations.at(0)->spectrumValue(_northIndex);
      vertical=_stations.at(0)->spectrumValue(_verticalIndex);
    } else {
      const GaussianFrequencyBand& frequencyFilter=*_timeWindows->frequencyFilter();
      int nFreq=frequencyFilter.maximumIndex();
      double filter;
      for(int iFreq=frequencyFilter.minimumIndex(); iFreq<=nFreq; iFreq++) {
        filter=frequencyFilter.taperValue(iFreq);
        east=_stations.at(0)->spectrumValue(_eastIndex, iFreq)*filter;
        north=_stations.at(0)->spectrumValue(_northIndex, iFreq)*filter;
        vertical=_stations.at(0)->spectrumValue(_verticalIndex, iFreq)*filter;
      }
    }
    Complex conjVertical=conjugate(vertical);
    Complex crossEV=east*conjVertical;
    Complex crossNV=north*conjVertical;
    // Find horizontal propagation direction that provides a cross product with a null real part
    // Its imaginary part should be -|S_z|^2 je+... write the noise and Love contributions
    double theta=atan(-crossEV.abs()/crossNV.abs()*
                      cos(crossEV.phase())/cos(crossNV.phase()));
    Complex h=cos(theta)*east+sin(theta)*north;
    double crossH=(h*conjVertical).im();
    // Make sure that delay between h and vertical has always the same sign
    Complex delay;
    if(crossH>0.0) {
      delay.setUnitExp(-vertical.phase());
    } else {
      delay.setUnitExp(vertical.phase());
    }
     // Set a null phase for vertical: real for vertical and imaginary positive for horizontal
    h*=delay;
    vertical*=delay;
    _matrix->at(0, 0)+=h;
    _matrix->at(0, 1)+=vertical;
    // TODO:
    //  - check that the stupid stack of abs values is as good
    //  - write it in a dedicated SingleEllipticityCrossSpectrum avoid fake grid search
    //  - check also average of ellipticity values (crossH/vertical.abs2())
    //  - number of blocks driven be stability of ellipticity
    //  - check that correlation coefficient from Hobiger is always 1, hence useless.
    //  - show effect of bandwidth
  }

  void FKCrossSpectrum::meanRayleighSingle()
  {
    _matrix->at(2, 2)=sqrt(_matrix->at(0, 0).abs2()/_matrix->at(0, 1).abs2());
  }

} // namespace ArrayCore

