/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-07-22
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ActiveConvFKRayleigh.h"
#include "FKCrossSpectrum.h"
#include "ActiveConvEllipticityEngine.h"

namespace ArrayCore {

  /*!
    \class ActiveConvFKRayleigh ActiveConvFKRayleigh.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  ActiveConvFKRayleigh::ActiveConvFKRayleigh(FKCache * cache)
    : AbstractFKRayleigh(cache)
  {
    _steering=nullptr;
  }

  ActiveConvFKRayleigh::~ActiveConvFKRayleigh()
  {
    delete _steering;
  }

  void ActiveConvFKRayleigh::initialize(const FKParameters& param)
  {
    _ellOptimization.setAbsolutePrecision(param.ellipticityAbsolutePrecision());
    AbstractFKRayleigh::initialize(param);
  }

  bool ActiveConvFKRayleigh::setSource(const VectorList<int>& blocks)
  {
    // Similar code as in ActiveConvFKVertical
    // Modifications:
    //   - added _steering.setRadialRotation()
    //   - 2N dimension for steering vectors
    bool ok=true;
    // According to distance parameters, the number of selected stations
    // after this call may be less than the number of stations of the
    // array in cache which is the parent of the array in cross spectrum.
    Point src=_crossSpectrum->setActiveSource(blocks, ok);
    if(!ok) {
      return false;
    }
    // When source grid is defined
    src+=_crossSpectrum->sourceOffset();
    // Number of selected stations is now correct
    setPowerEngine(new FKPower(_crossSpectrum->array().count(), 2));
    powerEngine()->setCrossSpectrum(_crossSpectrum->matrix());
    // Cache cannot be modified like this: not thread safe
    // Number of receivers and their indexes may vary, so not wise to put it in the main cache
    // Steering must be kept by the function itself
    if(!_steering) {
      _steering=new ActiveFKSteering;
    }
    _steering->setArray(&_crossSpectrum->array());
    _steering->setSource(src);
    _steering->setDistances();
    _steering->setAmplitudeFactors();
    _steering->setRadialRotations();
    if(_ellEngine) {
      static_cast<ActiveConvEllipticityEngine *>(_ellEngine)->shareSteering(this);
    }
    return true;
  }

  void ActiveConvFKRayleigh::shareSteering(const ActiveConvFKRayleigh * s)
  {
    setPowerEngine(new FKPower(_crossSpectrum->array().count(), 2));
    powerEngine()->setCrossSpectrum(_crossSpectrum->matrix());
    _steering=s->_steering;
  }

  void ActiveConvFKRayleigh::clearSharedSteering()
  {
    _steering=nullptr;
  }

  FunctionSearch * ActiveConvFKRayleigh::createSearch(const FKParameters& param)
  {
    GridSearch1D * grid=new GridSearch1D(3);
    grid->setRelativePrecision(0, param.velocityRelativePrecision());
    grid->setFunction(this);
    grid->setGrid(param.gridStep(), param.effectiveGridSize(), param.gridStep());
    return grid;
  }

  void ActiveConvFKRayleigh::resetCrossSpectrum()
  {
    _crossSpectrum->resetMatrix();
  }

  void ActiveConvFKRayleigh::addCrossSpectrum()
  {
    // Correction for geometrical attenuation can also be achieved in steering
    // However, the steering correction is not the same for Conventional and High Resolution
    // Hence, simpler to correct the cross-spectrum.
    _crossSpectrum->addActiveRayleigh(*_steering);
  }

  void ActiveConvFKRayleigh::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanMatrix(nBlocks);
  }

  void ActiveConvFKRayleigh::addIncoherentNoise(double horizontal, double vertical)
  {
    int n=_crossSpectrum->array().count();
    int rotCount=_crossSpectrum->rotationStepCount();
    for(int rotIndex=0; rotIndex<rotCount; rotIndex++) {
      ComplexMatrix& covmat=*_crossSpectrum->matrix();
      for(int i=0; i<n; i++) {
        covmat.at(i, i)+=horizontal;
        covmat.at(i+n, i+n)+=vertical;
      }
    }
  }

} // namespace ArrayCore

