/***************************************************************************
**
**  This file is part of waran.
**
**  waran is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  waran is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2012-02-08
**  Copyright: 2012-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SystemLink.h"
#include "Station.h"

/*!
  \class SystemLink SystemLink.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
SystemLink::SystemLink(Station * station)
  : DaemonLink(station)
{
  TRACE;
  setPort(2978);
  _refresh.setInterval(10000); // 10 s during startup
  QObject::connect(&_refresh, SIGNAL(timeout()), this, SLOT(refreshVariableParameters()));
  _refresh.start();
}

void SystemLink::connected()
{
  TRACE;
  DaemonLink::connected();
  refreshVariableParameters();
}

void SystemLink::refreshVariableParameters()
{
  if(isAvailable()) {
    diskSpace();
    battery();
    temperature();
  }
}

void SystemLink::highRateRefresh()
{
  _refresh.setInterval(10000); // 10 s during startup
}

/*!

*/
int SystemLink::bytesAvailable(const char * buffer, int bytesCount)
{
  TRACE;
  int bytesRead=0;
  while(bytesRead<bytesCount) {
    // Scan for a line
    int newBytesRead;
    for(newBytesRead=bytesRead; newBytesRead<bytesCount && buffer[newBytesRead]!='\n'; newBytesRead++) {}
    if(newBytesRead==bytesCount) {
      return bytesRead; // partial line
    }
    switch(buffer[bytesRead]) {
    case 'c':
      if(match(buffer, bytesRead, bytesCount, "comments=")) {
        if(!comments(buffer, bytesRead, bytesCount)) {
          return bytesRead;
        }
      }
      break;
    case 'd':
      if(match(buffer, bytesRead, bytesCount, "diskspace=")) {
        diskSpace(buffer, bytesRead, bytesCount);
      }
      break;
    case 'e':
      if(match(buffer, bytesRead, bytesCount, "externalbattery=")) {
        station()->setExternalBattery(QByteArray(buffer+bytesRead, newBytesRead-bytesRead));
      }
      break;
    case 'i':
      if(match(buffer, bytesRead, bytesCount, "internalbattery=")) {
        station()->setInternalBattery(QByteArray(buffer+bytesRead, newBytesRead-bytesRead));
      }
      break;
    case 'h':
      if(match(buffer, bytesRead, bytesCount, "halting")) {
        halting(buffer, bytesRead, bytesCount);
      }
      break;
    case 't':
      if(match(buffer, bytesRead, bytesCount, "temperature=")) {
        station()->setTemperature(QByteArray(buffer+bytesRead, newBytesRead-bytesRead));
      }
      break;
    default:
      break;
    }
    bytesRead=newBytesRead+1;
    // Skip blanks and additional end of line characters
    while(bytesRead<bytesCount && isspace(buffer[bytesRead])) {bytesRead++;}
  }
  return bytesRead;
}

void SystemLink::diskSpace(const char * buffer, int bytesRead, int bytesCount)
{
  TRACE;
  bool ok;
  double value=readDouble(buffer, bytesRead, bytesCount, ok);
  if(ok) {
    if(value>0.0 && _refresh.interval()<60000) { // Reduce refresh rate (once every minute)
      _refresh.setInterval(60000);
    }
    station()->setDiskSpace(value);
  }
}

bool SystemLink::comments(const char * buffer, int& bytesRead, int bytesCount)
{
  TRACE;
  bool ok;
  int value=readInteger(buffer, bytesRead, bytesCount, ok);
  if(ok) {
    if(bytesRead+1+value<=bytesCount) {
      station()->setComments(QByteArray(buffer+bytesRead+1, value));
      bytesRead+=value; // No new line at the end, the new line is automatically skipped above
      return true;
    } else {
      return false;
    }
  }
  return true;
}

void SystemLink::halting(const char *, int, int)
{
  TRACE;
  station()->setGpsState("Halting...");
  station()->setSeismicState("Halting...");
}

void SystemLink::powerOff()
{
  TRACE;
  send("poweroff\n");
}

void SystemLink::reboot()
{
  TRACE;
  send("reboot\n");
}

void SystemLink::temperature()
{
  TRACE;
  send("temperature\n");
}

void SystemLink::battery()
{
  TRACE;
  send("internalbattery\n");
  send("externalbattery\n");
}

void SystemLink::diskSpace()
{
  TRACE;
  send("diskspace\n");
}

void SystemLink::clearUsb()
{
  TRACE;
  send("clearusb\n");
}

void SystemLink::comments()
{
  TRACE;
  send("comments\n");
}

void SystemLink::addComment(const QString& c)
{
  TRACE;
  QByteArray buf=c.toUtf8();
  QByteArray msg;
  msg.setNum(buf.size());
  msg.prepend("addcomment=");
  msg.append("\n");
  send(msg.data(), msg.size());
  send(buf.data(), buf.size());
}
