/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef MODEWIDGETS_H
#define MODEWIDGETS_H

#include <QGpGuiMath.h>

#include "Samples.h"
#include "Reader.h"
#include "ClassificationParameters.h"

class ModeWidgets : public QObject
{
  Q_OBJECT
public:
  ModeWidgets(QObject * parent=nullptr);
  ~ModeWidgets();

  void setPlotType(Reader::PlotType t);
  Reader::PlotType plotType() const {return _plotType;}

  void addWidgets(GraphicSheet * sheet, const Reader& param);
  void setSamples(Samples * s, const Reader& param);
  void setLimits(bool normalize);
  void setComments(const QString& cmt);
  void initCurves(CurveBrowser * curves);

  void setColorMap(const ColorMap& map);

  bool contains(AxisWindow * w) const;
  LineLayer * pickLayer(AxisWindow * w) const;
  CurveProxy * curveProxy(AxisWindow * w) const;
  CurvePlotProxy * curvePlotProxy(AxisWindow * w) const;

  LineLayer * dispersionPickLayer() const;
  LineLayer * ellipticityPickLayer() const;
  LineLayer * verticalNoisePickLayer() const;
  LineLayer * horizontalNoisePickLayer() const;
  LineLayer * totalNoisePickLayer() const;
  LineLayer * deltaNoisePickLayer() const;
  LineLayer * sigmaNoisePickLayer() const;
  LineLayer * azimuthPickLayer() const;
  LineLayer * powerPickLayer() const;

  void classifySamples(bool normalize);
  void classifySamples(const ClassificationParameters& param,
                       bool normalize);
  void exportSamples();
  void pickToMean();
  void adjust();

  void addMeanCurve();
  void addMedianCurve();
  void addModeCurve();
  void addGaussianMixtureCurve();
  void addSmartMeanCurve();
private slots:
  void updatePlots();
  void addGaussianMixture(int index);
private:
  void initCurves(LineLayer * pickLayer);
  Histogram2D * categorizedHistogram(LineLayer * layer);
  Histogram2D * histogram(LineLayer * layer) const;
  void updateHistogram(Histogram2D * destHist, GraphContentsLayer * layer,
                       Histogram2D * srcHist, bool normalize);
  void updateHistograms(Histogram2D * hist, bool normalize);

  void autoPickDispersion(LineLayer * layer, Histogram2D * hist);
  void autoPickEllipticity(LineLayer * layer, Histogram2D * hist);
  void setPlot(GraphContentsLayer * layer, Histogram2D * hist, bool normalized);
  void setDispersionHistogram(const Samples * s, const Reader& param);
  void setEllipticityHistogram(const Samples * s, const Reader& param);
  void setVerticalNoiseHistogram(const Samples * s, const Reader& param);
  void setHorizontalNoiseHistogram(const Samples * s, const Reader& param);
  void setTotalNoiseHistogram(const Samples * s, const Reader& param);
  void setDeltaNoiseHistogram(const Samples * s, const Reader& param);
  void setSigmaNoiseHistogram(const Samples * s, const Reader& param);
  void setAzimuthHistogram(const Samples * s, const Reader& param);
  void setPowerHistogram(const Samples * s, const Reader& param);
  void setLimits(GraphContentsLayer * layer, bool normalize, bool setY);
  GaussianMixtureDistribution filterGaussianMixture(int index);
  void modeToPoints(const GaussianMixtureDistribution& d,
                    int histIndex, double x, SamplingOptions samp,
                    CurveProxy * proxy, CurvePlotProxy * plotProxy);
  AbstractLine * addMedianCurve(const Histogram2D * hist, LineLayer * layer, int category);
  void filter(Histogram2D * hist, LineLayer * layer, AbstractLine * line, int category);

  Samples * _samples;
  QString _name;
  Reader::PlotType _plotType;

  CurveBrowser * _curveBrowser;

  GraphContentsLayer * _dispersion;
  Histogram2D * _dispersionHist;
  LineLayer * _dispersionPick;

  GraphContentsLayer * _ellipticity;
  Histogram2D * _ellipticityHist;
  LineLayer * _ellipticityPick;

  GraphContentsLayer * _verticalNoise;
  Histogram2D * _verticalNoiseHist;
  LineLayer * _verticalNoisePick;

  GraphContentsLayer * _horizontalNoise;
  Histogram2D * _horizontalNoiseHist;
  LineLayer * _horizontalNoisePick;

  GraphContentsLayer * _totalNoise;
  Histogram2D * _totalNoiseHist;
  LineLayer * _totalNoisePick;

  GraphContentsLayer * _deltaNoise;
  Histogram2D * _deltaNoiseHist;
  LineLayer * _deltaNoisePick;

  GraphContentsLayer * _sigmaNoise;
  Histogram2D * _sigmaNoiseHist;
  LineLayer * _sigmaNoisePick;

  GraphContentsLayer * _azimuth;
  Histogram2D * _azimuthHist;
  LineLayer * _azimuthPick;

  GraphContentsLayer * _power;
  Histogram2D * _powerHist;
  LineLayer * _powerPick;

  LegendWidget * _categories;

  GaussianMixtureParameters _gaussianMixtureParameters;
  AbstractLine * _gaussianMixtureLines[3];
  HistogramInversion * _gaussianMixtureInversion;
};

#endif // MODEWIDGETS_H

