/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-16
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiWave.h>
#include <QGpGuiStat.h>

#include "ModeWidgets.h"
#include "SamplePlot.h"
#include "VerticalNoiseProxy.h"
#include "HorizontalNoiseProxy.h"
#include "TotalNoiseProxy.h"
#include "SigmaNoiseProxy.h"
#include "DeltaNoiseProxy.h"
#include "AzimuthProxy.h"
#include "PowerProxy.h"
#include "NoisePlotProxy.h"
#include "AzimuthPlotProxy.h"
#include "PowerPlotProxy.h"

/*!
  \class ModeWidgets ModeWidgets.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
ModeWidgets::ModeWidgets(QObject * parent)
  : QObject(parent)
{
  TRACE;
  _samples=nullptr;
  _plotType=Reader::Histograms;

  _dispersion=nullptr;
  _dispersionPick=nullptr;
  _dispersionHist=nullptr;
  _ellipticity=nullptr;
  _ellipticityPick=nullptr;
  _ellipticityHist=nullptr;
  _verticalNoise=nullptr;
  _verticalNoisePick=nullptr;
  _verticalNoiseHist=nullptr;
  _horizontalNoise=nullptr;
  _horizontalNoisePick=nullptr;
  _horizontalNoiseHist=nullptr;
  _totalNoise=nullptr;
  _totalNoisePick=nullptr;
  _totalNoiseHist=nullptr;
  _deltaNoise=nullptr;
  _deltaNoisePick=nullptr;
  _deltaNoiseHist=nullptr;
  _sigmaNoise=nullptr;
  _sigmaNoisePick=nullptr;
  _sigmaNoiseHist=nullptr;
  _azimuth=nullptr;
  _azimuthPick=nullptr;
  _azimuthHist=nullptr;
  _power=nullptr;
  _powerPick=nullptr;
  _powerHist=nullptr;

  _gaussianMixtureLines[0]=nullptr;
  _gaussianMixtureLines[1]=nullptr;
  _gaussianMixtureInversion=nullptr;
}

/*!
  Description of destructor still missing
*/
ModeWidgets::~ModeWidgets()
{
  TRACE;
  delete _gaussianMixtureInversion;
  delete _dispersionHist;
  delete _ellipticityHist;
  delete _verticalNoiseHist;
  delete _horizontalNoiseHist;
  delete _totalNoiseHist;
  delete _deltaNoiseHist;
  delete _sigmaNoiseHist;
  delete _azimuthHist;
  delete _powerHist;
  delete _samples;
}

void ModeWidgets::setPlotType(Reader::PlotType t)
{
  ASSERT(!_dispersion);
  _plotType=t;
}

void ModeWidgets::setPlot(GraphContentsLayer * layer, Histogram2D * hist, bool normalized)
{
  TRACE;
  if(layer && hist) {
    switch(_plotType) {
    case Reader::Histograms:
      if(normalized) {
        hist->normalizeArea(YAxis);
      }
      static_cast<IrregularGrid2DPlot *>(layer)->setGrid(*hist);
      break;
    case Reader::Dots:
      static_cast<SamplePlot *>(layer)->setHistogram(hist);
      break;
    }
  }
}

void ModeWidgets::setComments(const QString &cmt)
{
  TRACE;
  if(_dispersion) _dispersion->setComments(cmt);
  if(_ellipticity) _ellipticity->setComments(cmt);
  if(_verticalNoise) _verticalNoise->setComments(cmt);
  if(_horizontalNoise) _horizontalNoise->setComments(cmt);
  if(_totalNoise) _totalNoise->setComments(cmt);
  if(_deltaNoise) _deltaNoise->setComments(cmt);
  if(_sigmaNoise) _sigmaNoise->setComments(cmt);
  if(_azimuth) _azimuth->setComments(cmt);
  if(_power) _power->setComments(cmt);
}

void ModeWidgets::autoPickDispersion(LineLayer * layer, Histogram2D * hist)
{
  layer->clear();
  Curve<RealStatisticalPoint> all=hist->pickAll(4, 0.1, 0.2);
  layer->addLine(Pen(Qt::NoPen), Symbol(Symbol::Circle, 0.5, Pen(Qt::red), Brush(Qt::SolidPattern)));
  static_cast<RealStatisticalLine *>(layer->line(0))->curve()=all;

  QList<Curve<RealStatisticalPoint>> curves=all.split(1.2, LogScale, 0.1, 5);
  int n=curves.count();
  for(int i=0; i<n; i++) {
    layer->addLine(Pen(Qt::black), Symbol(Symbol::Circle, 0.8, Pen(Qt::black), Brush(Qt::SolidPattern)));
    Curve<RealStatisticalPoint>& c=static_cast<RealStatisticalLine *>(layer->line(i+1))->curve();
    c=curves.at(i);
  }
}

void ModeWidgets::autoPickEllipticity(LineLayer * layer, Histogram2D * hist)
{
  layer->clear();
  Curve<RealStatisticalPoint> all=hist->pickAll(3, 5.0, 10.0); // in deg.
  layer->addLine(Pen(Qt::NoPen), Symbol(Symbol::Circle, 0.5, Pen(Qt::red), Brush(Qt::SolidPattern)));
  static_cast<RealStatisticalLine *>(layer->line(0))->curve()=all;

  QList<Curve<RealStatisticalPoint>> curves=all.split(1.2, LogScale, 0.1, 5);
  int n=curves.count();
  for(int i=0; i<n; i++) {
    layer->addLine(Pen(Qt::black), Symbol(Symbol::Circle, 0.8, Pen(Qt::black), Brush(Qt::SolidPattern)));
    Curve<RealStatisticalPoint>& c=static_cast<RealStatisticalLine *>(layer->line(i+1))->curve();
    c=curves.at(i);
  }
}

void ModeWidgets::setSamples(Samples * s, const Reader& param)
{
  TRACE;
  _samples=s;
  if(_dispersion) {
    setDispersionHistogram(s, param);
    if(param.autoPick().value()) {
      autoPickDispersion(_dispersionPick, _dispersionHist);
    }
    setPlot(_dispersion, _dispersionHist, param.isNormalized());
  }
  if(_ellipticity) {
    setEllipticityHistogram(s, param);
    if(param.autoPick().value()) {
      autoPickEllipticity(_ellipticityPick, _ellipticityHist);
    }
    setPlot(_ellipticity, _ellipticityHist, param.isNormalized());
  }
  if(_verticalNoise) {
    setVerticalNoiseHistogram(s, param);
    setPlot(_verticalNoise, _verticalNoiseHist, param.isNormalized());
  }
  if(_horizontalNoise) {
    setHorizontalNoiseHistogram(s, param);
    setPlot(_horizontalNoise, _horizontalNoiseHist, param.isNormalized());
  }
  if(_totalNoise) {
    setTotalNoiseHistogram(s, param);
    setPlot(_totalNoise, _totalNoiseHist, param.isNormalized());
  }
  if(_deltaNoise) {
    setDeltaNoiseHistogram(s, param);
    setPlot(_deltaNoise, _deltaNoiseHist, param.isNormalized());
  }
  if(_sigmaNoise) {
    setSigmaNoiseHistogram(s, param);
    setPlot(_sigmaNoise, _sigmaNoiseHist, param.isNormalized());
  }
  if(_azimuth) {
    setAzimuthHistogram(s, param);
    setPlot(_azimuth, _azimuthHist, param.isNormalized());
  }
  if(_power) {
    setPowerHistogram(s, param);
    setPlot(_power, _powerHist, param.isNormalized());
  }
}

void ModeWidgets::addWidgets(GraphicSheet * sheet, const Reader& param)
{
  TRACE;
  _gaussianMixtureParameters.setKmin(0.5*param.arrayKmin());
  _gaussianMixtureParameters.setKmax(param.arrayKmax());
  _gaussianMixtureParameters.setVerbose(true);

  AxisWindow * w;
  ColorMap map;
  double width=8.0;
  double rightPos=sheet->printRight()+9.5;
  double topPos=0.5;
  map.generate(0, 19, ColorPalette::defaultSequentialWhite, ColorPalette::Reversed);
  map.setWhiteTransparent(true);

  IrregularGrid2DPlot * histPlot;
  SamplePlot * catPlot;

  if(param.showPlots() & Reader::Dispersion) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Dispersion");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _dispersion=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _dispersion=catPlot;
      break;
    }
    _dispersionPick=new LineLayer(w);
    _dispersionPick->setObjectName("Pick");
    ModalLine * l=new ModalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _dispersionPick->setReferenceLine(l);
    _dispersionPick->setLogErrorBar(true);
    _dispersionPick->setOpacity(0.5);
    DispersionLimitLayer * limits=new DispersionLimitLayer(w);
    limits->setObjectName("FK dispersion limits");
    limits->addArrayLimits();
    if(param.xSampling()) {
      limits->setFrequencySampling(*param.xSampling());
    }
    limits->setArrayKmin(param.arrayKmin());
    limits->setArrayKmax(param.arrayKmax());
    if(!param.referenceDispersionLayer().isEmpty()) {
      w->graphContents()->appendLayers(param.referenceDispersionLayer());
    }
    w->updateExternalGeometry();
    topPos+=7.0;
  }

  if(param.showPlots() & Reader::Ellipticity) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Ellipticity");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _ellipticity=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _ellipticity=catPlot;
      break;
    }
    _ellipticityPick=new LineLayer(w);
    _ellipticityPick->setObjectName("Pick");
    ModalLine * l=new ModalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _ellipticityPick->setReferenceLine(l);
    _ellipticityPick->setOpacity(0.5);
    _ellipticityPick->setSignThreshold(45.0);
    if(!param.referenceEllipticityLayer().isEmpty()) {
      w->graphContents()->appendLayers(param.referenceEllipticityLayer());
    }
    w->updateExternalGeometry();
    topPos+=7.0;
  }

  if(param.showPlots() & Reader::VerticalNoise) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"VerticalNoise");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(4.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _verticalNoise=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _verticalNoise=catPlot;
      break;
    }
    _verticalNoisePick=new LineLayer(w);
    _verticalNoisePick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _verticalNoisePick->setReferenceLine(l);
    _verticalNoisePick->setLogErrorBar(true);
    _verticalNoisePick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=4.0;
  }

  if(param.showPlots() & Reader::HorizontalNoise) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"HorizontalNoise");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(4.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _horizontalNoise=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _horizontalNoise=catPlot;
      break;
    }
    _horizontalNoisePick=new LineLayer(w);
    _horizontalNoisePick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _horizontalNoisePick->setReferenceLine(l);
    _horizontalNoisePick->setLogErrorBar(true);
    _horizontalNoisePick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=4.0;
  }

  if(param.showPlots() & Reader::TotalNoise) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"TotalNoise");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(4.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _totalNoise=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _totalNoise=catPlot;
      break;
    }
    _totalNoisePick=new LineLayer(w);
    _totalNoisePick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _totalNoisePick->setReferenceLine(l);
    _totalNoisePick->setLogErrorBar(true);
    _totalNoisePick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=4.0;
  }

  if(param.showPlots() & Reader::DeltaNoise) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"DeltaNoise");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(4.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _deltaNoise=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _deltaNoise=catPlot;
      break;
    }
    _deltaNoisePick=new LineLayer(w);
    _deltaNoisePick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _deltaNoisePick->setReferenceLine(l);
    _deltaNoisePick->setLogErrorBar(false);
    _deltaNoisePick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=4.0;
  }

  if(param.showPlots() & Reader::SigmaNoise) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"SigmaNoise");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(4.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _sigmaNoise=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _sigmaNoise=catPlot;
      break;
    }
    _sigmaNoisePick=new LineLayer(w);
    _sigmaNoisePick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _sigmaNoisePick->setReferenceLine(l);
    _sigmaNoisePick->setLogErrorBar(true);
    _sigmaNoisePick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=4.0;
  }

  if(param.showPlots() & Reader::Azimuth) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Azimuth");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _azimuth=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _azimuth=catPlot;
      break;
    }
    _azimuthPick=new LineLayer(w);
    _azimuthPick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _azimuthPick->setReferenceLine(l);
    _azimuthPick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=7.0;
  }

  if(param.showPlots() & Reader::Power) {
    w=new AxisWindow(sheet);
    w->setObjectName(_name+"Power");
    sheet->addObject(w);
    w->setAnchor(AxisWindow::TopRight);
    w->xAxis()->setSizeType(Axis::AxisSize);
    w->xAxis()->setSizeInfo(width);
    w->yAxis()->setNumberType('e');
    w->yAxis()->setSizeInfo(7.0);
    w->setPrintRight(rightPos);
    w->setPrintTop(topPos);
    switch(_plotType) {
    case Reader::Histograms:
      histPlot=new IrregularGrid2DPlot(w);
      histPlot->setObjectName("Histogram");
      histPlot->setSmooth(false);
      histPlot->setColorMap(map);
      _power=histPlot;
      break;
    case Reader::Dots:
      catPlot=new SamplePlot(w);
      catPlot->setObjectName("Dots");
      _power=catPlot;
      break;
    }
    _powerPick=new LineLayer(w);
    _powerPick->setObjectName("Pick");
    RealStatisticalLine * l=new RealStatisticalLine;
    l->setPen(Pen(Qt::black, 0.6));
    l->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                        Brush(Qt::black, Qt::SolidPattern)));
    _powerPick->setReferenceLine(l);
    _powerPick->setLogErrorBar(true);
    _powerPick->setOpacity(0.5);
    w->updateExternalGeometry();
    topPos+=7.0;
  }

  if(_plotType==Reader::Dots) {
    _categories=new LegendWidget;
    _categories->setPrintXAnchor(1.0);
    _categories->setPrintYAnchor(topPos);
    _categories->setTitle(tr("Categories"));
    _categories->setLineSize(0.5);
    Legend l;
    l.generate(1, 4);
    l.setSymbol(0, Symbol(Symbol::Circle, 1.2, Pen(Qt::gray), Brush(Qt::gray, Qt::SolidPattern)));
    l.setText(0, tr("Not categorized"));
    for(int i=0; i<4; i++) {
      Pen p=l.pen(i+1);
      l.setSymbol(i+1, Symbol(Symbol::Circle, 1.2, p, Brush(p.color(), Qt::SolidPattern)));
      l.setText(i+1, tr("Mode %1").arg(i));
      p.setLineStyle(Pen::NoLine);
      l.setPen(i+1, p);
    }
    _categories->legend()=l;
    _categories->setAdjustBox(true);
    _categories->setObjectName("CategoryLegend");
    connect(_categories, SIGNAL(changed(const Legend&)), this, SLOT(updatePlots()));
    sheet->addObject(_categories);

    if(param.showPlots() & Reader::Dispersion) {
      static_cast<SamplePlot *>(_dispersion)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::Ellipticity) {
      static_cast<SamplePlot *>(_ellipticity)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::VerticalNoise) {
      static_cast<SamplePlot *>(_verticalNoise)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::HorizontalNoise) {
      static_cast<SamplePlot *>(_horizontalNoise)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::TotalNoise) {
      static_cast<SamplePlot *>(_totalNoise)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::DeltaNoise) {
      static_cast<SamplePlot *>(_deltaNoise)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::SigmaNoise) {
      static_cast<SamplePlot *>(_sigmaNoise)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::Azimuth) {
      static_cast<SamplePlot *>(_azimuth)->setCategories(&_categories->legend());
    }
    if(param.showPlots() & Reader::Power) {
      static_cast<SamplePlot *>(_power)->setCategories(&_categories->legend());
    }
  }
}

void ModeWidgets::initCurves(LineLayer * pickLayer)
{
  if(pickLayer) {
    _curveBrowser->setProxy(curveProxy(pickLayer->graph()));
    _curveBrowser->setPlotProxy(curvePlotProxy(pickLayer->graph()));
    _curveBrowser->initLayer(pickLayer);
  }
}

void ModeWidgets::initCurves(CurveBrowser * curves)
{
  _curveBrowser=curves;
  initCurves(_dispersionPick);
  initCurves(_ellipticityPick);
  initCurves(_verticalNoisePick);
  initCurves(_horizontalNoisePick);
  initCurves(_totalNoisePick);
  initCurves(_deltaNoisePick);
  initCurves(_sigmaNoisePick);
  initCurves(_azimuthPick);
  initCurves(_powerPick);
}

void ModeWidgets::setDispersionHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Dispersion histogram\n"));
  _dispersionHist=param.histogram(param.dispersionSampling());
  if(_dispersionHist) {
    // TODO support alternative ways of plotting dispersion (same for ellipticity)
    _dispersionHist->setYScaleType(Scale::InverseLog);
    if(_dispersionHist->setSamples(s->dispersionSamples())) {
      return;
    }
  }
  delete _dispersion->graph();
  _dispersion=nullptr;
  _dispersionPick=nullptr;
  delete _dispersionHist;
  _dispersionHist=nullptr;
}

void ModeWidgets::setEllipticityHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Ellipticity histogram\n"));
  _ellipticityHist=param.histogram(param.ellipticitySampling());
  if(_ellipticityHist) {
    _ellipticityHist->setYScaleType(Scale::Linear);
    _ellipticityHist->setPeriodic(true);
    if(_ellipticityHist->setSamples(s->ellipticitySamples())) {
      return;
    }
  }
  delete _ellipticity->graph();
  _ellipticity=nullptr;
  _ellipticityPick=nullptr;
  delete _ellipticityHist;
  _ellipticityHist=nullptr;
}

void ModeWidgets::setVerticalNoiseHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Vertical noise histogram\n"));
  _verticalNoiseHist=param.histogram(param.verticalNoiseSampling());
  if(_verticalNoiseHist) {
    _verticalNoiseHist->setYScaleType(Scale::Log);
    if(_verticalNoiseHist->setSamples(s->verticalNoiseSamples())) {
      return;
    }
  }
  delete _verticalNoise->graph();
  _verticalNoise=nullptr;
  _verticalNoisePick=nullptr;
  delete _verticalNoiseHist;
  _verticalNoiseHist=nullptr;
}

void ModeWidgets::setHorizontalNoiseHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Horizontal noise histogram\n"));
  _horizontalNoiseHist=param.histogram(param.horizontalNoiseSampling());
  if(_horizontalNoiseHist) {
    _horizontalNoiseHist->setYScaleType(Scale::Log);
    if(_horizontalNoiseHist->setSamples(s->horizontalNoiseSamples())) {
      return;
    }
  }
  delete _horizontalNoise->graph();
  _horizontalNoise=nullptr;
  _horizontalNoisePick=nullptr;
  delete _horizontalNoiseHist;
  _horizontalNoiseHist=nullptr;
}

void ModeWidgets::setTotalNoiseHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Total noise histogram\n"));
  _totalNoiseHist=param.histogram(param.totalNoiseSampling());
  if(_totalNoiseHist) {
    _totalNoiseHist->setYScaleType(Scale::Log);
    if(_totalNoiseHist->setSamples(s->totalNoiseSamples())) {
      return;
    }
  }
  delete _totalNoise->graph();
  _totalNoise=nullptr;
  _totalNoisePick=nullptr;
  delete _totalNoiseHist;
  _totalNoiseHist=nullptr;
}

void ModeWidgets::setDeltaNoiseHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Delta noise histogram\n"));
  _deltaNoiseHist=param.histogram(param.deltaNoiseSampling());
  if(_deltaNoiseHist) {
    _deltaNoiseHist->setYScaleType(Scale::Linear);
    if(_deltaNoiseHist->setSamples(s->deltaNoiseSamples())) {
      return;
    }
  }
  delete _deltaNoise->graph();
  _deltaNoise=nullptr;
  _deltaNoisePick=nullptr;
  delete _deltaNoiseHist;
  _deltaNoiseHist=nullptr;
}

void ModeWidgets::setSigmaNoiseHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Sigma noise histogram\n"));
  _sigmaNoiseHist=param.histogram(param.sigmaNoiseSampling());
  if(_sigmaNoiseHist) {
    _sigmaNoiseHist->setYScaleType(Scale::Log);
    if(_sigmaNoiseHist->setSamples(s->sigmaNoiseSamples())) {
      return;
    }
  }
  delete _sigmaNoise->graph();
  _sigmaNoise=nullptr;
  _sigmaNoisePick=nullptr;
  delete _sigmaNoiseHist;
  _sigmaNoiseHist=nullptr;
}

void ModeWidgets::setAzimuthHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Azimuth histogram\n"));
  _azimuthHist=param.histogram(param.azimuthSampling());
  if(_azimuthHist) {
    _azimuthHist->setYScaleType(Scale::Linear);
    if(_azimuthHist->setSamples(s->azimuthSamples())) {
      return;
    }
  }
  delete _azimuth->graph();
  _azimuth=nullptr;
  _azimuthPick=nullptr;
  delete _azimuthHist;
  _azimuthHist=nullptr;
}

void ModeWidgets::setPowerHistogram(const Samples * s, const Reader& param)
{
  TRACE;
  App::log(tr("Power histogram\n"));
  _powerHist=param.histogram(param.powerSampling());
  if(_powerHist) {
    _powerHist->setYScaleType(Scale::Log);
    if(_powerHist->setSamples(s->powerSamples())) {
      return;
    }
  }
  delete _power->graph();
  _power=nullptr;
  _powerPick=nullptr;
  delete _powerHist;
  _powerHist=nullptr;
}

void ModeWidgets::setLimits(bool normalize)
{
  TRACE;
  if(_dispersion) setLimits(_dispersion, normalize, true);
  if(_ellipticity) setLimits(_ellipticity, normalize, false);
  if(_verticalNoise) setLimits(_verticalNoise, normalize, true);
  if(_horizontalNoise) setLimits(_horizontalNoise, normalize, true);
  if(_totalNoise) setLimits(_totalNoise, normalize, true);
  if(_deltaNoise) setLimits(_deltaNoise, normalize, true);
  if(_sigmaNoise) setLimits(_sigmaNoise, normalize, true);
  if(_azimuth) setLimits(_azimuth, normalize, false);
  if(_power) setLimits(_power, normalize, true);
}

void ModeWidgets::setLimits(GraphContentsLayer * layer, bool normalize, bool setY)
{
  TRACE;
  GraphContents * gc=layer->graphContents();
  AxisWindow * w=gc->graph();
  Rect r=layer->boundingRect();
  r.enlarge(0.01, gc->scaleX().sampling(), gc->scaleY().sampling());
  w->xAxis()->setRange(r.x1(), r.x2());
  if(setY) {
    w->yAxis()->setRange(r.y1(), r.y2());
  }

  switch(_plotType) {
  case Reader::Histograms: {
      IrregularGrid2DPlot * histLayer=static_cast<IrregularGrid2DPlot *>(layer);
      ColorMap pal=histLayer->colorMap();
      double m=histLayer->grid().maximumValue();
      if(normalize) {
        pal.setValues(0.0001*m, m, SamplingParameters::Log);
      } else {
        pal.generate(0, qRound(m), ColorPalette::defaultSequentialWhite, ColorPalette::Reversed);
        pal.setWhiteTransparent(true);
        pal.setValues(0.5, m-0.5, SamplingParameters::Linear);
      }
      histLayer->setColorMap(pal);
    }
    break;
  case Reader::Dots:
    break;
  }

  w->updateGeometry(); // TopRight anchor imply that position can change when changing axis limits
}

void ModeWidgets::setColorMap(const ColorMap& map)
{
  if(_dispersion) _dispersion->setColorMap(map);
  if(_ellipticity) _ellipticity->setColorMap(map);
  if(_verticalNoise) _verticalNoise->setColorMap(map);
  if(_horizontalNoise) _horizontalNoise->setColorMap(map);
  if(_totalNoise) _totalNoise->setColorMap(map);
  if(_deltaNoise) _deltaNoise->setColorMap(map);
  if(_sigmaNoise) _sigmaNoise->setColorMap(map);
  if(_azimuth) _azimuth->setColorMap(map);
  if(_power) _power->setColorMap(map);
}

bool ModeWidgets::contains(AxisWindow * w) const
{
  return (_dispersionPick && w==_dispersionPick->graph()) ||
         (_ellipticityPick && w==_ellipticityPick->graph()) ||
         (_verticalNoisePick && w==_verticalNoisePick->graph()) ||
         (_horizontalNoisePick && w==_horizontalNoisePick->graph()) ||
         (_totalNoisePick && w==_totalNoisePick->graph()) ||
         (_deltaNoisePick && w==_deltaNoisePick->graph()) ||
         (_sigmaNoisePick && w==_sigmaNoisePick->graph()) ||
         (_azimuthPick && w==_azimuthPick->graph()) ||
         (_powerPick && w==_powerPick->graph());
}

LineLayer * ModeWidgets::pickLayer(AxisWindow * w) const
{
  if(_dispersionPick && w==_dispersionPick->graph()) {
    return _dispersionPick;
  }
  if(_ellipticityPick && w==_ellipticityPick->graph()) {
    return _ellipticityPick;
  }
  if(_verticalNoisePick && w==_verticalNoisePick->graph()) {
    return _verticalNoisePick;
  }
  if(_horizontalNoisePick && w==_horizontalNoisePick->graph()) {
    return _horizontalNoisePick;
  }
  if(_totalNoisePick && w==_totalNoisePick->graph()) {
    return _totalNoisePick;
  }
  if(_deltaNoisePick && w==_deltaNoisePick->graph()) {
     return _deltaNoisePick;
   }
  if(_sigmaNoisePick && w==_sigmaNoisePick->graph()) {
     return _sigmaNoisePick;
   }
  if(_azimuthPick && w==_azimuthPick->graph()) {
     return _azimuthPick;
   }
  if(_powerPick && w==_powerPick->graph()) {
     return _powerPick;
  }
  return nullptr;
}

CurveProxy * ModeWidgets::curveProxy(AxisWindow * w) const
{
  if(_dispersionPick && w==_dispersionPick->graph()) {
    return new DispersionProxy;
  }
  if(_ellipticityPick && w==_ellipticityPick->graph()) {
    return new EllipticityProxy;
  }
  if(_verticalNoisePick && w==_verticalNoisePick->graph()) {
    return new VerticalNoiseProxy;
  }
  if(_horizontalNoisePick && w==_horizontalNoisePick->graph()) {
    return new HorizontalNoiseProxy;
  }
  if(_totalNoisePick && w==_totalNoisePick->graph()) {
    return new TotalNoiseProxy;
  }
  if(_deltaNoisePick && w==_deltaNoisePick->graph()) {
     return new DeltaNoiseProxy;
   }
  if(_sigmaNoisePick && w==_sigmaNoisePick->graph()) {
     return new SigmaNoiseProxy;
   }
  if(_azimuthPick && w==_azimuthPick->graph()) {
     return new AzimuthProxy;
   }
  if(_powerPick && w==_powerPick->graph()) {
     return new PowerProxy;
   }
  return new RealStatisticalProxy;
}

CurvePlotProxy * ModeWidgets::curvePlotProxy(AxisWindow * w) const
{
  if(_dispersionPick && w==_dispersionPick->graph()) {
    return new DispersionPlotProxy;
  }
  if(_ellipticityPick && w==_ellipticityPick->graph()) {
    return new EllipticityPlotProxy;
  }
  if((_verticalNoisePick && w==_verticalNoisePick->graph()) ||
     (_horizontalNoisePick && w==_horizontalNoisePick->graph()) ||
     (_totalNoisePick && w==_totalNoisePick->graph())) {
    return new NoisePlotProxy;
  }
  if(_deltaNoisePick && w==_deltaNoisePick->graph()) {
    return new NoiseDeltaPlotProxy;
  }
  if(_sigmaNoisePick && w==_sigmaNoisePick->graph()) {
    return new NoiseSigmaPlotProxy;
  }
  if(_azimuthPick && w==_azimuthPick->graph()) {
    return new AzimuthPlotProxy;
  }
  if(_powerPick && w==_powerPick->graph()) {
    return new PowerPlotProxy;
  }
  return new RealStatisticalPlotProxy;
}

LineLayer * ModeWidgets::dispersionPickLayer() const
{
  return _dispersionPick;
}

LineLayer * ModeWidgets::ellipticityPickLayer() const
{
  return _ellipticityPick;
}

LineLayer * ModeWidgets::verticalNoisePickLayer() const
{
  return _verticalNoisePick;
}

LineLayer * ModeWidgets::horizontalNoisePickLayer() const
{
  return _horizontalNoisePick;
}

LineLayer * ModeWidgets::totalNoisePickLayer() const
{
  return _totalNoisePick;
}

LineLayer * ModeWidgets::deltaNoisePickLayer() const
{
  return _deltaNoisePick;
}

LineLayer * ModeWidgets::sigmaNoisePickLayer() const
{
  return _sigmaNoisePick;
}

LineLayer * ModeWidgets::azimuthPickLayer() const
{
  return _azimuthPick;
}

LineLayer * ModeWidgets::powerPickLayer() const
{
  return _powerPick;
}

Histogram2D * ModeWidgets::histogram(LineLayer * layer) const
{
  if(layer==_dispersionPick) {
    return _dispersionHist;
  } else if(layer==_ellipticityPick) {
    return _ellipticityHist;
  } else if(layer==_verticalNoisePick) {
    return _verticalNoiseHist;
  } else if(layer==_horizontalNoisePick) {
    return _horizontalNoiseHist;
  } else if(layer==_totalNoisePick) {
    return _totalNoiseHist;
  } else if(layer==_deltaNoisePick) {
    return _deltaNoiseHist;
  } else if(layer==_sigmaNoisePick) {
    return _sigmaNoiseHist;
  } else if(layer==_azimuthPick) {
    return _azimuthHist;
  } else if(layer==_powerPick) {
    return _powerHist;
  } else {
    return nullptr;
  }
}

/*!

*/
void ModeWidgets::classifySamples(bool normalize)
{
  TRACE;
  Histogram2D * hist=histogram(_curveBrowser->currentLayer());
  if(!hist) {
    return;
  }
  if(_plotType==Reader::Dots) {
    if(SampleClassification::run(_curveBrowser, &_categories->legend(), hist)) {
      updateHistograms(hist, normalize);
    }
  } else {
    if(SampleClassification::run(_curveBrowser, nullptr, hist)) {
      updateHistograms(hist, normalize);
    }
  }
}

void ModeWidgets::classifySamples(const ClassificationParameters& param, bool normalize)
{
  Histogram2D * hist=nullptr;
  switch(param.data()) {
  case Reader::Dispersion:
    hist=_dispersionHist;
    break;
  case Reader::Ellipticity:
    hist=_ellipticityHist;
    break;
  case Reader::VerticalNoise:
    hist=_verticalNoiseHist;
    break;
  case Reader::HorizontalNoise:
    hist=_horizontalNoiseHist;
    break;
  case Reader::TotalNoise:
    hist=_totalNoiseHist;
    break;
  case Reader::DeltaNoise:
    hist=_deltaNoiseHist;
    break;
  case Reader::SigmaNoise:
    hist=_sigmaNoiseHist;
    break;
  case Reader::Azimuth:
    hist=_azimuthHist;
    break;
  case Reader::Power:
    hist=_powerHist;
    break;
  }
  if(!hist) {
    return;
  }
  CurvePropertiesWidget * p;
  p=_curveBrowser->find(param.curve1());
  CurveProxy * proxy1=p ? p->proxy() : nullptr;
  p=_curveBrowser->find(param.curve2());
  CurveProxy * proxy2=p ? p->proxy() : nullptr;
  hist->classifySamples(param, proxy1, proxy2);
  updateHistograms(hist, normalize);
}

void ModeWidgets::updatePlots()
{
  if(_dispersion) {
    _dispersion->deepUpdate();
  }
  if(_ellipticity) {
    _ellipticity->deepUpdate();
  }
  if(_verticalNoise) {
    _verticalNoise->deepUpdate();
  }
  if(_horizontalNoise) {
    _horizontalNoise->deepUpdate();
  }
  if(_totalNoise) {
    _totalNoise->deepUpdate();
  }
  if(_deltaNoise) {
    _deltaNoise->deepUpdate();
  }
  if(_sigmaNoise) {
    _sigmaNoise->deepUpdate();
  }
  if(_azimuth) {
    _azimuth->deepUpdate();
  }
  if(_power) {
    _power->deepUpdate();
  }
}

void ModeWidgets::updateHistogram(Histogram2D * destHist, GraphContentsLayer * layer,
                                  Histogram2D * srcHist, bool normalize)
{
  if(destHist) {
    if(srcHist!=destHist) {
      destHist->setClassification(srcHist);
    }
    if(normalize) {
      destHist->normalizeArea(YAxis);
    }
    if(layer) {
      if(_plotType==Reader::Histograms) {
        static_cast<IrregularGrid2DPlot *>(layer)->setGrid(*destHist);
      }
      layer->deepUpdate();
    }
  }
}

void ModeWidgets::updateHistograms(Histogram2D * hist, bool normalize)
{
  updateHistogram(_dispersionHist, _dispersion, hist, normalize);
  updateHistogram(_ellipticityHist, _ellipticity, hist, normalize);
  updateHistogram(_verticalNoiseHist, _verticalNoise, hist, normalize);
  updateHistogram(_horizontalNoiseHist, _horizontalNoise, hist, normalize);
  updateHistogram(_totalNoiseHist, _totalNoise, hist, normalize);
  updateHistogram(_deltaNoiseHist, _deltaNoise, hist, normalize);
  updateHistogram(_sigmaNoiseHist, _sigmaNoise, hist, normalize);
  updateHistogram(_azimuthHist, _azimuth, hist, normalize);
  updateHistogram(_powerHist, _power, hist, normalize);
}

void ModeWidgets::adjust()
{
  TRACE;
  Histogram2D * hist=histogram(_curveBrowser->currentLayer());
  if(!hist) {
    return;
  }
  int index=_curveBrowser->currentLine();
  CurveProxy * proxy=_curveBrowser->curveAt(index)->proxy();
  proxy->sort();
  proxy->followMaximumX(hist, proxy->minimumX(), proxy->maximumX());
}

Histogram2D * ModeWidgets::categorizedHistogram(LineLayer * layer)
{
  TRACE;
  Histogram2D * hist=histogram(layer);
  if(!hist) {
    return nullptr;
  }
  if(_plotType==Reader::Dots) {
    Dialog * d=new Dialog;
    CategoryChooser * cc=new CategoryChooser;
    cc->setCategories(_categories->legend());
    d->setWindowTitle(tr("Choose category"));
    d->setMainWidget(cc);
    Settings::getWidget(d, "ModeWidgets::addCurve");
    cc->updateAllFields();
    if(d->exec()==QDialog::Accepted) {
      Settings::setWidget(d, "ModeWidgets::addCurve");
      Histogram2D * catHist=new Histogram2D(*hist);
      SampleClassificationParameters param;
      param.setValueConditionType(SampleClassificationParameters::NoCondition);
      param.setCategoryConditionType(SampleClassificationParameters::AllCategories);
      param.setAction(SampleClassificationParameters::Invalidate);
      catHist->classifySamples(param);
      param.setCategoryConditionType(cc->type());
      param.setCategoryCondition(cc->category());
      param.setAction(SampleClassificationParameters::Validate);
      catHist->classifySamples(param);
      delete d;
      return catHist;
    } else {
      delete d;
      return nullptr;
    }
  } else {
    return new Histogram2D(*hist);
  }
}

void ModeWidgets::addMeanCurve()
{
  TRACE;
  LineLayer * layer=_curveBrowser->currentLayer();
  Histogram2D * hist=categorizedHistogram(layer);
  if(hist) {
    AbstractLine * line=layer->addLine();
    _curveBrowser->addLine(line);
    CurvePropertiesWidget * p=_curveBrowser->properties(line);
    _curveBrowser->setCurveName(tr("mean%1"), p);
    p->beginCurveChange();
    hist->meanCurve(p->proxy());
    p->endCurveChange();
    delete hist;
  }
}

void ModeWidgets::addMedianCurve()
{
  TRACE;
  LineLayer * layer=_curveBrowser->currentLayer();
  Histogram2D * hist=categorizedHistogram(layer);
  if(hist) {
    AbstractLine * line=layer->addLine();
    _curveBrowser->addLine(line);
    CurvePropertiesWidget * p=_curveBrowser->properties(line);
    _curveBrowser->setCurveName(tr("median%1"), p);
    p->beginCurveChange();
    hist->medianCurve(p->proxy());
    p->endCurveChange();
    delete hist;
  }
}

void ModeWidgets::addModeCurve()
{
  TRACE;
  LineLayer * layer=_curveBrowser->currentLayer();
  Histogram2D * hist=categorizedHistogram(layer);
  if(hist) {
    AbstractLine * line=layer->addLine();
    _curveBrowser->addLine(line);
    CurvePropertiesWidget * p=_curveBrowser->properties(line);
    _curveBrowser->setCurveName(tr("mode%1"), p);
    p->beginCurveChange();
    hist->modeCurve(p->proxy());
    p->endCurveChange();
    delete hist;
  }
}

void ModeWidgets::addSmartMeanCurve()
{
  TRACE;

  LineLayer * layer=_curveBrowser->currentLayer();
  Histogram2D * hist=categorizedHistogram(layer);
  if(hist) {
    AbstractLine * line=layer->addLine();
    _curveBrowser->addLine(line);
    CurvePropertiesWidget * p=_curveBrowser->properties(line);
    _curveBrowser->setCurveName(tr("mean%1"), p);
    p->beginCurveChange();
    hist->meanCurve(p->proxy());
    p->endCurveChange();
    delete hist;
  }
}

void ModeWidgets::addGaussianMixtureCurve()
{
  TRACE;
  _gaussianMixtureLines[0]=_dispersionPick->addLine();
  _gaussianMixtureLines[0]->setPen(Pen(Qt::NoPen));
  _gaussianMixtureLines[0]->setSymbol(Symbol(Symbol::Circle, 1.2,
                                       Pen(Qt::black, 0.6, Pen::SolidLine),
                                       Qt::NoBrush));
  _curveBrowser->addLine(_gaussianMixtureLines[0], _dispersionPick);
  _gaussianMixtureLines[1]=_ellipticityPick->addLine();
  _gaussianMixtureLines[1]->setPen(Pen(Qt::NoPen));
  _gaussianMixtureLines[1]->setSymbol(Symbol(Symbol::Circle, 1.2,
                                       Pen(Qt::black, 0.6, Pen::SolidLine),
                                       Qt::NoBrush));
  _curveBrowser->addLine(_gaussianMixtureLines[1], _ellipticityPick);
  _gaussianMixtureLines[2]=_ellipticityPick->addLine();
  /*_gaussianMixtureLines[2]->setPen(Pen(Qt::NoPen));
  _gaussianMixtureLines[2]->setSymbol(Symbol(Symbol::Circle, 1.2,
                                       Pen(Qt::black, 0.6, Pen::SolidLine),
                                       Qt::NoBrush));
  _curveBrowser->addLine(_gaussianMixtureLines[2], _sigmaNoisePick);*/

  GaussianMixtureEM em;
  //int oneFrequency=_dispersionHist->xSamplingParameters().index(1.40);
  int oneFrequency=-1;
  int nIndex=_dispersionHist->xSamplingParameters().count();
  for(int currentIndex=0; currentIndex<nIndex; currentIndex++) {
    if(oneFrequency>-1 && oneFrequency!=currentIndex) {
      continue;
    }
    MultivariateHistogram histogram(2);
    histogram.setSampling(0, _dispersionHist->ySamplingParameters());
    histogram.setSampling(1, _ellipticityHist->ySamplingParameters());
    PrivateVector<double> x(2);
    const Histogram2D& hist0=*_dispersionHist;
    for(int is=_dispersionHist->sampleCount()-1; is>=0; is--) {
      const Histogram2D::Sample& s=hist0.sample(is);
      if(s.isValid() && hist0.indexOfX(s.x())==currentIndex) {
        if(_dispersionHist->ySampling() & LogScale) {
          x[0]=log(_dispersionHist->sample(is).y());
        } else if(_ellipticityHist->ySampling() & InverseScale) {
          x[0]=1.0/_dispersionHist->sample(is).y();
        } else {
          x[0]=_dispersionHist->sample(is).y();
        }
        if(_ellipticityHist->ySampling() & LogScale) {
          x[1]=log(_ellipticityHist->sample(is).y());
        } else if(_ellipticityHist->ySampling() & InverseScale) {
          x[1]=1.0/_ellipticityHist->sample(is).y();
        } else {
          x[1]=_ellipticityHist->sample(is).y();
        }
        histogram.addSample(x);
      }
    }
    App::log(tr("%1 buckets in multivariate histogram (%2 samples)\n")
             .arg(histogram.bucketCount()).arg(histogram.sampleCount()));
    histogram.removeSmallBucketValues(1.0);
    App::log(tr("  --> %1 buckets in multivariate histogram (%2 samples)\n")
             .arg(histogram.bucketCount()).arg(histogram.sampleCount()));
    if(!histogram.isEmpty()) {
      em.init(&histogram, 4);
      PrivateVector<double> min(2);
      PrivateVector<double> max(2);
      PrivateVector<double> stddev(2);
      min[0]=log(_dispersionHist->ySamplingParameters().minimum());
      max[0]=log(_dispersionHist->ySamplingParameters().maximum());
      stddev[0]=0.06;
      min[1]=-90.0;
      max[1]=90.0;
      stddev[1]=15.0;
      em.uniform(min, max, stddev);
      for(int i=0; i<100; i++) {
        em.iterate();
        if(oneFrequency==-1 || oneFrequency!=currentIndex) {
          continue;
        }
        QTextStream(stdout) << em.distribution().toString(false) << Qt::endl;
        //GaussianMixtureDistribution d(em.distribution());
        //d.bestWeights(2);
        // Remove modes with large standard deviation and small amplitude
        //d.filterDominantModes(2.0);
        // Remove small weights
        /*d.filterWeight(0.05);
        double freq=_dispersionHist->x(currentIndex);
        modeToPoints(d, 0, freq, _dispersionHist->ySampling(),
                     curveProxy(_dispersionPick->graph()),
                     curvePlotProxy(_dispersionPick->graph()));
        _dispersionPick->graph()->deepUpdate();
        modeToPoints(d, 1, freq, _ellipticityHist->ySampling(),
                     curveProxy(_ellipticityPick->graph()),
                     curvePlotProxy(_dispersionPick->graph()));
        _ellipticityPick->graph()->deepUpdate();*/
        //char rep[10];
        //scanf("%s", rep);
      }
      GaussianMixtureDistribution d(em.distribution());
      d.sortMean();
      d.unique();
      d.sortWeight();
      d.bestWeights(2);
      QTextStream(stdout) << d.toString(false) << Qt::endl;
      //GaussianMixtureDistribution d(em.distribution());
      // Remove modes with large standard deviation and small amplitude
      //d.filterDominantModes(2.0);
      // Remove small weights
      //d.filterWeight(0.05);
      double freq=_dispersionHist->x(currentIndex);
      modeToPoints(d, 0, freq, _dispersionHist->ySampling(),
                   curveProxy(_dispersionPick->graph()),
                   curvePlotProxy(_dispersionPick->graph()));
      _dispersionPick->graph()->deepUpdate();
      modeToPoints(d, 1, freq, _ellipticityHist->ySampling(),
                   curveProxy(_ellipticityPick->graph()),
                   curvePlotProxy(_dispersionPick->graph()));
      _ellipticityPick->graph()->deepUpdate();
    }
  }

  /*(delete _gaussianMixtureInversion;
  _gaussianMixtureInversion=new HistogramInversion(3);
  _gaussianMixtureInversion->setHistogram(0, _dispersionHist);
  _gaussianMixtureInversion->setHistogram(1, _ellipticityHist);
  _gaussianMixtureInversion->setHistogram(2, _sigmaNoiseHist);
  _gaussianMixtureInversion->setParameters(_gaussianMixtureParameters);
  connect(_gaussianMixtureInversion, SIGNAL(histogramFinished(int)),
          this, SLOT(addGaussianMixture(int)));
  _gaussianMixtureInversion->start();*/
}

GaussianMixtureDistribution ModeWidgets::filterGaussianMixture(int index)
{
  TRACE;
  double freq=_dispersionHist->x(index);
  App::freeze(false);
  App::log(tr("Best misfit %1 at x %2 Hz (model index %3 over %4)\n")
                   .arg(_gaussianMixtureInversion->bestMisfit())
                   .arg(freq)
                   .arg(_gaussianMixtureInversion->bestMisfitIndex().value())
                   .arg(_gaussianMixtureInversion->modelCount()));
  App::log(_gaussianMixtureInversion->humanSolution(index, "# "));

  GaussianMixtureDistribution d=_gaussianMixtureInversion->distribution(index);
  // Remove modes with large standard deviation and small amplitude
  d.filterDominantModes(2.0);
  // Remove small weights
  d.filterWeight(0.01);
  // Exclude outside kmin and kmax
  if(_gaussianMixtureInversion->parameters().kmin()>0.0 &&
     _gaussianMixtureInversion->parameters().kmax()>_gaussianMixtureInversion->parameters().kmin()) {
    // Remove 1% to account for precision errors in parameter discretization
    // We want to remove solution that at too close to limits
    double invOmega=0.5/(M_PI*freq);
    SamplingOptions samp=_dispersionHist->ySampling();
    double min=1.01*invOmega*_gaussianMixtureInversion->parameters().kmin();
    double max=0.99*invOmega*_gaussianMixtureInversion->parameters().kmax();
    if(samp & LogScale) {
      d.filterRange(0, log(min), log(max));
    } else if(samp & InverseScale) {
      d.filterRange(0, 1.0/max, 1.0/min);
    } else {
      d.filterRange(0, min, max);
    }
  }
  App::freeze(true);
  return d;
}

void ModeWidgets::modeToPoints(const GaussianMixtureDistribution& d,
                               int histIndex, double x, SamplingOptions samp,
                               CurveProxy * proxy, CurvePlotProxy * plotProxy)
{
  TRACE;
  _curveBrowser->beginCurveChange(_gaussianMixtureLines[histIndex]);
  plotProxy->setCurve(_gaussianMixtureLines[histIndex], proxy);
  int i0=proxy->sampleCount();
  proxy->resize(i0+d.modeCount());
  //int i0=0;
  //proxy->resize(d.modeCount());
  for(int i=0; i<d.modeCount(); i++) {
    int index=i0+i;
    proxy->setX(index, x);
    if(samp & LogScale) {
      proxy->setY(index, exp(d.mode(i).mean(histIndex)), nullptr);
      proxy->setStddev(index, exp(d.mode(i).stddev(histIndex)));
    } else if(samp & InverseScale) {
      proxy->setY(index, 1.0/d.mode(i).mean(histIndex), nullptr);
      proxy->setStddev(index, 1.0/d.mode(i).stddev(histIndex));
    } else {
      proxy->setY(index, d.mode(i).mean(histIndex), nullptr);
      proxy->setStddev(index, d.mode(i).stddev(histIndex));
    }
    proxy->setWeight(index, d.weight(i));
  }
  _curveBrowser->endCurveChange(_gaussianMixtureLines[histIndex]);
  delete proxy;
  delete plotProxy;
}

void ModeWidgets::addGaussianMixture(int index)
{
  TRACE;
  if(!_gaussianMixtureLines[0]) {
    return;
  }
  GaussianMixtureDistribution d=filterGaussianMixture(index);
  double x=_dispersionHist->x(index);

  modeToPoints(d, 0, x, _dispersionHist->ySampling(),
               curveProxy(_dispersionPick->graph()),
               curvePlotProxy(_dispersionPick->graph()));
  _dispersionPick->graph()->deepUpdate();
  modeToPoints(d, 1, x, _ellipticityHist->ySampling(),
               curveProxy(_ellipticityPick->graph()),
               curvePlotProxy(_dispersionPick->graph()));
  _ellipticityPick->graph()->deepUpdate();
}

void ModeWidgets::exportSamples()
{
  TRACE;
  LineLayer * layer=_curveBrowser->currentLayer();
  Histogram2D * hist=categorizedHistogram(layer);
  if(hist) {
    QString fileName=Message::getSaveFileName(tr("Save samples"),
                                              tr("Max file (*.max)"));
    if(!fileName.isEmpty()) {
      _samples->setValid(*hist);
      _samples->save(fileName);
    }
    delete hist;
  }
}

AbstractLine * ModeWidgets::addMedianCurve(const Histogram2D * hist, LineLayer * layer, int category)
{
  TRACE;
  if(layer) {
    Histogram2D * catHist=new Histogram2D(*hist);
    SampleClassificationParameters param;
    param.setValueConditionType(SampleClassificationParameters::NoCondition);
    param.setCategoryConditionType(SampleClassificationParameters::AllCategories);
    param.setAction(SampleClassificationParameters::Invalidate);
    catHist->classifySamples(param);
    param.setCategoryConditionType(SampleClassificationParameters::OneCategory);
    param.setCategoryCondition(category);
    param.setAction(SampleClassificationParameters::Validate);
    catHist->classifySamples(param);
    if(catHist) {
      AbstractLine * line=layer->addLine();
      _curveBrowser->addLine(line, layer);
      _curveBrowser->beginCurveChange(line);
      CurveProxy * proxy=curveProxy(layer->graph());
      CurvePlotProxy * plotProxy=curvePlotProxy(layer->graph());
      plotProxy->setCurve(line, proxy);
      catHist->medianCurve(proxy);
      _curveBrowser->endCurveChange(line);
      delete proxy;
      delete plotProxy;
      delete catHist;
      return line;
    }
  }
  return nullptr;
}

void ModeWidgets::filter(Histogram2D * hist, LineLayer * layer,
                         AbstractLine * line, int category)
{
  TRACE;
  if(layer && line) {
    CurveProxy * proxy=curveProxy(layer->graph());
    CurvePlotProxy * plotProxy=curvePlotProxy(layer->graph());
    plotProxy->setCurve(line, proxy);
    SampleClassificationParameters param;
    param.setValueConditionType(SampleClassificationParameters::CurveStddevOutsideRange);
    param.setStddevRange(2.0);
    param.setCategoryConditionType(SampleClassificationParameters::OneCategory);
    param.setCategoryCondition(category);
    param.setAction(SampleClassificationParameters::Categorize);
    param.setCategory(0);
    hist->classifySamples(param, proxy);
    delete proxy;
    delete plotProxy;
    updateHistograms(hist, true);
  }
}

void ModeWidgets::pickToMean()
{
  TRACE;
  SampleClassificationParameters paramSelect;
  paramSelect.setValueConditionType(SampleClassificationParameters::CurveRelativeInsideRange);
  paramSelect.setRelativeRange(0.3);
  paramSelect.setCategoryConditionType(SampleClassificationParameters::OneCategory);
  paramSelect.setCategory(0);
  paramSelect.setAction(SampleClassificationParameters::Categorize);
  DispersionProxy dispersionProxy;
  DispersionPlotProxy dispersionPlotProxy;

  int n=_dispersionPick->count();
  for(int i=0; i<n; i++) {
    App::log(tr("Gathering samples for mode %1 (30% around curve)\n").arg(i+1));
    dispersionPlotProxy.setCurve(_dispersionPick->line(i), &dispersionProxy);
    paramSelect.setCategory(i+1);
    _dispersionHist->classifySamples(paramSelect, &dispersionProxy);
    updateHistograms(_dispersionHist, true);
  }

  AbstractLine * medianLines[7];
  for(int i=1; i<=n; i++) {
    App::log(tr("Calculate median and median deviation for mode %1\n").arg(i));
    dispersionPlotProxy.setCurve(_dispersionPick->line(i-1), &dispersionProxy);
    medianLines[0]=addMedianCurve(_dispersionHist, _dispersionPick, i);
    medianLines[1]=addMedianCurve(_ellipticityHist, _ellipticityPick, i);
    medianLines[2]=addMedianCurve(_verticalNoiseHist, _verticalNoisePick, i);
    medianLines[3]=addMedianCurve(_horizontalNoiseHist, _horizontalNoisePick, i);
    medianLines[4]=addMedianCurve(_totalNoiseHist, _totalNoisePick, i);
    medianLines[5]=addMedianCurve(_deltaNoiseHist, _deltaNoisePick, i);
    medianLines[6]=addMedianCurve(_sigmaNoiseHist, _sigmaNoisePick, i);
    App::log(tr("Excluding outliers (>2 sigma) for mode %1\n").arg(i));
    filter(_dispersionHist, _dispersionPick, medianLines[0], i);
    filter(_ellipticityHist, _ellipticityPick, medianLines[1], i);
    filter(_verticalNoiseHist, _verticalNoisePick, medianLines[2], i);
    filter(_horizontalNoiseHist, _horizontalNoisePick, medianLines[3],i);
    filter(_totalNoiseHist, _totalNoisePick, medianLines[4], i);
    filter(_deltaNoiseHist, _deltaNoisePick, medianLines[5], i);
    filter(_sigmaNoiseHist, _sigmaNoisePick, medianLines[6], i);
    _curveBrowser->remove(_dispersionPick, medianLines[0]);
    _curveBrowser->remove(_ellipticityPick, medianLines[1]);
    _curveBrowser->remove(_verticalNoisePick, medianLines[2]);
    _curveBrowser->remove(_horizontalNoisePick, medianLines[3]);
    _curveBrowser->remove(_totalNoisePick, medianLines[4]);
    _curveBrowser->remove(_deltaNoisePick, medianLines[5]);
    _curveBrowser->remove(_sigmaNoisePick, medianLines[6]);
    App::log(tr("Calculate new median and median deviation for mode %1\n").arg(i));
    medianLines[0]=addMedianCurve(_dispersionHist, _dispersionPick, i);
    medianLines[1]=addMedianCurve(_ellipticityHist, _ellipticityPick, i);
    medianLines[2]=addMedianCurve(_verticalNoiseHist, _verticalNoisePick, i);
    medianLines[3]=addMedianCurve(_horizontalNoiseHist, _horizontalNoisePick, i);
    medianLines[4]=addMedianCurve(_totalNoiseHist, _totalNoisePick, i);
    medianLines[5]=addMedianCurve(_deltaNoiseHist, _deltaNoisePick, i);
    medianLines[6]=addMedianCurve(_sigmaNoiseHist, _sigmaNoisePick, i);
    _curveBrowser->setCurveName(_dispersionPick, medianLines[0], dispersionProxy.name(),
                                new DispersionProxy, new DispersionPlotProxy);
    _curveBrowser->setCurveName(_ellipticityPick, medianLines[1], dispersionProxy.name(),
                                new EllipticityProxy, new EllipticityPlotProxy);
    _curveBrowser->setCurveName(_verticalNoisePick, medianLines[2], dispersionProxy.name(),
                                new VerticalNoiseProxy, new NoisePlotProxy);
    _curveBrowser->setCurveName(_horizontalNoisePick, medianLines[3], dispersionProxy.name(),
                                new HorizontalNoiseProxy, new NoisePlotProxy);
    _curveBrowser->setCurveName(_totalNoisePick, medianLines[4], dispersionProxy.name(),
                                new TotalNoiseProxy, new NoisePlotProxy);
    _curveBrowser->setCurveName(_deltaNoisePick, medianLines[5], dispersionProxy.name(),
                                new DeltaNoiseProxy, new NoiseDeltaPlotProxy);
    _curveBrowser->setCurveName(_sigmaNoisePick, medianLines[6], dispersionProxy.name(),
                                new SigmaNoiseProxy, new NoiseSigmaPlotProxy);
  }
}
