/***************************************************************************
**
**  This file is part of gpcoord.
**
**  gpcoord is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpcoord is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-11-06
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "TranslationRotationForward.h"

/*!
  \class TranslationRotationForward TranslationRotationForward.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
TranslationRotationForward::TranslationRotationForward()
    : AbstractForward()
{
  TRACE;
}

/*!
  Description of constructor still missing
*/
TranslationRotationForward::TranslationRotationForward(const TranslationRotationForward& o)
    : AbstractForward()
{
  TRACE;
  _refPoints=o._refPoints;
  _points=o._points;
}

void TranslationRotationForward::setFixedTranslation()
{
  QMap<QString, Point>::iterator itRef;
  int n=_points.count();
  Point2D center, refCenter;
  int count=0;
  for(int i=0; i<n; i++) {
    const NamedPoint& p=_points.at(i);
    itRef=_refPoints.find(p.name());
    if(itRef!=_refPoints.end()) {
      center+=p;
      refCenter+=itRef.value();
      count++;
    }
  }
  center/=count;
  // Calculate maximum distance from center, used to adjust admissible range for translations
  _maxDistance=0.0;
  for(int i=0; i<n; i++) {
    const NamedPoint& p=_points.at(i);
    itRef=_refPoints.find(p.name());
    if(itRef!=_refPoints.end()) {
      double d=p.distanceTo(center);
      if(d>_maxDistance) {
        _maxDistance=d;
      }
    }
  }
  refCenter/=count;
  _fixedTranslation=refCenter-center;
  App::log(1, tr("Maximum distance= %1\n").arg(_maxDistance));
  App::log(1, tr("Fixed translation %1").arg(_fixedTranslation.toString('f', 3)));
}

AbstractForward * TranslationRotationForward::clone() const
{
  TRACE;
  TranslationRotationForward * f=new TranslationRotationForward(*this);
  f->setFixedTranslation();
  f->parameterSpace().addParameter("dx", "m", -f->_maxDistance, f->_maxDistance, ParameterGrid::Linear, 0.001);
  f->parameterSpace().addParameter("dy", "m", -f->_maxDistance, f->_maxDistance, ParameterGrid::Linear, 0.001);
  f->parameterSpace().addParameter("r", "deg", 0.0, 360.0, ParameterGrid::Linear, 0.001);
  return f;
}

double TranslationRotationForward::misfit(double * model, bool&)
{
  TRACE;
  Angle dr;
  dr.setDegrees(model[2]);
  QMap<QString, Point>::iterator itRef;
  double rms_val=0.0;
  int n=_points.count();
  Point2D totalTranslation=_fixedTranslation+Point2D(model[0], model[1]);
  for(int i=0; i<n; i++) {
    const NamedPoint& p=_points.at(i);
    itRef=_refPoints.find(p.name());
    if(itRef!=_refPoints.end()) {
      Point2D p2(p);
      p2.rotate(dr);
      p2.translate(totalTranslation);
      rms_val+=p2.distanceTo(itRef.value());
    }
  }
  return rms_val;
}

void TranslationRotationForward::reportErrors(QTextStream& s, double dx, double dy, double rot) const
{
  Angle dr;
  dr.setDegrees(rot);
  QMap<QString, Point>::const_iterator itRef;
  int n=_points.count();
  Point2D totalTranslation=_fixedTranslation+Point2D(dx, dy);
  for(int i=0; i<n; i++) {
    const NamedPoint& p=_points.at(i);
    itRef=_refPoints.find(p.name());
    if(itRef!=_refPoints.end()) {
      Point2D p2(p);
      p2.rotate(dr);
      p2.translate(totalTranslation);
      s << tr("%1: %2 m @ %3 deg from north\n")
               .arg(p.name(), 10)
               .arg(p2.distanceTo(itRef.value()))
               .arg(itRef.value().geographicalAzimuthTo(p2));
    }
  }
}
