/***************************************************************************
**
**  This file is part of gpcoord.
**
**  gpcoord is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpcoord is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-06-19
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include <ArrayCore.h>

#include "CoordReader.h"
#include "TranslationRotationForward.h"

/*!
  \class CoordReader CoordReader.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
CoordReader::CoordReader()
    : ArgumentStdinReader(),
    _circleCount(1),
    _radiusFactor(3.0),
    _minimumDistance(0.0),
    _maximumStep(0.0),
    _maximumDeviation(0.0)
{
  _mode=None;
  _angleMode=Angle::Degrees;
  _averageCount=0;
  _kml=nullptr;
  _distanceStddev=0.01; // 1 cm
  _priorPrecision=5.0;  // 2 m
  _posteriorPrecision=0;  // No posterior precision specified
  _fromDistancesOutMode=NoOutput;
  _progress=new ConsoleProgress;
  _clusterCount=500;
  _sensorCount=15;
  _stepCount=2;
  _firstIndex=0;
  _maximumRadius=100.0;
  _ns0=10;
  _ns=10000;
  _nr=5;
}

CoordReader::~CoordReader()
{
  delete _progress;
}

bool CoordReader::terminate()
{
  QTextStream sOut(stdout);
  sOut.setRealNumberPrecision(20);
  switch(_mode) {
  case Distance:
  case DistanceLookup:
  case DistanceGeo:
  case Azimuth:
  case AzimuthGeo:
  case AzimuthLookup:
  case FromGeo:
  case ToGeo:
  case FromDMS:
  case ToDMS:
  case FromDM:
  case ToDM:
  case ToUTM:
  case FromUTM:
  case Translate:
  case Rotate:
  case Project:
  case Localize:
  case LineMove:
  case Lookup:
  case UTMDeclination:
  case MagDeclination:
  case AllDistances:
  case None:
    break;
  case GenerateCircles:
    _sensorCount=_sensorCount/_circleCount.value();
    if(_sensorCount<3) {
      _sensorCount=3;
    }
    return generateCircles();
  case GenerateTriangles:
    if(!_circleCount.isSet()) {
      if(_firstIndex==0) {
        _circleCount.setValue((_sensorCount-1)/3);
      } else {
        _circleCount.setValue(_sensorCount/3);
      }
      if(_circleCount.value()<1) {
        _circleCount.setValue(1);
      }
    }
    if(!_radiusFactor.isSet()) {
      _radiusFactor.setValue(2.0);
    }
    _sensorCount=3;
    return generateCircles();
  case GenerateRandom:
    if(!_minimumDistance.isSet()) {
      _minimumDistance.setValue(_maximumRadius/ceil(sqrt(_sensorCount)));
    }
    return generateRandom();
  case GenerateMarkovChain:
    if(!_minimumDistance.isSet()) {
      _minimumDistance.setValue(_maximumRadius/ceil(sqrt(_sensorCount)));
    }
    if(!_maximumStep.isSet()) {
      _maximumStep.setValue(_maximumRadius/ceil(sqrt(_sensorCount)));
    }
    return generateMarkovChain();
  case GenerateGridCircle:
    return generateGridCircle();
  case GenerateRandomGridCircle:
    if(!_maximumDeviation.isSet()) {
      _maximumDeviation.setValue(_maximumRadius/ceil(sqrt(_sensorCount)));
    }
    return generateRandomGridCircle();
  case TriangulateDistance:
    return triangulateDistance();
  case TriangulateAzimuth:
    if(_lines.count()>=2) {
      bool ok;
      int n=0;
      Point2D pAver(0, 0), pDev(0, 0);
      for(int i=0; i<_lines.count(); i++) {
        for(int j=i+1; j<_lines.count(); j++) {
          Point2D p=_lines.at(i).intersect(_lines.at(j), ok);
          if(ok) {
            pAver+=p;
            pDev+=p*p;
            n++;
          }
        }
      }
      if(n>0) {
        pAver/=n;
        pDev/=n;
        pDev-=pAver * pAver;
      } else {
        App::log(tr("No intersection\n") );
        return false;
      }
      sOut << pAver.x() << " " << pAver.y() << " "
           << sqrt(pDev.x()) << " " << sqrt(pDev.y()) << Qt::endl;
    } else {
      App::log(tr("At least two azimuth are required to triangulate.\n") );
      return false;
    }
    break;
  case Average:
    if(_averageCount>0) {
      _average/=static_cast<double>(_averageCount);
      sOut << _average.toString('g', 20) << Qt::endl;
    } else {
      App::log(tr("No point available to calculate average\n") );
      return false;
    }
    break;
  case ToKML:
    _kml->save(_kmlFileName);
    delete _kml;
    break;
  case FromKML: {
      GoogleEarthKML kml;
      XMLHeader hdr(&kml);
      if(hdr.xml_restoreFile(_kmlFileName, 0, XMLClass::XmlFile)!=XMLClass::NoError) {
        App::log(tr("Error while reading file %1\n").arg(_kmlFileName) );
        return false;
      }
      QList<NamedPoint> list;
      kml.document().collectPoints(list);
      for(QList<NamedPoint>::iterator it=list.begin(); it!=list.end(); it++) {
        sOut << (*it).toString('g', 20) << Qt::endl;
      }
    }
    break;
  case FromDistances: {
      Node * origin=_fromDistanceFactory.node(_origin);
      if(!origin) {
        App::log(tr("origin node '%1' does not exist\n").arg(_origin) );
        return false;
      }
      Node * north=_fromDistanceFactory.node(_north);
      if(!north) {
        App::log(tr("north node '%1' does not exist\n").arg(_north) );
        return false;
      }
      Node * eastward=_fromDistanceFactory.node(_eastward);
      if(!eastward) {
        App::log(tr("eastward node '%1' does not exist\n").arg(_eastward) );
        return false;
      }
      if(_posteriorPrecision>0.0) {
        _fromDistanceFactory.setPosteriorPrecision(_posteriorPrecision);
      }
      CoreApplication::instance()->debugUserInterrupts(false);
      _fromDistanceFactory.aggregate(_clusterCount, _distanceStddev);
      if(_progress) {
        _progress->end(_fromDistanceFactory.clusters().count());
      }
      CoreApplication::instance()->debugUserInterrupts(true);
      _fromDistanceFactory.setCoordinates(origin, north, eastward);
      if(!_refPoints.isEmpty() ) {
        _fromDistanceFactory.setPriorCoordinates(_refPoints, _priorPrecision);
      }
      QList<Node *> nodes=_fromDistanceFactory.nodes();
      for(QList<Node *>::iterator it=nodes.begin(); it!=nodes.end(); it++) {
        Node * n=*it;
        if(_fromDistancesOutMode & Raw) {
          QSet<Point2D> coordinates=_fromDistanceFactory.solutions(n->name());
          for(QSet<Point2D>::iterator its=coordinates.begin(); its!=coordinates.end(); its++) {
            sOut << its->toString() << " 0 " << n->name() << Qt::endl;
          }
        }
        if(_fromDistancesOutMode & (Mean | Ellipse)) {
          Covariance cov=_fromDistanceFactory.covariance(n->name());
          if(_fromDistancesOutMode & Ellipse) {
            sOut << cov.stddev2D().toString() << Qt::endl;
          }
          if(_fromDistancesOutMode & Mean) {
            sOut << cov.mean(0) << " " << cov.mean(1) << " 0 " <<  n->name() << Qt::endl;
          }
        }
      }
      if(_fromDistancesOutMode & Clusters) {
        _fromDistanceFactory.printClusters();
      }
    }
    break;
  case ArrayLimits: {
      double kmin, kmax;
      if(arrayLimits(kmin, kmax)) {
        sOut << "#kmin kmax [rad/m, according to Wathelet et al. 2008]" << Qt::endl;
        sOut << kmin << " " << kmax << Qt::endl;
      }
    }
    break;
  case ArrayPattern: {
      double kmin, kmax;
      if(arrayLimits(kmin, kmax)) {
        TheoreticalFK fk(_array);
        PrivateVector<double> vk(2, 0.0);
        sOut << "# k P\n";
        for(double k=-0.5; k<0.5; k+=0.001) {
          vk[0]=k*_angle.cos();
          vk[1]=k*_angle.sin();
          sOut << k << " " << fk.value(vk) << "\n";
        }
        sOut << "# k dP/dk\n";
        for(double k=-0.5; k<0.5; k+=0.001) {
          vk[0]=k*_angle.cos();
          vk[1]=k*_angle.sin();
          sOut << k << " " << fk.radialFirstDerivative(vk) << "\n";
        }
        sOut << "# k d^2P/dk^2\n";
        for(double k=-0.5; k<0.5; k+=0.001) {
          vk[0]=k*_angle.cos();
          vk[1]=k*_angle.sin();
          sOut << k << " " << fk.radialSecondDerivative(vk) << "\n";
        }
      }
    }
    break;
  case InvertTranslationRotation:
      invertTranslationRotation();
      break;
  case Misfit:
      misfit();
      break;
  }
  return true;
}

bool CoordReader::setOptions(int& argc, char ** argv)
{
  TRACE;
  QString coordFile;
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-distance") {
        _mode=Distance;
      } else if(arg=="-distance-lookup") {
        _mode=DistanceLookup;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-distance-geo") {
        _mode=DistanceGeo;
      } else if(arg=="-all-distances") {
        _mode=AllDistances;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-azimuth") {
        _mode=Azimuth;
      } else if(arg=="-azimuth-geo") {
        _mode=AzimuthGeo;
      } else if(arg=="-azimuth-lookup") {
        _mode=AzimuthLookup;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-from-dms") {
        _mode=FromDMS;
      } else if(arg=="-to-dms") {
        _mode=ToDMS;
      } else if(arg=="-from-dm") {
        _mode=FromDM;
      } else if(arg=="-to-dm") {
        _mode=ToDM;
      } else if(arg=="-from-geo") {
        _mode=FromGeo;
      } else if(arg=="-to-geo") {
        _mode=ToGeo;
      } else if(arg=="-to-utm") {
        if(CoreApplication::checkOptionArg(i, argc, argv, false)) {
          _utmZone.fromString(argv[i]);
          if(!_utmZone.isValid()) {
            App::log(UtmZone::parsingErrorMessage(argv[i])+"\n");
            return false;
          }
        }
        _mode=ToUTM;
      } else if(arg=="-from-utm") {
        if(CoreApplication::checkOptionArg(i, argc, argv, false)) {
          _utmZone.fromString(argv[i]);
          if(!_utmZone.isValid()) {
            App::log(UtmZone::parsingErrorMessage(argv[i])+"\n");
            return false;
          }
        }
        _mode=FromUTM;
      } else if(arg=="-to-kml") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _kmlFileName=argv[i];
        _mode=ToKML;
        delete _kml;
        _kml=new GoogleEarthKML;
        GoogleEarthKML::Document& doc=_kml->document();
        QFileInfo fi(_kmlFileName);
        doc.setName(fi.baseName());
        GoogleEarthKML::Folder * f=doc.mainFolder();
        f->setName(fi.baseName());
      } else if(arg=="-from-kml") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _kmlFileName=argv[i];
        _mode=FromKML;
      } else if(arg=="-reference") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _geoReference.setX(CoreApplication::toDouble(i, i-1, argv));
        CoreApplication::checkOptionArg(i, argc, argv, true, i-1);
        _geoReference.setY(CoreApplication::toDouble(i, i-2, argv));
      } else if(arg=="-translate") {
        _mode=Translate;
        CoreApplication::checkOptionArg(i, argc, argv);
        _translateVector.setX(CoreApplication::toDouble(i, i-1, argv));
        CoreApplication::checkOptionArg(i, argc, argv, true, i-1);
        _translateVector.setY(CoreApplication::toDouble(i, i-2, argv));
      } else if(arg=="-rotate") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _mode=Rotate;
        _angle.setValue(_angleMode, CoreApplication::toDouble(i, i-1, argv));
      } else if(arg=="-project") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _mode=Project;
        bool ok=true;
        QList<NamedPoint> points=loadFile(argv[i], ok);
        for(int i=0; i<points.size(); i++) {
          _polygon.addPoint(points.at(i));
        }
      } else if(arg=="-average") {
        _mode=Average;
      } else if(arg=="-invert-trans-rot") {
        _mode=InvertTranslationRotation;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-misfit") {
        _mode=Misfit;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-localize") {
        _mode=Localize;
        CoreApplication::checkOptionArg(i, argc, argv);
        _cartReference.setX(CoreApplication::toDouble(i, i-1, argv));
        CoreApplication::checkOptionArg(i, argc, argv, true, i-1);
        _cartReference.setY(CoreApplication::toDouble(i, i-2, argv));
        CoreApplication::checkOptionArg(i, argc, argv, false);
        _cartReference.setZ(CoreApplication::toDouble(i, i-3, argv));
      } else if(arg=="-line-move") {
        _mode=LineMove;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-lookup") {
        _mode=Lookup;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-trig-dist") {
        _mode=TriangulateDistance;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-trig-az") {
        _mode=TriangulateAzimuth;
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-circles") {
        _mode=GenerateCircles;
      } else if(arg=="-triangles") {
        _mode=GenerateTriangles;
      } else if(arg=="-random") {
        _mode=GenerateRandom;
      } else if(arg=="-markov-chain") {
        _mode=GenerateMarkovChain;
      } else if(arg=="-grid-circle") {
        _mode=GenerateGridCircle;
      } else if(arg=="-random-grid-circle") {
        _mode=GenerateRandomGridCircle;
      } else if(arg=="-size") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _sensorCount=CoreApplication::toInt(i, i-1, argv);
        if(_sensorCount<3) {
          App::log(tr("minimum number of sensors is 3 (%1)\n").arg(_sensorCount));
          return false;
        }
      } else if(arg=="-radius") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maximumRadius=CoreApplication::toDouble(i, i-1, argv);
        if(_maximumRadius<=0.0) {
          App::log(tr("Radius must be positive (%1)\n").arg(_maximumRadius));
          return false;
        }
      } else if(arg=="-first-index") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _firstIndex=CoreApplication::toInt(i, i-1, argv);
        if(_firstIndex<0) {
          App::log(tr("first index must be positive or null (%1)\n").arg(_firstIndex));
          return false;
        }
      } else if(arg=="-circle-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _circleCount.setValue(CoreApplication::toInt(i, i-1, argv));
        if(_circleCount.value()<=0) {
          App::log(tr("number of circles must be positive (%1)\n").arg(_circleCount.value()));
          return false;
        }
      } else if(arg=="-radius-factor") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _radiusFactor.setValue(CoreApplication::toDouble(i, i-1, argv));
        if(_radiusFactor.value()<=1.0) {
          App::log(tr("radius factor must be greater than 1 (%1)\n").arg(_radiusFactor.value()));
          return false;
        }
      } else if(arg=="-minimum-distance") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minimumDistance.setValue(CoreApplication::toDouble(i, i-1, argv));
        if(_minimumDistance.value()<0.0) {
          App::log(tr("minimum distance must be positive or null (%1)\n").arg(_minimumDistance.value()));
          return false;
        }
      } else if(arg=="-step-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _stepCount=CoreApplication::toInt(i, i-1, argv);
        if(_stepCount<=0.0) {
          App::log(tr("number of steps must be positive (%1)\n").arg(_stepCount));
          return false;
        }
      } else if(arg=="-maximum-step") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maximumStep.setValue(CoreApplication::toDouble(i, i-1, argv));
        if(_maximumStep.value()<=0.0) {
          App::log(tr("maximum step must be positive (%1)\n").arg(_maximumStep.value()));
          return false;
        }
      } else if(arg=="-maximum-deviation") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _maximumDeviation.setValue(CoreApplication::toDouble(i, i-1, argv));
        if(_maximumDeviation.value()<0) {
          App::log(tr("maximum deviation must be positive or null (%1)\n").arg(_maximumDeviation.value()));
          return false;
        }
      } else if(arg=="-add-sensors") {
        CoreApplication::checkOptionArg(i, argc, argv);
        bool ok=true;
        _fixedSensors=loadFile(argv[i], ok);
        if(!ok) {
          return false;
        }
      } else if(arg=="-add-include-zone") {
        CoreApplication::checkOptionArg(i, argc, argv);
        bool ok=true;
        QList<NamedPoint> points=loadFile(argv[i], ok);
        if(ok) {
          addZone(_includeZones, points);
        } else {
          return false;
        }
      } else if(arg=="-add-exclude-zone") {
        CoreApplication::checkOptionArg(i, argc, argv);
        bool ok=true;
        QList<NamedPoint> points=loadFile(argv[i], ok);
        if(ok) {
          addZone(_excludeZones, points);
        } else {
          return false;
        }
      } else if(arg=="-center") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _translateVector.setX(CoreApplication::toDouble(i, i-1, argv));
        CoreApplication::checkOptionArg(i, argc, argv, true, i-1);
        _translateVector.setY(CoreApplication::toDouble(i, i-2, argv));
      } else if(arg=="-from-distances") {
        _mode=FromDistances;
      } else if(arg=="-array-limits") {
        _mode=ArrayLimits;
      } else if(arg=="-array-pattern") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _angle.setGeographicValue(_angleMode, CoreApplication::toDouble(i, i-1, argv));
        _mode=ArrayPattern;
      } else if(arg=="-stddev") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _distanceStddev=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-origin") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _origin=argv[i];
      } else if(arg=="-north") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _north=argv[i];
      } else if(arg=="-eastward") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _eastward=argv[i];
      } else if(arg=="-prior-positions") {
        CoreApplication::checkOptionArg(i, argc, argv);
        coordFile=argv[i];
      } else if(arg=="-prior-precision") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _priorPrecision=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-posterior-precision") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _posteriorPrecision=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-cluster-count") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _clusterCount=CoreApplication::toInt(i, i-1, argv);
      } else if(arg=="-no-progress") {
        delete _progress;
        _progress=nullptr;
      } else if(arg=="-utm-declination") {
        _mode=UTMDeclination;
      } else if(arg=="-mag-declination") {
        _mode=MagDeclination;
        CoreApplication::checkOptionArg(i, argc, argv);
        int year=CoreApplication::toInt(i, i-1, argv);
        CoreApplication::checkOptionArg(i, argc, argv);
        int month=CoreApplication::toInt(i, i-2, argv);
        CoreApplication::checkOptionArg(i, argc, argv);
        int day=CoreApplication::toInt(i, i-3, argv);
        _date=QDate(year, month, day);
      } else if(arg=="-out") {
        CoreApplication::checkOptionArg(i, argc, argv);
        QString m=argv[i];
        m=m.toLower();
        if(m=="raw") {
          _fromDistancesOutMode=static_cast<FromDistancesOutMode>(_fromDistancesOutMode | Raw);
        } else if(m=="ellipse") {
          _fromDistancesOutMode=static_cast<FromDistancesOutMode>(_fromDistancesOutMode| Ellipse);
        } else if(m=="clusters") {
          _fromDistancesOutMode=static_cast<FromDistancesOutMode>(_fromDistancesOutMode | Clusters);
        } else if(m=="mean") {
          _fromDistancesOutMode=static_cast<FromDistancesOutMode>(_fromDistancesOutMode | Mean);
        } else {
          App::log(tr("unrecognized output mode '%1'', see -help\n").arg(argv[i]) );
          return false;
        }
      } else if(arg=="-degrees") {
        _angleMode=Angle::Degrees;
      } else if(arg=="-radians") {
        _angleMode=Angle::Radians;
      } else if(arg=="-gradians") {
        _angleMode=Angle::Gradians;
      } else if(arg=="-report") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _reportName=argv[i];
      } else if(arg=="-ns0") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ns0=CoreApplication::toInt(i, i-1, argv);
      } else if(arg=="-ns") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _ns=CoreApplication::toInt(i, i-1, argv);
      } else if(arg=="-nr") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _nr=CoreApplication::toInt(i, i-1, argv);
      } else {
        App::log(tr("bad option %1, see -help\n").arg(argv[i]) );
        return false;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(!coordFile.isEmpty()) {
    bool ok=true;
    QList<NamedPoint> points=loadFile(coordFile, ok);
    for(int i=points.size()-1; i>=0; i--) {
      const NamedPoint& p=points.at(i);
      _refPoints.insert(p.name(), p);
    }
  }
  if(_mode==FromDistances) {
    if(_origin.isEmpty()) {
      App::log(tr("missing option '-origin'\n") );
      return false;
    }
    if(_north.isEmpty()) {
      App::log(tr("missing option '-north'\n") );
      return false;
    }
    if(_eastward.isEmpty()) {
      App::log(tr("missing option '-eastward'\n") );
      return false;
    }
    if(_origin==_north) {
      App::log(tr("origin and north cannot refer to the same point\n") );
      return false;
    }
    if(_origin==_eastward) {
      App::log(tr("origin and eastward cannot refer to the same point\n") );
      return false;
    }
    if(_north==_eastward) {
      App::log(tr("north and eastward cannot refer to the same point\n") );
      return false;
    }
    if(_fromDistancesOutMode==NoOutput) {
      _fromDistancesOutMode=Mean;
    }
    _fromDistanceFactory.setProgess(_progress);
  }
  return true;
}

QList<NamedPoint> CoordReader::loadFile(const QString& fileName, bool& ok)
{
  QList<NamedPoint> points;
  QFile fCoord(fileName);
  if(!fCoord.open(QIODevice::ReadOnly)) {
    App::log(tr("cannot read file %1.\n").arg(fileName));
    ok=false;
    return points;
  }
  QTextStream sCoord(&fCoord);
  LineParser p;
  int iLine=0;
  int pointIndex=0;
  while(!sCoord.atEnd()) {
    QString l=sCoord.readLine();
    iLine++;
    if(!l.isEmpty() && l[0]!='#') {
      p.setString(l);
      double x, y, z;
      QString name;
      x=p.toDouble(0, ok);
      y=p.toDouble(1, ok);
      switch(p.count()) {
      case 2:
        z=0.0;
        name=tr("S%1").arg(pointIndex++, 3, 'f', QChar('0'));
        break;
      case 3:
        z=0.0;
        name=p.toString(2, ok);
        break;
      case 4:
        z=p.toDouble(2, ok);
        name=p.toString(3, ok);
        break;
      default:
        ok=false;
        break;
      }
      if(!ok) {
        App::log(tr("error reading file %1 at line %2\n").arg(fileName).arg(iLine));
        ok=false;
        points.clear();
        return points;
      }
      points.append(NamedPoint(x, y, z, name));
    }
  }
  return points;
}

bool CoordReader::hasInput() const
{
  TRACE;
  switch(_mode) {
  case Distance:
  case DistanceLookup:
  case DistanceGeo:
  case Azimuth:
  case AzimuthGeo:
  case AzimuthLookup:
  case FromGeo:
  case ToGeo:
  case FromDMS:
  case ToDMS:
  case FromDM:
  case ToDM:
  case ToUTM:
  case FromUTM:
  case ToKML:
  case ArrayLimits:
  case ArrayPattern:
  case UTMDeclination:
  case MagDeclination:
  case Translate:
  case Rotate:
  case Project:
  case InvertTranslationRotation:
  case Misfit:
  case Localize:
  case LineMove:
  case Average:
  case Lookup:
  case TriangulateDistance:
  case TriangulateAzimuth:
  case FromDistances:
  case AllDistances:
  case None:
    break;
  case FromKML:
  case GenerateCircles:
  case GenerateTriangles:
  case GenerateRandom:
  case GenerateMarkovChain:
  case GenerateGridCircle:
  case GenerateRandomGridCircle:
    return false;
  }
  return true;
}

bool CoordReader::parse(QTextStream& s)
{
  TRACE;
  QTextStream sOut(stdout);

  QString buf;
  bool ok=true;
  while(!s.atEnd()) {
    buf=s.readLine();
    if(!buf.isEmpty() && buf[0]!='\n') {
      if(buf[0]=='#') {
        switch(_mode) {
        case Distance:
        case DistanceLookup:
        case DistanceGeo:
        case Azimuth:
        case AzimuthGeo:
        case AzimuthLookup:
        case FromGeo:
        case ToGeo:
        case FromDMS:
        case ToDMS:
        case FromDM:
        case ToDM:
        case ToUTM:
        case FromUTM:
        case ToKML:
        case ArrayLimits:
        case ArrayPattern:
        case UTMDeclination:
        case MagDeclination:
        case Translate:
        case Rotate:
        case Project:
        case InvertTranslationRotation:
        case Misfit:
        case Localize:
        case LineMove:
        case Average:
        case Lookup:
        case TriangulateAzimuth:
        case FromDistances:
        case AllDistances:
        case None:
        case FromKML:
        case GenerateCircles:
        case GenerateTriangles:
        case GenerateRandom:
        case GenerateMarkovChain:
        case GenerateGridCircle:
        case GenerateRandomGridCircle:
          break;
        case TriangulateDistance:
          triangulateDistance();
          _circles.clear();
          break;
        }
        sOut << buf << Qt::endl;
      } else {
        switch(_mode) {
        case Distance: {
            LineParser par(buf);
            if(par.count()==6) {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok), par.toDouble(2,ok));
              Point p2(par.toDouble(3,ok), par.toDouble(4,ok), par.toDouble(5,ok));
              sOut << p1.distanceTo(p2) << Qt::endl;
            } else {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok));
              Point p2(par.toDouble(2,ok), par.toDouble(3,ok));
              sOut << p1.distanceTo(p2) << Qt::endl;
            }
          }
          break;
        case DistanceLookup: {
            LineParser par(buf);
            QString name1=par.toString(0,ok);
            QString name2=par.toString(1,ok);
            QMap<QString, Point>::iterator it1=_refPoints.find(name1);
            if(it1==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(name1) << Qt::endl;
              return true;
            }
            QMap<QString, Point>::iterator it2=_refPoints.find(name2);
            if(it2==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(name2) << Qt::endl;
              return true;
            }
            Point p1=it1.value();
            Point p2=it2.value();
            double calcDist=p1.distanceTo(p2);
            if(par.count()==3) {
              double refDist=par.toDouble(2,ok);
              sOut << QString("%1 %2 %3 %4 %5 %6").arg(name1).arg(name2)
                      .arg(calcDist, 0, 'f', 2)
                      .arg(refDist, 0, 'f', 2)
                      .arg(refDist-calcDist, 0, 'f', 2)
                      .arg(100.0*(refDist-calcDist)/refDist, 0, 'f', 1) << Qt::endl;
            } else {
              sOut << QString("%1 %2 %3").arg(name1).arg(name2).arg(calcDist, 0, 'f', 2) << Qt::endl;
            }
          }
          break;
        case DistanceGeo: {
            LineParser par(buf);
            if(par.count()==6) {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok), par.toDouble(2,ok));
              Point p2(par.toDouble(3,ok), par.toDouble(4,ok), par.toDouble(5,ok));
              sOut << p1.geographicalDistanceTo(p2) << Qt::endl;
            } else {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok));
              Point p2(par.toDouble(2,ok), par.toDouble(3,ok));
              sOut << p1.geographicalDistanceTo(p2) << Qt::endl;
            }
          }
          break;
        case AllDistances: {
            NamedPoint p;
            ok=p.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input: %1\n").arg(buf) );
              return false;
            }
            for(QMap<QString, Point>::const_iterator it=_refPoints.begin();
                it!=_refPoints.end(); it++) {
              sOut << p.distanceTo(it.value()) << " " << p.name() << " " << it.key() << Qt::endl;
            }
          }
          break;
        case Azimuth: {
            LineParser par(buf);
            double az;
            if(par.count()==6) {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok), par.toDouble(2,ok));
              Point p2(par.toDouble(3,ok), par.toDouble(4,ok), par.toDouble(5,ok));
              az=p1.azimuthTo(p2);
            } else {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok));
              Point p2(par.toDouble(2,ok), par.toDouble(3,ok));
              az=p1.azimuthTo(p2);
            }
            sOut << Angle::radiansToDegrees(Angle::mathToGeographic(az)) << Qt::endl;
          }
          break;
        case AzimuthGeo: {
            LineParser par(buf);
            double az;
            if(par.count()==6) {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok), par.toDouble(2,ok));
              Point p2(par.toDouble(3,ok), par.toDouble(4,ok), par.toDouble(5,ok));
              az=p1.geographicalAzimuthTo(p2);
            } else {
              Point p1(par.toDouble(0,ok), par.toDouble(1,ok));
              Point p2(par.toDouble(2,ok), par.toDouble(3,ok));
              az=p1.geographicalAzimuthTo(p2);
            }
            sOut << Angle::radiansToDegrees(az) << Qt::endl;
          }
          break;
        case AzimuthLookup: {
            LineParser par(buf);
            QString name1=par.toString(0,ok);
            QString name2=par.toString(1,ok);
            QMap<QString, Point>::iterator it1=_refPoints.find(name1);
            if(it1==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(name1) << Qt::endl;
              return true;
            }
            QMap<QString, Point>::iterator it2=_refPoints.find(name2);
            if(it2==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(name2) << Qt::endl;
              return true;
            }
            Point p1=it1.value();
            Point p2=it2.value();
            double a=Angle::radiansToDegrees(Angle::mathToGeographic(p1.azimuthTo(p2)));
            sOut << QString("%1 %2 %3").arg(name1).arg(name2).arg(a, 0, 'f', 4) << " deg" << Qt::endl;
        }
        break;
        case FromGeo: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input: %1\n").arg(buf) );
              return false;
            }
            longLat.geographicalToRectangular(_geoReference);
            sOut << " " << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case ToGeo: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            longLat.rectangularToGeographical(_geoReference);
            sOut << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case FromDMS: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input: %1\n").arg(buf) );
              return false;
            }
            longLat.DMSToDegrees();
            sOut << " " << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case ToDMS: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            longLat.degreesToDMS();
            sOut << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case FromDM: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input: %1\n").arg(buf) );
              return false;
            }
            longLat.DMToDegrees();
            sOut << " " << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case ToDM: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            longLat.degreesToDM();
            sOut << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case ToUTM: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            UtmZone zone=_utmZone;
            if(!zone.isValid()) {
              zone=UtmZone(longLat);
            }
            longLat.geographicalToUtm(zone);
            sOut << zone.toString() << " " << longLat.toString('g', 20) << Qt::endl;
          }
          break;
        case FromUTM: {
            LineParser par(buf);
            bool ok=true;
            UtmZone zone;
            zone.fromString(par.toString(0, ok));
            if(!ok) {
              App::log(tr("Error parsing input, cannot read first column.\n") );
              return false;
            }
            QString pointString;
            if(zone.isValid()) {
              pointString=par.toString(1, par.count(), ok);
              if(!ok) {
                App::log(tr("Error parsing input, missong point coordinates.\n") );
                return false;
              }
            } else {
              pointString=buf;
            }
            NamedPoint utmCoord;
            ok=utmCoord.fromString(pointString);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            if(zone.isValid()) {
              utmCoord.utmToGeographical(zone);
            } else if(_utmZone.isValid()){
              utmCoord.utmToGeographical(_utmZone);
            } else {
              App::log(tr("Missing specification of UTM zone (stdin or argument, see -help)\n") );
              return false;
            }
            sOut << utmCoord.toString('g', 20) << Qt::endl;
          }
          break;
        case ToKML: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            GoogleEarthKML::Document& doc=_kml->document();
            GoogleEarthKML::Folder * f=doc.mainFolder();
            GoogleEarthKML::Placemark * p=new GoogleEarthKML::Placemark;
            p->setName(longLat.name());
            p->setCoordinates(longLat);
            f->addPlacemark(p);
          }
          break;
        case ArrayLimits:
        case ArrayPattern: {
            LineParser par(buf);
            Point2D p(par.toDouble(0,ok), par.toDouble(1,ok));
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            _array.append(p);
          }
          break;
        case FromKML:
        case GenerateCircles:
        case GenerateTriangles:
        case GenerateRandom:
        case GenerateMarkovChain:
        case GenerateGridCircle:
        case GenerateRandomGridCircle:
          ASSERT(false);
          break;
        case Translate: {
            NamedPoint p;
            ok=p.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            p.translate(_translateVector);
            sOut << p.toString('g', 20) << Qt::endl;
          }
          break;
        case Rotate: {
            NamedPoint p;
            ok=p.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            p.rotate(_angle);
            sOut << p.toString('g', 20) << Qt::endl;
          }
          break;
        case Project: {
            NamedPoint p;
            ok=p.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            p=_polygon.project(p, p.z());
            sOut << p.toString('g', 20) << Qt::endl;
          }
          break;
        case Misfit:
        case InvertTranslationRotation: {
            NamedPoint p;
            ok=p.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            _points.append(p);
        }
        break;
        case Localize: {
            NamedPoint v, p;
            ok=v.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            Angle ah, av;
            ah.setGeographicValue(_angleMode, v.y());
            av.setGeographicValue(_angleMode, v.z());
            double dh=v.x()*av.cos();
            p.setX(_cartReference.x()+dh*ah.cos());
            p.setY(_cartReference.y()+dh*ah.sin());
            p.setZ(_cartReference.z()+v.x()*av.sin());
            p.setName(v.name());
            sOut << p.toString('g', 20) << Qt::endl;
          }
          break;
        case LineMove: {
            NamedPoint p;
            LineParser par(buf);
            QString name1=par.toString(0,ok);
            QString name2=par.toString(1,ok);
            double distance=par.toDouble(2,ok);
            QString name=par.toString(3,ok);
            QMap<QString, Point>::iterator it1=_refPoints.find(name1);
            if(it1==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(name1) << Qt::endl;
              return true;
            }
            QMap<QString, Point>::iterator it2=_refPoints.find(name2);
            if(it2==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(name2) << Qt::endl;
              return true;
            }
            Point p1=it1.value();
            Point p2=it2.value();
            Angle a;
            a.setRadians(p1.azimuthTo(p2));
            p.setX(p1.x()+distance*a.cos());
            p.setY(p1.y()+distance*a.sin());
            p.setName(name);
            sOut << p.toString('g', 20) << Qt::endl;
          }
          break;
        case Average: {
            Point p;
            ok=p.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            _average+=p;
            _averageCount++;
          }
          break;
        case Lookup: {
            QMap<QString, Point>::iterator it=_refPoints.find(buf);
            if(it==_refPoints.end()) {
              sOut << tr("Error unknown point name: %1").arg(buf) << Qt::endl;
              return false;
            }
            sOut << it.value().toString('g', 20) << " " << it.key() << Qt::endl;
          }
          break;
        case TriangulateDistance: {
            LineParser par(buf);
            if(par.count()==2) {
              QString name=par.toString(0,ok);
              if(!ok) {
                App::log(tr("Error parsing input, missing reference point name.\n") );
                return false;
              }
              QMap<QString, Point>::iterator it=_refPoints.find(name);
              if(it==_refPoints.end()) {
                App::log(tr("Error unknown point name: %1\n").arg(name) );
                return false;
              }
              _circles.append(Circle(it.value().x(), it.value().y(), par.toDouble(1,ok)));
              if(!ok) {
                App::log(tr("Error parsing input, missing distance for point %1.\n").arg(name) );
                return false;
              }
            } else {
              App::log(tr("Error parsing input, NAME DISTANCE expected\n") );
              return false;
            }
          }
          break;
        case TriangulateAzimuth: {
            LineParser par(buf);
            if(par.count()==2) {
              QString name=par.toString(0,ok);
              if(!ok) {
                App::log(tr("Error parsing input, missing reference point name.\n") );
                return false;
              }
              QMap<QString, Point>::iterator it=_refPoints.find(name);
              if(it==_refPoints.end()) {
                App::log(tr("Error unknown point name: %1\n").arg(name) );
                return false;
              }
              double az=par.toDouble(1,ok)*M_PI/180.0;
              _lines.append(Line2D(it.value().x(), it.value().y(), az));
              if(!ok) {
                App::log(tr("Error parsing input, missing azimuth for point %1.\n").arg(name) );
                return false;
              }
            } else {
              App::log(tr("Error parsing input, NAME AZIMUTH expected\n") );
              return false;
            }
          }
          break;
        case FromDistances: {
            LineParser par(buf);
            if(par.count()==3) {
              QString name1=par.toString(0,ok);
              if(!ok) {
                App::log(tr("Error parsing input, missing first point name.\n") );
                return false;
              }
              QString name2=par.toString(1,ok);
              if(!ok) {
                App::log(tr("Error parsing input, missing second point name.\n") );
                return false;
              }
              double d=par.toDouble(2,ok);
              if(!ok) {
                App::log(tr("Error parsing input, missing distance for couple %1-%2.\n").arg(name1).arg(name2) );
                return false;
              }
              if(name1==name2) {
                App::log(tr("Error parsing input, name of points must differ within the same couple ('%1').\n").arg(name1) );
                return false;
              }
              _fromDistanceFactory.addDistance(name1, name2, d);
            } else {
              App::log(tr("Error parsing input, NAME1 NAME2 DISTANCE expected\n") );
              return false;
            }
          }
          break;
        case UTMDeclination: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            Point2D north;
            UtmZone zone;
            north=longLat;
            north.geographicalToUtm(zone);
            north+=Point2D(0.0, 100.0); // 100 m in UTM north direction
            north.utmToGeographical(zone);
            double angle=longLat.geographicalAzimuthTo(north);
            if(angle>M_PI) {
              angle-=2*M_PI;
            }
            sOut << Angle::radiansToDegrees(angle) << Qt::endl;
          }
          break;
        case MagDeclination: {
            NamedPoint longLat;
            ok=longLat.fromString(buf);
            if(!ok) {
              App::log(tr("Error parsing input\n") );
              return false;
            }
            QStringList args;
            args << "-O" << "-" << "--quiet"
                 << QString("https://www.ngdc.noaa.gov/geomag-web/calculators/calculateDeclination?"
                            "lon1=%1&lat1=%2&model=IGRF&startYear=%3&startMonth=%4&startDay=%5&"
                            "resultFormat=xml")
                    .arg(longLat.x())
                    .arg(longLat.y())
                    .arg(_date.year())
                    .arg(_date.month())
                    .arg(_date.day());
            App::log(1, "wget "+args.join(" ")+"\n");
            QProcess p;
            p.start("wget", args, QIODevice::ReadOnly);
            p.waitForFinished();
            QString xml=p.readAllStandardOutput();
            App::log(2, xml);
            XMLGenericItem item("maggridresult");
            XMLHeader header(&item);
            XMLClass::Error err=header.xml_restoreString(xml);
            if(err!=XMLClass::NoError) {
              App::log(XMLClass::message(err)+"\n");
              return false;
            }
            QList<XMLGenericItem *> result=item.children("result");
            if(result.count()!=1) {
              App::log(tr("Bad XML response from server, increase verbosity to get details\n") );
              return false;
            }
            QList<XMLGenericItem *> subResults=result.first()->children("declination");
            if(subResults.count()!=1) {
              App::log(tr("Bad XML response from server, increase verbosity to get details\n") );
              return false;
            }
            sOut << QString::number(subResults.first()->value().toDouble()) << Qt::endl;
          }
          break;
        case None:
          App::log(tr("no operation defined, see -h\n") );
          return false;
        }
      }
    }
  }
  return true;
}

bool CoordReader::arrayLimits(double& kmin, double& kmax)
{
  KminSolver kminSolver(_array);
  bool ok=true;
  kmin=kminSolver.calculate(ok);
  if(ok && kmin>0.0) {
    KmaxSolver kmaxSolver(_array, kmin, 0.25);
    kmaxSolver.calculate();
    while(!kmaxSolver.isRunning()) {}
    if(!kmaxSolver.wait(10000)) {
      QTextStream(stdout) << "# kmax is at least... search stopped after 10 s" << Qt::endl;
    }
    kmax=kmaxSolver.kmax(ok);
    if(ok) {
      return true;
    } else {
      App::log(tr("error computing kmax\n"));
      return false;
    }
  } else {
    App::log(tr("error computing kmin (linear array?)\n"));
    return false;
  }
}

bool CoordReader::generateCircles()
{
  QTextStream sOut(stdout);
  Angle a;
  NamedPoint p;
  int sensorIndex=_firstIndex;
  if(sensorIndex==0) {
    p.translate(_translateVector);
    p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
    sOut << p.toString('g', 20) << Qt::endl;
  }
  double radius=_maximumRadius*pow(_radiusFactor.value(), 1-_circleCount.value());
  double a0=0.0;
  for(int i=0; i<_circleCount.value(); i++) {
    for(int j=0; j<_sensorCount; j++) {
      a.setGeographicValue(_angleMode, a0+j*360.0/_sensorCount);
      p.setX(radius*a.cos());
      p.setY(radius*a.sin());
      p.translate(_translateVector);
      p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
      sOut << p.toString('g', 20) << Qt::endl;
    }
    a0+=180.0/_sensorCount;
    radius*=_radiusFactor.value();
  }
  return true;
}

bool CoordReader::generateRandom()
{
  if(_minimumDistance.value()>=_maximumRadius || _maximumRadius<=0.0) {
    App::log(tr("bad radius (%1) or minimum distance (%2).\n").arg(_maximumRadius).arg(_minimumDistance.value()));
    return false;
  }
  QTextStream sOut(stdout);
  NamedPoint p;
  Random r;
  QList<NamedPoint> plist=_fixedSensors;
  int j=0;
  int sensorIndex=_firstIndex;
  CoreApplication::instance()->debugUserInterrupts(false);
  while(j<_sensorCount) {
    p.setX(r.uniform(-_maximumRadius, _maximumRadius));
    p.setY(r.uniform(-_maximumRadius, _maximumRadius));
    if(acceptPoint(p)) {
      p.translate(_translateVector);
      int ip=plist.size()-1;
      while(ip>=0) {
        if(p.distanceTo(plist.at(ip))<_minimumDistance.value()) {
          break;
        } else {
          ip--;
        }
      }
      if(ip<0) {
        App::log(tr("Found %1 points\n").arg(plist.size()));
        p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
        plist.append(p);
        sOut << p.toString('g', 20) << Qt::endl;
        j++;
      }
    }
  }
  return true;
}

bool CoordReader::generateMarkovChain()
{
  if(_minimumDistance.value()>=_maximumRadius || _maximumRadius<=0.0) {
    App::log(tr("bad radius (%1) or minimum distance (%2).\n").arg(_maximumRadius).arg(_minimumDistance.value()));
    return false;
  }
  QTextStream sOut(stdout);
  NamedPoint p;
  Random r;
  Angle a;
  QList<NamedPoint> plist=_fixedSensors;
  int sensorIndex=_firstIndex;
  p.translate(_translateVector);
  p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
  plist.append(p);
  sOut << p.toString('g', 20) << Qt::endl;
  int j=1;
  double totalWalkDistance=0.0;
  CoreApplication::instance()->debugUserInterrupts(false);
  while(j<_sensorCount) {
    Point2D nextPoint(p);
    for(int step=0; step<_stepCount; step++) {
      a.setGeographicValue(_angleMode, r.uniform(0.0, 360.0));
      double d=r.uniform(0.0, _maximumStep.value());
      nextPoint.move(d, a);
      totalWalkDistance+=d;
    }
    if(acceptPoint(nextPoint)) {
      nextPoint.translate(_translateVector);
      int ip=plist.size()-1;
      while(ip>=0) {
        if(nextPoint.distanceTo(plist.at(ip))<_minimumDistance.value()) {
          break;
        } else {
          ip--;
        }
      }
      if(ip<0) {
        App::log(tr("Found %1 points (walked %2 m)\n").arg(j).arg(totalWalkDistance));
        p=nextPoint;
        p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
        plist.append(p);
        sOut << p.toString('g', 20) << Qt::endl;
        j++;
      } else {
        // Not satisfying conditions, return to last good point
        totalWalkDistance+=p.distanceTo(nextPoint);
      }
    }
  }
  App::log(tr("Total walked distance %1 m\n").arg(totalWalkDistance));
  return true;
}

bool CoordReader::generateGridCircle()
{
  QTextStream sOut(stdout);
  NamedPoint p;
  int nSide=qCeil(0.5*sqrt(_sensorCount));
  double step=_maximumRadius/nSide;
  int sensorIndex=_firstIndex;
  for(int ix=-nSide; ix<=nSide; ix++) {
    p.setX(ix*step);
    for(int iy=-nSide; iy<=nSide; iy++) {
      p.setY(iy*step);
      if(acceptPoint(p)) {
        p.translate(_translateVector);
        p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
        sOut << p.toString('g', 20) << Qt::endl;
      }
    }
  }
  return true;
}

bool CoordReader::generateRandomGridCircle()
{
  QTextStream sOut(stdout);
  NamedPoint p;
  Random r;
  Angle a;
  int nSide=qCeil(0.5*sqrt(_sensorCount));
  double step=_maximumRadius/nSide;
  int sensorIndex=_firstIndex;
  for(int ix=-nSide; ix<=nSide; ix++) {
    for(int iy=-nSide; iy<=nSide; iy++) {
      p.setX(ix*step);
      p.setY(iy*step);
      if(acceptPoint(p)) {
        while(true) {
        for(int mcs=0; mcs<_maximumStep.value(); mcs++) {
            a.setGeographicValue(_angleMode, r.uniform(0.0, 360.0));
            double d=r.uniform(0.0, _maximumDeviation.value());
            p.Point2D::move(d, a);
          }
          if(acceptPoint(p)) {
            p.translate(_translateVector);
            p.setName(QString("S%1").arg(sensorIndex++, 3, 10, QChar('0')));
            sOut << p.toString('g', 20) << Qt::endl;
          }
        }
      }
    }
  }
  return true;
}

bool CoordReader::acceptPoint(Point2D p) const
{
  if(p.length()<_maximumRadius) {
    p.translate(_translateVector);
    QPointF qp=p.pointF();
    int i;
    for(i=_includeZones.size()-1; i>=0; i--) {
      if(_includeZones.at(i).contains(qp)) {
        break;
      }
    }
    if(i>=0 || _includeZones.isEmpty()) {
      for(i=_excludeZones.size()-1; i>=0; i--) {
        if(_excludeZones.at(i).contains(qp)) {
          return false;
        }
      }
      return true;
    }
  }
  return false;
}

void CoordReader::addZone(QList<Polygon>& zoneList, QList<NamedPoint>& newZone)
{
  Polygon newPolygon;
  int i;
  if(newZone.first()==newZone.last()) {
    i=newZone.size()-2;
  } else {
    i=newZone.size()-1;
  }
  while(i>=0) {
    newPolygon.addPoint(newZone.at(i--));
  }
  if(newPolygon.isValid())
  zoneList.append(newPolygon);
}

void CoordReader::invertTranslationRotation()
{
  TranslationRotationForward forward;
  forward.setReferencePoints(_refPoints);
  forward.setPoints(_points);

  ModelRepository na;
  na.setForward(&forward);
  na.setStorage();
  na.setMaximumModelCount(_ns0);
  if(!_reportName.isEmpty()) {
    na.openReport(_reportName);
  }
  na.start(1, Generator::MonteCarlo);
  na.wait();
  na.setMaximumModelCount(_ns);
  na.setBestModelCount(_nr);
  na.setStableMisfitCount(1000);
  na.start(1, Generator::Neighborhood);
  na.wait();
  if(na.validModelCount()>0) {
    const SetIndex& iBest=na.bestIndexEver();
    TranslationRotationForward * f=static_cast<TranslationRotationForward *>(na.forward());
    Point2D dxy=f->fixedTranslation();
    dxy+=Point2D(na.variableParameterValue(iBest, 0),
                 na.variableParameterValue(iBest, 1));
    QTextStream out(stdout);
    out << tr("gpcoord -rotate %1 | gpcoord -translate %2\n")
               .arg(na.variableParameterValue(iBest, 2))
               .arg(dxy.toString('f', 3));
    out << tr("Best model at index %1 (%2)\n").arg(iBest.value()).arg(na.misfit(iBest));
    f->reportErrors(out,
                    na.variableParameterValue(iBest, 0),
                    na.variableParameterValue(iBest, 1),
                    na.variableParameterValue(iBest, 2));
  }
}

void CoordReader::misfit()
{
  TranslationRotationForward forward;
  forward.setReferencePoints(_refPoints);
  forward.setPoints(_points);

  double model[3];
  model[0]=0.0;
  model[1]=0.0;
  model[2]=0.0;
  bool ok=true;
  double m=forward.misfit(model, ok);

  QTextStream out(stdout);
  out << tr("Misfit= %1\n").arg(m);
  forward.reportErrors(out, 0.0, 0.0, 0.0);
}

bool CoordReader::triangulateDistance()
{
  QTextStream sOut(stdout);
  sOut.setRealNumberPrecision(20);
  if(_circles.count()==2) {
    VectorList<Point2D> p=_circles.first().intersect(_circles.last());
    for(VectorList<Point2D>::iterator it=p.begin(); it!=p.end(); it++) {
      sOut << it->x() << " " << it->y() << Qt::endl;
    }
  } else if(_circles.count()>=3) {
    bool ok;
    int n=0;
    Point2D pAver(0, 0), pDev(0, 0);
    for(int i=0; i<_circles.count(); i++) {
      for(int j=i+1; j<_circles.count(); j++) {
        // Used to discriminate between the classical
        // two solutions of the intersection of two circles.
        int disc;
        if(j+1<_circles.count()) {
          disc=j+1;
        } else if(i+1<j) {
          disc=i+1;
        } else {
          disc=i-1;
        }
        Point2D p=_circles.at(i).intersect(_circles.at(j), ok, _circles.at(disc));
        if(ok) {
          pAver+=p;
          pDev+=p*p;
          n++;
        }
      }
    }
    if(n>0) {
      pAver/=n;
      pDev/=n;
      pDev-=pAver * pAver;
    } else {
      App::log(tr("No intersection\n") );
      return false;
    }
    sOut << pAver.x() << " " << pAver.y() << " "
         << sqrt(pDev.x()) << " " << sqrt(pDev.y()) << Qt::endl;
  } else {
    App::log(tr("At least two distances are required to triangulate.\n") );
    return false;
  }
  return true;
}
